package org.phosphoresce.common.graphics.paint;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GradientPaint;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;

import org.phosphoresce.common.graphics.util.ImageGraphicsUtil;

/**
 * 線形ペイントコンテキストクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/06/08	Kitagawa		新規作成
 *-->
 */
public class LinearGradientPaintContext extends AbstractGradientPaintContext {

	/**
	 * コンストラクタ<br>
	 * @param model Paintデータを受け取るColorModelオブジェクト
	 * @param deviceBounds 描画されるグラフィックスプリミティブのデバイス空間でのバウンディングボックス
	 * @param userBounds 描画されるグラフィックスプリミティブのユーザ空間でのバウンディングボックス
	 * @param transform ユーザ空間からデバイス空間へのAffineTransform
	 * @param hints コンテキストオブジェクトが描画の選択肢を選択するときに使用するヒント
	 * @param angle 角度
	 * @param colorDefine グラデーションカラー定義
	 * @param alphaDefine グラデーションアルファ定義
	 * @param smoothness 滑らかさ(0.0f～1.0f)
	 * @param raito 比率(0.0f～100.0f)
	 */
	public LinearGradientPaintContext(ColorModel model, Rectangle deviceBounds, Rectangle2D userBounds, AffineTransform transform, RenderingHints hints, double angle, //
			GradientColorDefine colorDefine, GradientAlphaDefine alphaDefine, float smoothness, float raito) {
		super(model, deviceBounds, userBounds, transform, hints, angle, colorDefine, alphaDefine, smoothness, raito);
	}

	/**
	 * 指定されたグラフィックスオブジェクトに対して回転角0度時のグラデーションイメージを描画します。<br>
	 * @param graphics グラフィックスオブジェクト
	 * @param dimension 描画サイズ
	 * @param gradient グラデーション定義
	 * @see org.phosphoresce.common.graphics.paint.AbstractGradientPaintContext#fillGradientImage(java.awt.Graphics2D, java.awt.Dimension, org.phosphoresce.common.graphics.paint.GradientDefine)
	 */
	protected void fillGradientImage(Graphics2D graphics, Dimension dimension, GradientDefine gradient) {
		// グラデーション定義取得
		GradientDefine[] defines = (GradientDefine[]) gradient.getAllDefines();

		/*
		 * バッファイメージに対してグラデーションを描画
		 */

		/*
		 * バッファイメージ作成
		 */
		// バッファイメージサイズ算出
		int bufferImageWidth = (int) (dimension.getWidth() * getRatio());
		int bufferImageHeight = (int) (dimension.getHeight());
		bufferImageWidth = bufferImageWidth <= 0 ? 1 : bufferImageWidth;
		bufferImageHeight = bufferImageHeight <= 0 ? 1 : bufferImageHeight;

		// バッファイメージ生成
		BufferedImage bufferImage = ImageGraphicsUtil.createEmptyBufferedImage(bufferImageWidth, bufferImageHeight);

		// バッファイメージ描画
		Color headColor = null;
		Color tailColor = null;
		for (int i = 0; i <= defines.length - 2; i++) {
			GradientDefine define1 = defines[i];
			GradientDefine define2 = defines[i + 1];
			Color color1 = define1.getGradientColor();
			Color color2 = define2.getGradientColor();

			// グラデーション描画後の余白ベタ塗用のカラーを退避
			if (i == 0) {
				headColor = color1;
				tailColor = color2;
			} else {
				tailColor = color2;
			}

			// 描画情報算出
			float x1 = define1.getLocation() * bufferImageWidth;
			float y1 = 0.0f;
			float x2 = define2.getLocation() * bufferImageWidth;
			float y2 = 0.0f;
			float smooth = ((x2 - x1) / 8) * getSmoothness();

			// 各種描画オブジェクト生成
			Point2D point1 = new Point2D.Float(x1 + smooth, y1);
			Point2D point2 = new Point2D.Float(x2 - smooth, y2);
			Shape shape = new Rectangle2D.Float(x1, 0, x2, bufferImageHeight);
			GradientPaint paint = new GradientPaint(point1, color1, point2, color2);

			// バッファイメージ描画
			Graphics2D g = (Graphics2D) bufferImage.getGraphics();
			g.setPaint(paint);
			g.fill(shape);
		}

		/*
		 * グラフィックスオブジェクト描画
		 */
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

		// 倍率が1.0未満の場合は余白を塗りつぶす
		if (getRatio() < 1.0f) {
			graphics.setPaint(headColor);
			graphics.fill(new Rectangle(0, 0, dimension.width / 2, bufferImageHeight));
			graphics.setPaint(tailColor);
			graphics.fill(new Rectangle(dimension.width / 2, 0, dimension.width, bufferImageHeight));
		}

		// バッファイメージを実イメージオブジェクトに描画
		int x = -(bufferImageWidth - dimension.width) / 2;
		int y = 0;
		int width = bufferImageWidth;
		int height = bufferImageHeight;
		graphics.drawImage(bufferImage, x, y, width, height, null);
	}
}
