package org.phosphoresce.common.graphics.paint;

import java.awt.Color;
import java.util.LinkedList;
import java.util.List;

/**
 * グラデーション定義上位抽象クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/06/08	Kitagawa		新規作成
 *-->
 */
public abstract class GradientDefine {

	/** 位置 */
	protected float location;

	/** 後方定義 */
	protected GradientDefine behindDefine;

	/** 後方分岐ポイント */
	protected float behindDivergence;

	/** 前方分岐ポイント */
	protected float frontDivergence;

	/** 前方定義 */
	protected GradientDefine frontDefine;

	/**
	 * コンストラクタ<br>
	 * @param location 位置
	 * @param behindDivergence 後方分岐ポイント
	 * @param frontDivergence 前方分岐ポイント
	 */
	public GradientDefine(float location, float behindDivergence, float frontDivergence) {
		super();
		this.behindDefine = null;
		this.frontDefine = null;
		this.setBehindDivergence(behindDivergence);
		this.setFrontDivergence(frontDivergence);
		this.setLocation(location);
	}

	/**
	 * コンストラクタ<br>
	 * @param location 位置
	 */
	public GradientDefine(float location) {
		this(location, 0.5f, 0.5f);
	}

	/**
	 * コンストラクタ<br>
	 */
	private GradientDefine() {
		this(0.0f, 0.5f, 0.5f);
	}

	/**
	 * グラデーション定義が提供するカラーを取得します。<br>
	 * @return グラデーション定義が提供するカラー
	 */
	public abstract Color getGradientColor();

	/**
	 * 位置を取得します。<br>
	 * @return 位置
	 */
	public float getLocation() {
		return location;
	}

	/**
	 * 位置を設定します。<br>
	 * @param location 位置
	 */
	public void setLocation(float location) {
		GradientDefine behind = this.behindDefine;
		GradientDefine front = this.frontDefine;
		remove(this);
		this.location = location;
		if (this.location > 1.0f) {
			this.location = 1.0f;
		} else if (this.location < 0.0f) {
			this.location = 0.0f;
		}
		if (behind != null) {
			behind.add(this);
		} else if (front != null) {
			front.add(this);
		}
	}

	/**
	 * 後方分岐ポイントを取得します。<br>
	 * @return 後方分岐ポイント
	 */
	public float getBehindDivergence() {
		return behindDivergence;
	}

	/**
	 * 後方分岐ポイントを設定します。<br>
	 * @param behindDivergence 後方分岐ポイント
	 */
	public void setBehindDivergence(float behindDivergence) {
		this.behindDivergence = behindDivergence;
		if (this.behindDivergence < 0.0f) {
			this.behindDivergence = 0.0f;
		} else if (this.behindDivergence > 1.0f) {
			this.behindDivergence = 1.0f;
		}
	}

	/**
	 * 前方分岐ポイントを取得します。<br>
	 * @return 前方分岐ポイント
	 */
	public float getFrontDivergence() {
		return frontDivergence;
	}

	/**
	 * 前方分岐ポイントを設定します。<br>
	 * @param frontDivergence 前方分岐ポイント
	 */
	public void setFrontDivergence(float frontDivergence) {
		this.frontDivergence = frontDivergence;
		if (this.frontDivergence < 0.0f) {
			this.frontDivergence = 0.0f;
		} else if (this.frontDivergence > 1.0f) {
			this.frontDivergence = 1.0f;
		}
	}

	/**
	 * 自身の定義が先頭のグラデーションカラー定義であるか判定します。<br>
	 * @return 自身の定義が先頭のグラデーションカラー定義である場合にtrueを返却
	 */
	public boolean isHead() {
		return frontDefine == null;
	}

	/**
	 * 自身の定義が末尾のグラデーションカラー定義であるか判定します。<br>
	 * @return 自身の定義が末尾のグラデーションカラー定義である場合にtrueを返却
	 */
	public boolean isTail() {
		return behindDefine == null;
	}

	/**
	 * 先頭のグラデーションカラー定義を取得します。<br>
	 * @return 先頭のグラデーションカラー定義
	 */
	public GradientDefine getHeadDefine() {
		if (isHead()) {
			return this;
		} else {
			return frontDefine.getHeadDefine();
		}
	}

	/**
	 * 末尾のグラデーションカラー定義を取得します。<br>
	 * @return 末尾のグラデーションカラー定義
	 */
	public GradientDefine getTailDefine() {
		if (isTail()) {
			return this;
		} else {
			return behindDefine;
		}
	}

	/**
	 * 全てのグラデーションカラー定義リストを取得します。<br>
	 * ここで提供するリストの順序を変更しても実際の定義関係には反映されません。<br>
	 * @return 全てのグラデーションカラー定義リスト
	 */
	protected List getAllDefineList() {
		List list = new LinkedList();
		GradientDefine define = getTailDefine();
		list.add(define);
		while (!define.isHead()) {
			define = define.frontDefine;
			list.add(define);
		}
		return list;
	}

	/**
	 * 全てのグラデーションカラー定義を取得します。<br>
	 * @return 全てのグラデーションカラー定義
	 */
	public abstract GradientDefine[] getAllDefines();

	/**
	 * 指定されたグラデーションカラー定義が含まれているか判定します。<br>
	 * @param define グラデーションカラー定義
	 * @return 指定されたグラデーションカラー定義が含まれている場合にtrueを返却
	 */
	public boolean contains(GradientDefine define) {
		if (define == null) {
			return false;
		} else {
			GradientDefine[] defines = getAllDefines();
			for (int i = 0; i <= defines.length - 1; i++) {
				if (define.equals(defines[i])) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * 指定されたグラデーションカラー定義を追加します。<br>
	 * @param define グラデーションカラー定義
	 */
	public void add(GradientDefine define) {
		if (define == null || contains(define)) {
			return;
		}
		GradientDefine[] defines = getAllDefines();
		for (int i = 0; i <= defines.length - 1; i++) {
			if (defines[i].location == define.location) {
				throw new IllegalArgumentException("指定された定義のロケーションは既に存在します(=" + define.location + ")");
			}
		}
		float location = define.location;
		if (location < this.location) {
			if (isTail()) {
				define.behindDefine = null;
				define.frontDefine = this;
				this.behindDefine = define;
			} else {
				if (location < behindDefine.location) {
					this.behindDefine.add(define);
				} else {
					define.behindDefine = this.behindDefine;
					define.frontDefine = this;
					this.behindDefine.frontDefine = define;
					this.behindDefine = define;
				}
			}
		} else {
			if (isHead()) {
				frontDefine = define;
				define.frontDefine = null;
				define.behindDefine = this;
			} else {
				if (location > frontDefine.location) {
					this.frontDefine.add(define);
				} else {
					define.frontDefine = this.frontDefine;
					define.behindDefine = this;
					this.frontDefine.behindDefine = define;
					this.frontDefine = define;
				}
			}
		}
	}

	/**
	 * 指定されたグラデーションカラー定義を削除します。<br>
	 * @param define グラデーションカラー定義
	 */
	public void remove(GradientDefine define) {
		if (define == null || (isHead() && isTail()) || define.getAllDefines().length < 2) {
			return;
		} else {
			if (define.isHead()) {
				define.behindDefine.frontDefine = null;
			} else if (define.isTail()) {
				define.frontDefine.behindDefine = null;
			} else {
				define.behindDefine.frontDefine = define.frontDefine;
				define.frontDefine.behindDefine = define.behindDefine;
			}
		}
	}

	/**
	 * クラス情報文字列を取得します。<br>
	 * @return クラス情報文字列
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer buffer = new StringBuffer();
		GradientDefine[] defines = getAllDefines();
		buffer.append("[");
		for (int i = 0; i <= defines.length - 1; i++) {
			buffer.append("{");
			buffer.append("Color=" + defines[i].getGradientColor());
			buffer.append(",");
			buffer.append("Location=" + defines[i].location);
			buffer.append("}");
			if (i < defines.length - 1) {
				buffer.append(",");
			}
		}
		buffer.append("]");
		return buffer.toString();
	}
}
