package org.phosphoresce.commons.thread;

/**
 * 処理非同期実行リスナインタフェース<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2009/05/15	Kitagawa		新規作成
 *-->
 */
public interface ProcessExecutorListener {

	/**
	 * 処理結果に対して値が更新された際の処理を行います。<br>
	 * このリスナメソッドは処理中の結果設定のタイミングで複数回呼び出されます。<br>
	 * すなわち実際の処理側においては、処理途中の結果を常に設定していくことでリスナはその進捗状況を把握することが可能となります。<br>
	 * @param executor 処理実行オブジェクト
	 * @param key 結果キー
	 * @param oldValue 更新前結果オブジェクト
	 * @param newValue 更新後結果オブジェクト
	 */
	public void updateResult(ProcessExecutor executor, String key, Object oldValue, Object newValue);

	/**
	 * 処理中メッセージが更新された際の処理を行います。<br>
	 * @param executor 処理実行オブジェクト
	 * @param message メッセージ(nullであることはありません)
	 */
	public void updateMessage(ProcessExecutor executor, String message);

	/**
	 * 処理進捗カウントが更新された際の処理を行います。<br>
	 * @param executor 処理実行オブジェクト
	 * @param min 最小進捗カウント
	 * @param max 最大進捗カウント
	 * @param count 現在進捗カウント
	 */
	public void updateProgress(ProcessExecutor executor, int min, int max, int count);

	/**
	 * 不特定進捗カウントフラグが更新された際の処理を行います。<br>
	 * @param executor 処理実行オブジェクト
	 * @param progressUnspecified 進捗カウントが不特定の場合にtrueを設定
	 */
	public void updateProgressUnspecified(ProcessExecutor executor, boolean progressUnspecified);

	/**
	 * 処理が開始されるタイミングの処理を行います。<br>
	 * 正確にはこのリスナメソッド処理が完了したタイミングで処理が実行されます。<br>
	 * @param executor 処理実行オブジェクト
	 */
	public void processStart(ProcessExecutor executor);

	/**
	 * 処理が完了したタイミングでの処理を行います。<br>
	 * キャンセルで終了した場合には{@link #processCanceled(ProcessExecutor)}が呼び出されます。<br>
	 * @param executor 処理実行オブジェクト
	 */
	public void processDone(ProcessExecutor executor);

	/**
	 * 処理のキャンセルタイミングでの処理を行います。<br>
	 * 正確にはこのキャンセルリスナ処理が完了したタイミングで処理のキャンセルが行われるため、
	 * このリスナ処理中には実際の処理スレッドは処理中であることに留意してください。<br>
	 * @param executor 処理実行オブジェクト
	 */
	public void processCanceling(ProcessExecutor executor);

	/**
	 * 処理がキャンセルされたタイミングの処理を行います。<br>
	 * このリスナ処理が呼び出される時点で実際の処理は既にキャンセルされ、停止していることになります。<br>
	 * @param executor 処理実行オブジェクト
	 */
	public void processCanceled(ProcessExecutor executor);

	/**
	 * 処理が例外の発生によって終了したタイミングでの処理を行います。<br>
	 * このリスナ処理が呼び出される時点で実際の処理は例外によって既に停止していることになります。<br>
	 * @param executor 処理実行オブジェクト
	 * @param e 処理中断要因となった例外オブジェクト
	 */
	public void processException(ProcessExecutor executor, Throwable e);
}
