package org.phosphoresce.commons.ldaps;

import java.io.Serializable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.BasicAttribute;

/**
 * LDAPオブジェクト属性クラス<br>
 *
 * @author Kitagawa<br>
 *
 *<!--
 * 更新日		更新者			更新内容
 * 2008/11/27	Kitagawa		新規作成
 *-->
 */
public class LDAPAttribute implements Serializable {

	/** 属性名 */
	private String name;

	/** 属性値 */
	private List values;

	/**
	 * コンストラクタ<br>
	 * @param name 属性名
	 */
	public LDAPAttribute(String name) {
		super();
		this.name = name;
		this.values = new LinkedList();
	}

	/**
	 * コンストラクタ<br>
	 */
	private LDAPAttribute() {
		this(null);
	}

	/**
	 * 指定された属性オブジェクトからこのクラスの情報に変換して提供します。<br>
	 * @param attribute 属性オブジェクト
	 * @return LDAPオブジェクト属性クラスインスタンス
	 * @throws NamingException LDAP属性オブジェクトからの情報取得を正常に行えなかった場合に発生
	 */
	public static LDAPAttribute createAttribute(Attribute attribute) throws NamingException {
		LDAPAttribute ldapAttribute = new LDAPAttribute(attribute.getID());
		NamingEnumeration enumeration = attribute.getAll();
		while (enumeration.hasMore()) {
			Object object = enumeration.next();
			ldapAttribute.addValue(object);
		}
		return ldapAttribute;
	}

	/**
	 * クラスの情報から属性オブジェクトを新規に生成します。<br>
	 * @return 属性オブジェクト
	 */
	public BasicAttribute createBasicAttribute() {
		BasicAttribute attribute = new BasicAttribute(name);
		for (Iterator iterator = values.iterator(); iterator.hasNext();) {
			attribute.add(iterator.next());
		}
		return attribute;
	}

	/**
	 * 属性名を取得します。<br>
	 * @return 属性名
	 */
	public String getName() {
		return name;
	}

	/**
	 * 属性値を取得します。<br>
	 * 提供されるリストの内部エレメントクラスは属性型に依存します。<br>
	 * @return 属性値
	 */
	public List getValues() {
		return values;
	}

	/**
	 * 指定されたインデックスの属性値を取得します。<br>
	 * @param index インデックス
	 * @return 属性値
	 */
	public Object getValue(int index) {
		return values.get(index);
	}

	/**
	 * 属性値を全てクリアします。<br>
	 */
	public void clearValues() {
		values.clear();
	}

	/**
	 * 属性値数を取得します。<br>
	 * @return 属性値数
	 */
	public int getValueCount() {
		return values.size();
	}

	/**
	 * 属性値を追加します。<br>
	 * @param value 属性値
	 */
	public void addValue(Object value) {
		values.add(value);
	}

	/**
	 * 指定されたインデックスに属性値を設定します。<br>
	 * @param index インデックス
	 * @param value 属性値
	 */
	public void setValue(int index, Object value) {
		values.set(index, value);
	}

	/**
	 * 指定された属性値が保持されているか判定します。<br>
	 * @param value 属性値
	 * @return 属性値が保持されている場合にtrueを返却
	 */
	public boolean containsValue(String value) {
		return values.contains(value);
	}

	/**
	 * オブジェクトの文字列表現を返します。<br>
	 * @return オブジェクトの文字列表現
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("{");
		buffer.append("name=");
		buffer.append(name);
		buffer.append(",");
		buffer.append("values=");
		buffer.append(values);
		buffer.append("}");
		return buffer.toString();
	}
}

