package org.phosphoresce.socket.proxy.http.util;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.io.StringWriter;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.phosphoresce.commons.socket.http.HttpMimeMapping;
import org.phosphoresce.commons.socket.http.container.HttpSocketResponse;
import org.phosphoresce.commons.socket.http.io.HttpSocketContentInputStream;
import org.phosphoresce.socket.proxy.http.HttpReverseProxyGlobal;
import org.phosphoresce.socket.proxy.http.config.HttpReverseProxyConfig;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;

/**
 * HTTPリバースプロキシサーバーVelocityリソースレスポンスファクトリクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/25	Kitagawa		新規作成
 *-->
 */
public class HttpReverseProxyVelocityResourceResponseFactory extends HttpReverseProxyResponseFactory {

	/** リソースパス */
	private String resource;

	/** MIMEタイプマッピング */
	private HttpMimeMapping mime;

	/** Velocityパラメータ */
	private Map parameter;

	/**
	 * コンストラクタ<br>
	 * @param serverSession サーバーセッション
	 * @param controllerSession コントローラーセッション
	 */
	public HttpReverseProxyVelocityResourceResponseFactory(HttpReverseProxyServerSession serverSession, HttpReverseProxyControllerSession controllerSession) {
		super(serverSession, controllerSession);
		this.parameter = new HashMap();
	}

	/**
	 * コンストラクタ<br>
	 */
	public HttpReverseProxyVelocityResourceResponseFactory() {
		super();
		this.parameter = new HashMap();
	}

	/**
	 * レスポンスオブジェクトを生成します。<br>
	 * @return レスポンスオブジェクト
	 * @throws Throwable 正常にレスポンスオブジェクトを生成できなかった場合に発生
	 * @see org.phosphoresce.socket.proxy.http.util.HttpReverseProxyResponseFactory#create()
	 */
	public HttpSocketResponse create() throws Throwable {
		HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();

		// Velocity拡張子ではない場合は静的リソースとして処理
		if (!resource.endsWith(config.getServerContentVelocityExtension())) {
			HttpReverseProxyStaticResourceResponseFactory factory = new HttpReverseProxyStaticResourceResponseFactory(serverSession, controllerSession);
			factory.setContentLength(getContentLength());
			factory.setMessage(getMessage());
			factory.setProtocol(getProtocol());
			factory.setStatus(getStatus());
			factory.setVersion(getVersion());
			String[] headers = getHeaderStrings();
			for (int i = 0; i <= headers.length - 1; i++) {
				String header = headers[i];
				factory.addHeader(header);
			}
			String[] cookies = getCookieStrings();
			for (int i = 0; i <= cookies.length - 1; i++) {
				String cookie = cookies[i];
				factory.addCookie(cookie);
			}
			factory.setMime(getMime());
			factory.setResource(getResource());
			return factory.create();
		}

		HttpSocketResponse response = createRenderResponse();

		// MIMEタイプ設定
		String extension = resource.endsWith(config.getServerContentVelocityExtension()) ? resource.substring(0, resource.length()
				- config.getServerContentVelocityExtension().length()) : resource;
		extension = extension.indexOf(".") >= 0 ? extension.substring(extension.lastIndexOf(".") + 1) : null;
		HttpMimeMapping mime = this.mime == null ? HttpMimeMapping.getMapping(extension) : this.mime;
		if (mime.getMimeType().startsWith("text")) {
			response.addHeader("Content-Type", mime.getMimeType() + "; charset=" + config.getServerContentCharset());
		} else {
			response.addHeader("Content-Type", mime.getMimeType());
		}

		// 共通Velocityパラメータ設定
		parameter.put(HttpReverseProxyGlobal.VELOCITY_BIND_KEY_CONFIG, HttpReverseProxyConfig.instance());
		parameter.put(HttpReverseProxyGlobal.VELOCITY_BIND_KEY_SERVER_SESSION, serverSession);
		parameter.put(HttpReverseProxyGlobal.VELOCITY_BIND_KEY_CONTROLLER_SESSION, controllerSession);

		// リソース情報取得
		String content = createVelocityContent(resource, parameter, config.getServerContentCharset());
		byte[] contentData = content.getBytes(config.getServerContentCharset());
		InputStream stream = new ByteArrayInputStream(contentData);
		int streamLength = contentData.length;

		// リソース日付設定
		response.addHeader("Date", HttpReverseProxyUtil.createDateHeaderString(new Date(new File(resource).lastModified())));

		// コンテンツストリーム設定
		response.setContentStream(new HttpSocketContentInputStream(stream, false, streamLength));

		return response;
	}

	/**
	 * 指定されたリソースを読み込みパラメータを設定したコンテンツ文字列を生成します。<br>
	 * @param resource Velocityテンプレートリソースパス
	 * @param parameter パラメータマップ
	 * @param charset 文字コード
	 * @return コンテンツ文字列
	 * @throws Exception 正常にVelocityテンプレートからコンテンツを生成できなかった場合にスローされます
	 */
	private String createVelocityContent(String resource, Map parameter, String charset) throws Exception {
		//Velocity.init();
		VelocityContext context = new VelocityContext(parameter);
		Template template = Velocity.getTemplate(resource, charset);
		StringWriter writer = new StringWriter();
		template.merge(context, writer);
		return writer.toString();
	}

	/**
	 * リソースパスを取得します。<br>
	 * @return リソースパス
	 */
	public String getResource() {
		return resource;
	}

	/**
	 * リソースパスを設定します。<br>
	 * @param resource リソースパス
	 */
	public void setResource(String resource) {
		this.resource = resource;
	}

	/**
	 * MIMEタイプマッピングを取得します。<br>
	 * @return MIMEタイプマッピング
	 */
	public HttpMimeMapping getMime() {
		return mime;
	}

	/**
	 * MIMEタイプマッピングを設定します。<br>
	 * @param mime MIMEタイプマッピング
	 */
	public void setMime(HttpMimeMapping mime) {
		this.mime = mime;
	}

	/**
	 * Velocityコンテンツパラメータを設定します。<br>
	 * @param name パラメータキー
	 * @param value パラメータ値
	 */
	public void putParameter(String name, Object value) {
		parameter.put(name, value);
	}

	/**
	 * Velocityコンテンツパラメータを取得します。<br>
	 * @param name パラメータキー
	 * @return パラメータ値
	 */
	public Object getParameter(String name) {
		return (String) parameter.get(name);
	}

	/**
	 * Velocityコンテンツパラメータを削除します。<br>
	 * @param name パラメータキー
	 */
	public void removeParameter(String name) {
		parameter.remove(name);
	}
}
