package org.phosphoresce.socket.proxy.http.util;

import org.phosphoresce.commons.socket.http.HttpResponseStatus;
import org.phosphoresce.commons.socket.http.container.HttpSocketCookie;
import org.phosphoresce.commons.socket.http.container.HttpSocketHeader;
import org.phosphoresce.commons.socket.http.container.HttpSocketResponse;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;

/**
 * HTTPリバースプロキシサーバーレスポンス基底ファクトリクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/25	Kitagawa		新規作成
 *-->
 */
public abstract class HttpReverseProxyResponseFactory {

	/** サーバーセッション */
	protected HttpReverseProxyServerSession serverSession;

	/** コントローラーセッション */
	protected HttpReverseProxyControllerSession controllerSession;

	/** ヘッダ情報保持用レスポンスオブジェクト */
	private HttpSocketResponse response;

	/**
	 * コンストラクタ<br>
	 * @param serverSession サーバーセッション
	 * @param controllerSession コントローラーセッション
	 */
	public HttpReverseProxyResponseFactory(HttpReverseProxyServerSession serverSession, HttpReverseProxyControllerSession controllerSession) {
		super();
		this.serverSession = serverSession;
		this.controllerSession = controllerSession;
		this.response = new HttpSocketResponse();
	}

	/**
	 * コンストラクタ<br>
	 */
	public HttpReverseProxyResponseFactory() {
		this(null, null);
	}

	/**
	 * レスポンスオブジェクトを生成します。<br>
	 * @return レスポンスオブジェクト
	 * @throws Throwable 正常にレスポンスオブジェクトを生成できなかった場合に発生
	 */
	public abstract HttpSocketResponse create() throws Throwable;

	/**
	 * createメソッドで提供するためのファクトリクラスに設定されたヘッダ情報を保持する
	 * 基底のレスポンスオブジェクトを新たに生成します。<br>
	 * @return レスポンスオブジェクト
	 */
	protected HttpSocketResponse createRenderResponse() {
		HttpSocketResponse response = new HttpSocketResponse();
		response.setContentLength(getContentLength());
		response.setMessage(getMessage());
		response.setProtocol(getProtocol());
		response.setStatus(getStatus());
		response.setVersion(getVersion());
		String[] headers = getHeaderStrings();
		for (int i = 0; i <= headers.length - 1; i++) {
			String header = headers[i];
			response.addHeader(header);
		}
		String[] cookies = getCookieStrings();
		for (int i = 0; i <= cookies.length - 1; i++) {
			String cookie = cookies[i];
			//response.addCookie(cookie);
			response.addHeader(cookie);
		}
		return response;
	}

	// Response Transfer Method

	/**
	 * 指定されたクッキー文字列からクッキー情報を追加します。<br>
	 * @param cookieString クッキー文字列
	 */
	public void addCookie(String cookieString) {
		response.addCookie(cookieString);
	}

	/**
	 * 指定されたキーと値でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * @param name ヘッダキー
	 * @param value ヘッダ値
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#addHeader(java.lang.String, int)
	 */
	public void addHeader(String name, int value) {
		response.addHeader(name, value);
	}

	/**
	 * 指定されたキーと値でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * @param name ヘッダキー
	 * @param value ヘッダ値
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#addHeader(java.lang.String, long)
	 */
	public void addHeader(String name, long value) {
		response.addHeader(name, value);
	}

	/**
	 * 指定されたキーと値でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * @param name ヘッダキー
	 * @param value ヘッダ値
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#addHeader(java.lang.String, java.lang.String)
	 */
	public void addHeader(String name, String value) {
		response.addHeader(name, value);
	}

	/**
	 * 指定されたヘッダ文字列からヘッダ情報を追加します。<br>
	 * ここにクッキー情報を保持するヘッダが指定された場合は、
	 * ヘッダ情報としてではなくクッキー情報として追加されます。<br>
	 * @param headerString ヘッダ文字列
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#addHeader(java.lang.String)
	 */
	public void addHeader(String headerString) {
		response.addHeader(headerString);
	}

	/**
	 * 指定されたクッキー文字列からクッキー情報を追加します。<br>
	 * 既に同一のクッキーキーでクッキー情報が存在する場合、
	 * 既存のクッキー情報は削除され新たにクッキー情報が追加されます。<br>
	 * @param cookieString クッキー文字列
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#putCookie(java.lang.String)
	 */
	public void putCookie(String cookieString) {
		response.putCookie(cookieString);
	}

	/**
	 * 指定されたキーと値でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * また、既に同一のヘッダキーでヘッダ情報が存在する場合、
	 * 既存のヘッダ情報は削除され新たにヘッダ情報が追加されます。<br>
	 * @param name ヘッダキー
	 * @param value ヘッダ値
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#putHeader(java.lang.String, int)
	 */
	public void putHeader(String name, int value) {
		response.putHeader(name, value);
	}

	/**
	 * 指定されたキーと値でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * また、既に同一のヘッダキーでヘッダ情報が存在する場合、
	 * 既存のヘッダ情報は削除され新たにヘッダ情報が追加されます。<br>
	 * @param name ヘッダキー
	 * @param value ヘッダ値
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#putHeader(java.lang.String, long)
	 */
	public void putHeader(String name, long value) {
		response.putHeader(name, value);
	}

	/**
	 * 指定されたキーと値でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * また、既に同一のヘッダキーでヘッダ情報が存在する場合、
	 * 既存のヘッダ情報は削除され新たにヘッダ情報が追加されます。<br>
	 * @param name ヘッダキー
	 * @param value ヘッダ値
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#putHeader(java.lang.String, java.lang.String)
	 */
	public void putHeader(String name, String value) {
		response.putHeader(name, value);
	}

	/**
	 * 指定されたヘッダ文字列でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * また、既に同一のヘッダキーでヘッダ情報が存在する場合、
	 * 既存のヘッダ情報は削除され新たにヘッダ情報が追加されます。<br>
	 * @param headerString ヘッダ文字列
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#putHeader(java.lang.String)
	 */
	public void putHeader(String headerString) {
		response.putHeader(headerString);
	}

	/**
	 * コンテンツボディデータの長さを設定します。<br>
	 * @param length コンテンツボディデータの長さ
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#setContentLength(long)
	 */
	public void setContentLength(long length) {
		response.setContentLength(length);
	}

	/**
	 * 指定されたレスポンス識別行をクラスに対して設定します。<br>
	 * @param identifier レスポンス識別行
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketResponse#setIdentifier(java.lang.String)
	 */
	public void setIdentifier(String identifier) {
		response.setIdentifier(identifier);
	}

	/**
	 * メッセージを設定します。<br>
	 * @param message メッセージ
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketResponse#setMessage(java.lang.String)
	 */
	public void setMessage(String message) {
		response.setMessage(message);
	}

	/**
	 * プロトコルを設定します。<br>
	 * @param protocol プロトコル
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketResponse#setProtocol(java.lang.String)
	 */
	public void setProtocol(String protocol) {
		response.setProtocol(protocol);
	}

	/**
	 * ステータスを設定します。<br>
	 * @param status ステータス
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketResponse#setStatus(org.phosphoresce.commons.socket.http.HttpResponseStatus)
	 */
	public void setStatus(HttpResponseStatus status) {
		response.setStatus(status);
	}

	/**
	 * プロトコルバージョンを設定します。<br>
	 * @param version プロトコルバージョン
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketResponse#setVersion(java.lang.String)
	 */
	public void setVersion(String version) {
		response.setVersion(version);
	}

	/**
	 * 指定されたキーのクッキー情報が保持されているか判定します。<br>
	 * @param name クッキーキー
	 * @return クッキー情報が保持されている場合にtrueを返却
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#containsCookie(java.lang.String)
	 */
	public boolean containsCookie(String name) {
		return response.containsCookie(name);
	}

	/**
	 * 指定されたキーのヘッダ情報が保持されているか判定します。<br>
	 * @param name ヘッダキー
	 * @return ヘッダ情報が保持されている場合にtrueを返却
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#containsHeader(java.lang.String)
	 */
	public boolean containsHeader(String name) {
		return response.containsHeader(name);
	}

	/**
	 * コンテンツボディデータの長さを取得します。<br>
	 * ヘッダ情報よりデータ長を特定できない場合は-1が返却されます。<br>
	 * @return コンテンツボディデータの長さ
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#getContentLength()
	 */
	public int getContentLength() {
		return response.getContentLength();
	}

	/**
	 * 指定されたクッキーキーで保持されているクッキー情報数を取得します。<br>
	 * @param name クッキーキー
	 * @return 保持されているクッキー情報数
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#getCookieCountOf(java.lang.String)
	 */
	public int getCookieCountOf(String name) {
		return response.getCookieCountOf(name);
	}

	/**
	 * 指定されたクッキーキーのクッキー情報を取得します。<br>
	 * @param name クッキーキー
	 * @return クッキー情報
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#getCookies(java.lang.String)
	 */
	public HttpSocketCookie[] getCookies(String name) {
		return response.getCookies(name);
	}

	/**
	 * 指定されたクッキーキーのクッキー情報を取得します。<br>
	 * @param name クッキーキー
	 * @return クッキー情報
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#getCookie(java.lang.String)
	 */
	public HttpSocketCookie getCookie(String name) {
		return response.getCookie(name);
	}

	/**
	 * クッキー文字列配列を取得します。<br>
	 * @return クッキー文字列配列
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#getCookieStrings()
	 */
	public String[] getCookieStrings() {
		return response.getCookieStrings();
	}

	/**
	 * 指定されたクッキーキーのクッキー値を取得します。<br>
	 * @param name クッキーキー
	 * @return クッキー値
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#getCookieValueOfInteger(java.lang.String)
	 */
	public int getCookieValueOfInteger(String name) {
		return response.getCookieValueOfInteger(name);
	}

	/**
	 * 指定されたクッキーキーのクッキー値を取得します。<br>
	 * @param name クッキーキー
	 * @return クッキー値
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#getCookieValueOfLong(java.lang.String)
	 */
	public long getCookieValueOfLong(String name) {
		return response.getCookieValueOfLong(name);
	}

	/**
	 * 指定されたクッキーキーのクッキー値を取得します。<br>
	 * @param name クッキーキー
	 * @return クッキー値
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#getCookieValueOfString(java.lang.String)
	 */
	public String getCookieValueOfString(String name) {
		return response.getCookieValueOfString(name);
	}

	/**
	 * 指定されたヘッダキーのヘッダ情報を取得します。<br>
	 * @param name ヘッダキー
	 * @return ヘッダ情報
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#getHeader(java.lang.String)
	 */
	public HttpSocketHeader getHeader(String name) {
		return response.getHeader(name);
	}

	/**
	 * ヘッダ文字列配列を取得します。<br>
	 * @return ヘッダ文字列配列
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#getHeaderStrings()
	 */
	public String[] getHeaderStrings() {
		return response.getHeaderStrings();
	}

	/**
	 * 指定されたヘッダキーのヘッダ値を取得します。<br>
	 * @param name ヘッダキー
	 * @return ヘッダ値
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#getHeaderValueOfInteger(java.lang.String)
	 */
	public int getHeaderValueOfInteger(String name) {
		return response.getHeaderValueOfInteger(name);
	}

	/**
	 * 指定されたヘッダキーのヘッダ値を取得します。<br>
	 * @param name ヘッダキー
	 * @return ヘッダ値
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#getHeaderValueOfLong(java.lang.String)
	 */
	public long getHeaderValueOfLong(String name) {
		return response.getHeaderValueOfLong(name);
	}

	/**
	 * 指定されたヘッダキーのヘッダ値を取得します。<br>
	 * @param name ヘッダキー
	 * @return ヘッダ値
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#getHeaderValueOfString(java.lang.String)
	 */
	public String getHeaderValueOfString(String name) {
		return response.getHeaderValueOfString(name);
	}

	/**
	 * レスポンス識別行を取得します。<br>
	 * @return レスポンス識別行
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketResponse#getIdentifier()
	 */
	public String getIdentifier() {
		return response.getIdentifier();
	}

	/**
	 * メッセージを取得します。<br>
	 * @return メッセージ
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketResponse#getMessage()
	 */
	public String getMessage() {
		return response.getMessage();
	}

	/**
	 * プロトコルを取得します。<br>
	 * @return プロトコル
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketResponse#getProtocol()
	 */
	public String getProtocol() {
		return response.getProtocol();
	}

	/**
	 * ステータスを取得します。<br>
	 * @return ステータス
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketResponse#getStatus()
	 */
	public HttpResponseStatus getStatus() {
		return response.getStatus();
	}

	/**
	 * プロトコルバージョンを取得します。<br>
	 * @return プロトコルバージョン
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketResponse#getVersion()
	 */
	public String getVersion() {
		return response.getVersion();
	}

	/**
	 * コンテンツボディデータがChunkedデータであるかを判定します。<br>
	 * @return コンテンツボディデータがChunkedデータである場合にtrueを返却
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#isChunkedContent()
	 */
	public boolean isChunkedContent() {
		return response.isChunkedContent();
	}

	/**
	 * コネクションがKeep-Aliveであるか判定します。<br>
	 * @return Keep-Aliveである場合にtrueを返却
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#isKeepAliveConnection()
	 */
	public boolean isKeepAliveConnection() {
		return response.isKeepAliveConnection();
	}

	/**
	 * 指定されたクッキーキーのクッキー情報を削除します。<br>
	 * 同一キーで複数のクッキー情報が存在する場合はそれら全てが削除されます。<br>
	 * @param name クッキーキー
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#removeCookie(java.lang.String)
	 */
	public void removeCookie(String name) {
		response.removeCookie(name);
	}

	/**
	 * 指定されたヘッダキーのヘッダ情報を削除します。<br>
	 * 同一キーで複数のヘッダ情報が存在する場合はそれら全てが削除されます。<br>
	 * @param name ヘッダキー
	 * @see org.phosphoresce.commons.socket.http.container.HttpSocketHeaderContainer#removeHeader(java.lang.String)
	 */
	public void removeHeader(String name) {
		response.removeHeader(name);
	}
}
