package org.phosphoresce.socket.proxy.http.server;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.ConnectException;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.phosphoresce.commons.socket.http.HttpResponseStatus;
import org.phosphoresce.commons.socket.http.container.HttpSocketResponse;
import org.phosphoresce.commons.socket.http.util.HttpSocketClientUtil;
import org.phosphoresce.commons.util.FileUtil;
import org.phosphoresce.commons.util.StringUtil;
import org.phosphoresce.socket.proxy.http.config.HttpReverseProxyConfig;
import org.phosphoresce.socket.proxy.http.handler.HttpReverseProxyHandler;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;
import org.phosphoresce.socket.proxy.http.util.HttpReverseProxyVelocityResourceResponseFactory;

/**
 * HTTPリバースプロキシコントロールスレッドクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/08	Kitagawa		新規作成
 *-->
 */
public class HttpReverseProxyController extends Thread {

	/** ロギングオブジェクト */
	private Log log = LogFactory.getLog(getClass());

	/** サーバーセッションオブジェクト */
	private HttpReverseProxyServerSession serverSession;

	/** コントローラーセッションオブジェクト */
	private HttpReverseProxyControllerSession controllerSession;

	/**
	 * コンストラクタ<br>
	 * @param serverSession サーバーセッションオブジェクト
	 */
	public HttpReverseProxyController(HttpReverseProxyServerSession serverSession, HttpReverseProxyControllerSession controllerSession) {
		super();
		this.serverSession = serverSession;
		this.controllerSession = controllerSession;
	}

	/**
	 * コンストラクタ<br>
	 * @throws IOException ソケットストリームの確立に失敗した場合にスローされます
	 */
	private HttpReverseProxyController() throws IOException {
		this(null, null);
	}

	/**
	 * HTTPソケットデータプロキシ処理を実行します。<br>
	 * @see java.lang.Thread#run()
	 */
	public void run() {
		// TODO Keep-Alive処理が未実装
		try {
			while (execute())
				;
		} catch (Throwable e) {
			handleException(e);
		} finally {
			try {
				close();
			} catch (Throwable e) {
				log.error("Failed to close socket connection", e);
			}
		}
	}

	/**
	 * ソケットストリーム及び、ソケットをクローズします。<br>
	 * @throws IOException 正常にソケットのクローズが行えなかった場合にスローされます
	 */
	public void close() throws IOException {
		if (controllerSession.getForwardSocket() != null) {
			controllerSession.getForwardSocket().close();
		}
		if (controllerSession.getClientSocket() != null) {
			controllerSession.getClientSocket().close();
		}
	}

	/**
	 * 指定されたクラスからハンドラインスタンスを生成します。<br>
	 * @param clazz ハンドラクラス
	 * @return ハンドラクラスインスタンス
	 * @throws Throwable 正常にクラスからインスタンスを生成できなかった場合に発生
	 */
	private HttpReverseProxyHandler createHandlerInstance(Class clazz) throws Throwable {
		Constructor constructor = clazz.getConstructor(new Class[] { HttpReverseProxyServerSession.class, HttpReverseProxyControllerSession.class });
		HttpReverseProxyHandler handler = (HttpReverseProxyHandler) constructor.newInstance(new Object[] { serverSession, controllerSession });
		return handler;
	}

	/**
	 * ソケットデータのリレー処理を実行します。<br>
	 * @return 再起処理を行う場合にtrueを返却
	 * @throws Throwable ソケットデータリレー処理中に予期せぬ例外が発生した場合にスローされます
	 */
	public boolean execute() throws Throwable {
		// 環境定義オブジェクト取得
		HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();

		// リクエスト受信側ハンドラ実行
		for (Iterator iterator = config.getReciveRequestHandlerIterator(); iterator.hasNext();) {
			Class clazz = (Class) iterator.next();
			HttpReverseProxyHandler handler = createHandlerInstance(clazz);
			if (!handler.execute()) {
				return false;
			}
		}

		// TODO レスポンスステータスが100のときは明示的にループする必要がある？(IISのバグのせい？)
		while (true) {
			// レスポンス転送側ハンドラ実行
			for (Iterator iterator = config.getForwardResponseHandlerIterator(); iterator.hasNext();) {
				Class clazz = (Class) iterator.next();
				HttpReverseProxyHandler handler = createHandlerInstance(clazz);
				if (!handler.execute()) {
					return false;
				}
			}

			HttpSocketResponse response = controllerSession.getResponse();
			HttpResponseStatus status = response == null ? null : response.getStatus();
			int statusCode = status == null ? -1 : status.getCode();
			if (statusCode != 100) {
				break;
			}
		}

		return false;
	}

	/**
	 * 例外オブジェクトをハンドルし、エラー処理をします。<br>
	 * @param throwable 例外オブジェクト
	 */
	public void handleException(Throwable throwable) {
		// 例外情報取得
		String message = throwable == null ? "null" : throwable.toString();
		String timestamp = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS").format(new Date());
		String identifier = StringUtil.toMD5Hash(timestamp);

		// クライアントソケットクローズ状態の場合はログ出力のみ
		Socket clientSocket = controllerSession.getClientSocket();
		if (clientSocket == null || clientSocket.isClosed()) {
			log.error("SERVER-CONTROLLER-ERROR (" + identifier + ")", throwable);
			return;
		}

		try {
			// 例外ごとのサーバーレスポンス処理を行う
			if (throwable instanceof SocketTimeoutException) {
				// タイムアウトエラー
				responseTimeoutError(throwable, timestamp, message, identifier);
			} else if (throwable instanceof ConnectException) {
				// ソケット接続エラー
				responseFatalError(throwable, timestamp, message, identifier);
			} else {
				// その他の例外を致命的として処理
				responseFatalError(throwable, timestamp, message, identifier);
			}
		} catch (Throwable e) {
			// エラーコンテンツ出力で失敗した場合はテキストレベルのページの出力を試みる
			StringBuffer buffer = new StringBuffer();
			buffer.append("<html>");
			buffer.append("<head>");
			buffer.append("<title>Internal Server Error</title>");
			buffer.append("<style><!-- * {font-family:Verdana;font-size:12px;} --></style>");
			buffer.append("</head>");
			buffer.append("<body>");
			buffer.append("<b>Internal Server Error (Sorry, can't handled exception)</b><br>");
			buffer.append("<br>");
			buffer.append("Exception Information").append("<br>");
			buffer.append("Timestamp : ").append(timestamp).append("<br>");
			buffer.append("Message : ").append(message).append("<br>");
			buffer.append("Identifier : ").append(identifier).append("<br>");
			buffer.append("</body>");
			buffer.append("</html>");
			byte[] content = buffer.toString().getBytes();
			HttpSocketResponse response = new HttpSocketResponse();
			response.setStatus(HttpResponseStatus.STATUS_500_INTERNAL_SERVER_ERROR);
			response.setContentStream(new ByteArrayInputStream(content));
			response.setContentLength(content.length);
			try {
				HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();
				HttpSocketClientUtil.writeResponse(clientSocket.getOutputStream(), response, config.getSocketBufferSize());
			} catch (Throwable e1) {
				// エラーレスポンス出力で失敗した場合は無視(Log4J側に情報出力しているため)
			}
		}
	}

	/**
	 * サーバーの致命的な例外をクライアントに対してレスポンスします。<br>
	 * @param throwable 発生した例外オブジェクト
	 * @param timestamp 発生タイムスタンプ
	 * @param message 例外メッセージ
	 * @param identifier 例外追跡用ユニークキー
	 * @throws Throwable レスポンス出力時にエラーが発生した場合に元の例外をスローします(レスポンス出力時の例外はスローしません)
	 */
	public void responseFatalError(Throwable throwable, String timestamp, String message, String identifier) throws Throwable {
		try {
			HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();
			HttpReverseProxyVelocityResourceResponseFactory factory = new HttpReverseProxyVelocityResourceResponseFactory(serverSession, controllerSession);
			factory.setStatus(HttpResponseStatus.STATUS_500_INTERNAL_SERVER_ERROR);
			factory.setResource(FileUtil.connectPath(new String[] { config.getServerContentRootPath(), config.getServerContentAlias(), config.getServerFatalErrorContent() }));
			factory.putParameter("timestamp", timestamp);
			factory.putParameter("message", message);
			factory.putParameter("identifier", identifier);
			HttpSocketClientUtil.writeResponse(controllerSession.getClientSocket().getOutputStream(), factory.create(), config.getSocketBufferSize());

			log.error("SERVER-CONTROLLER-ERROR (" + identifier + ")", throwable);
		} catch (Throwable e) {
			//throw e;
			throw throwable;
		}
	}

	/**
	 * サーバーのタイムアウトをクライアントに対してレスポンスします。<br>
	 * @param throwable 発生した例外オブジェクト
	 * @param timestamp 発生タイムスタンプ
	 * @param message 例外メッセージ
	 * @param identifier 例外追跡用ユニークキー
	 * @throws Throwable レスポンス出力時にエラーが発生した場合に元の例外をスローします(レスポンス出力時の例外はスローしません)
	 */
	public void responseTimeoutError(Throwable throwable, String timestamp, String message, String identifier) throws Throwable {
		try {
			HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();
			HttpReverseProxyVelocityResourceResponseFactory factory = new HttpReverseProxyVelocityResourceResponseFactory(serverSession, controllerSession);
			factory.setStatus(HttpResponseStatus.STATUS_500_INTERNAL_SERVER_ERROR);
			factory.setResource(FileUtil.connectPath(new String[] { config.getServerContentRootPath(), config.getServerContentAlias(), config.getServerTimeoutErrorContent() }));
			factory.putParameter("timestamp", timestamp);
			factory.putParameter("message", message);
			factory.putParameter("identifier", identifier);
			factory.putParameter("siteUrl", controllerSession.getRequest().getPath());
			HttpSocketClientUtil.writeResponse(controllerSession.getClientSocket().getOutputStream(), factory.create(), config.getSocketBufferSize());

			log.debug("SERVER-CONTROLLER-ERROR (" + identifier + ")", throwable);
		} catch (Throwable e) {
			//throw e;
			throw throwable;
		}
	}
}
