package org.phosphoresce.socket.proxy.http.handler;

import java.net.Socket;

import org.phosphoresce.commons.socket.http.container.HttpSocketRequest;
import org.phosphoresce.commons.socket.http.util.HttpSocketClientUtil;
import org.phosphoresce.socket.proxy.http.config.HttpReverseProxyConfig;
import org.phosphoresce.socket.proxy.http.config.RequestForwardConfig;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;
import org.phosphoresce.socket.proxy.http.util.HttpReverseProxyVelocityResourceResponseFactory;

/**
 * HTTPリバースプロキシリクエスト転送先確立ハンドラークラス<br>
 * <br>
 * 当ハンドラクラスは以下の仕様で動作させることを想定しています。<br>
 * <b>前提</b>
 * ・コントローラーセッションにクライアントソケットオブジェクトが設定されていること<br>
 * ・コントローラーセッションにクライアントリクエスト情報が設定されていること<br>
 * <br>
 * <b>処理</b>
 * ・リクエストされたパスが転送定義に設定されたパス以外の場合はエラーレスポンスを送信<br>
 * ・転送定義が特定できた場合は転送先ソケットを確立し、コントローラーセッションに設定<br>
 * <br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/17	Kitagawa		新規作成
 *-->
 */
public class RequestForwardEstablishHandler extends HttpReverseProxyHandlerAdapter {

	/**
	 * コンストラクタ<br>
	 * @param serverSession サーバーセッションオブジェクト
	 * @param controllerSession コントローラーセッションオブジェクト
	 */
	public RequestForwardEstablishHandler(HttpReverseProxyServerSession serverSession, HttpReverseProxyControllerSession controllerSession) {
		super(serverSession, controllerSession);
	}

	/**
	 * イベントハンドル処理を実行します。<br>
	 * @return イベントハンドル処理後、プロキシコントローラーの後続処理を実施する場合にtrueを返却
	 * @throws Throwable 正常にイベントハンドル処理が行えなかった場合にスローされます
	 * @see org.phosphoresce.socket.proxy.http.handler.HttpReverseProxyHandler#execute()
	 */
	public boolean execute() throws Throwable {
		// 共通フィールド取得
		HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();
		HttpReverseProxyServerSession serverSession = getServerSession();
		HttpReverseProxyControllerSession controllerSession = getControllerSession();
		Socket clientSocket = getClientSocket();
		HttpSocketRequest request = getRequest();

		// フォワード定義取得
		RequestForwardConfig forwardConfig = config.getForwardingConfig(getRequest().getPath());

		// リクエストされたパスが転送定義に設定されたパス以外の場合はエラーレスポンスを送信
		if (forwardConfig == null) {
			log.debug("UNDEFINED " + getRequest().getPath());
			HttpReverseProxyVelocityResourceResponseFactory factory = new HttpReverseProxyVelocityResourceResponseFactory(serverSession, controllerSession);
			factory.setResource(config.getServerContentResourcePath(config.getServerInvalidMappingContent()));
			HttpSocketClientUtil.writeResponse(clientSocket.getOutputStream(), factory.create(), config.getSocketBufferSize());
			return false;
		}

		// 転送定義が特定できた場合は転送先ソケットを確立し、コントローラーセッションに設定
		if (forwardConfig != null) {
			log.trace("UNDEFINED Check validity " + getRequest().getPath());
			controllerSession.setData(RequestForwardConfig.SESSION_KEY, forwardConfig);
			setForwardSocket(new Socket(forwardConfig.getForwardAddress(), forwardConfig.getForwardPort()));
			getForwardSocket().setKeepAlive(true);
			getForwardSocket().setSoLinger(true, 10);
			getForwardSocket().setTcpNoDelay(true);
			getForwardSocket().setReceiveBufferSize(config.getSocketBufferSize());
			getForwardSocket().setSendBufferSize(config.getSocketBufferSize());
			getForwardSocket().setSoTimeout(config.getSocketTimeout());
			return true;
		}

		return false;
	}
}
