package org.phosphoresce.socket.proxy.http.handler;

import java.net.Socket;

import org.phosphoresce.commons.socket.http.container.HttpSocketRequest;
import org.phosphoresce.commons.socket.http.container.HttpSocketResponse;
import org.phosphoresce.commons.socket.http.util.HttpSocketClientUtil;
import org.phosphoresce.commons.socket.http.util.HttpSocketLoggingUtil;
import org.phosphoresce.socket.proxy.http.config.HttpReverseProxyConfig;
import org.phosphoresce.socket.proxy.http.config.RequestForwardConfig;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;

/**
 * HTTPリバースプロキシアプリケーションレスポンス転送ハンドラークラス<br>
 * <br>
 * 当ハンドラクラスは以下の仕様で動作させることを想定しています。<br>
 * <b>前提</b>
 * ・コントローラーセッションにクライアントソケットオブジェクトが設定されていること<br>
 * ・コントローラーセッションにクライアントリクエスト情報が設定されていること<br>
 * ・コントローラーセッションに転送先ソケットオブジェクトが設定されていること<br>
 * ・コントローラーセッションに転送先レスポンス情報が設定されていること<br>
 * <br>
 * <b>処理</b>
 * ・コントローラーセッション上のレスポンス情報をクライアントに送信<br>
 * <br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/11	Kitagawa		新規作成
 *-->
 */
public class ForwardResponseHandler extends HttpReverseProxyHandlerAdapter {

	/**
	 * コンストラクタ<br>
	 * @param serverSession サーバーセッションオブジェクト
	 * @param controllerSession コントローラーセッションオブジェクト
	 */
	public ForwardResponseHandler(HttpReverseProxyServerSession serverSession, HttpReverseProxyControllerSession controllerSession) {
		super(serverSession, controllerSession);
	}

	/**
	 * イベントハンドル処理を実行します。<br>
	 * @return イベントハンドル処理後、プロキシコントローラーの後続処理を実施する場合にtrueを返却
	 * @throws Throwable 正常にイベントハンドル処理が行えなかった場合にスローされます
	 * @see org.phosphoresce.socket.proxy.http.handler.HttpReverseProxyHandler#execute()
	 */
	public boolean execute() throws Throwable {
		// 共通フィールド取得
		HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();
		HttpReverseProxyServerSession serverSession = getServerSession();
		HttpReverseProxyControllerSession controllerSession = getControllerSession();
		Socket clientSocket = getClientSocket();
		String clientAddress = getClientSocket().getInetAddress().getHostAddress();
		HttpSocketRequest request = getRequest();
		HttpSocketResponse response = getResponse();

		// コントローラーセッション上のレスポンス情報をクライアントに送信
		HttpSocketClientUtil.writeResponse(clientSocket.getOutputStream(), response, config.getSocketBufferSize());

		// ログ出力のみ(転送先定義取得は業務的には不要)
		RequestForwardConfig fConfig = (RequestForwardConfig) getControllerSessionData(RequestForwardConfig.SESSION_KEY);
		if (HttpSocketLoggingUtil.isTraceTarget(getRequest())) {
			log.debug("FORWARD " + clientAddress + " -> " + getRequest().getPath() + " -> " + fConfig.getForwardAddress() + ":" + fConfig.getForwardPort());
		}

		return true;
	}
}
