package org.phosphoresce.socket.proxy.http.daemon;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.phosphoresce.socket.proxy.http.HttpReverseProxyGlobal;
import org.phosphoresce.socket.proxy.http.exception.DaemonServiceException;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;

/**
 * リバースプロキシサーバーデーモンサービスレジストリクラス<br>
 * <br>
 * 当クラスは全てのデーモンサービスを統括管理し、サービスデーモンの
 * 状況を常に監視し、破棄対象のデーモンの破棄を行ったりします。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/12/05	Kitagawa		新規作成
 *-->
 */
public final class HttpReverseProxyDaemonRegistry {

	/** ロギングオブジェクト */
	private static Log log = LogFactory.getLog(HttpReverseProxyDaemonRegistry.class);

	/** セルフインスタンス */
	private static HttpReverseProxyDaemonRegistry instance;

	/** サーバーセッションオブジェクト */
	private HttpReverseProxyServerSession serverSession;

	/** デーモン監視スレッド */
	private RegistryWatchThread watchThread;

	/** デーモンサービスキャッシュ */
	private Map daemons;

	/**
	 * コンストラクタ<br>
	 */
	private HttpReverseProxyDaemonRegistry() {
		this.daemons = new LinkedHashMap();
	}

	/**
	 * デーモンサービスインスタンスを取得します。<br>
	 * @return デーモンサービスインスタンス
	 * @throws DaemonServiceException デーモンサービスが開始されていない状態で取得しようとした場合にスローされます
	 */
	public static HttpReverseProxyDaemonRegistry instance() throws DaemonServiceException {
		if (instance == null) {
			instance = new HttpReverseProxyDaemonRegistry();
		}
		return instance;
	}

	/**
	 * デーモンサービスを開始します。<br>
	 * 当メソッドが呼び出されたタイミングでデーモンサービスインスタンスが初期化されます。<br>
	 * @param serverSession サーバーセッションオブジェクト
	 */
	public static void start(HttpReverseProxyServerSession serverSession) {
		if (instance == null) {
			instance = new HttpReverseProxyDaemonRegistry();
		}
		// 監視デーモンスタート
		instance.serverSession = serverSession;
		instance.startWatchThread();
		log.info("Server daemon service registry start");
	}

	/**
	 * デーモンサービス監視スレッドを開始します。<br>
	 */
	private void startWatchThread() {
		if (watchThread == null) {
			watchThread = new RegistryWatchThread();
			watchThread.start();
		} else {
			log.warn("Server daemon service registry watch thread already started");
		}
	}

	/**
	 * サーバーセッションオブジェクトを取得します。<br>
	 * @return サーバーセッションオブジェクト
	 */
	public HttpReverseProxyServerSession getServerSession() {
		return serverSession;
	}

	/**
	 * デーモンサービスプロセスIDを生成します。<br>
	 * @return デーモンサービスプロセスID
	 */
	private String createProcessId() {
		return String.valueOf(System.currentTimeMillis());
	}

	/**
	 * 登録されているデーモンサービスリストサイズを取得します。<br>
	 * @return デーモンサービスリストサイズ
	 */
	public int size() {
		return daemons.size();
	}

	/**
	 * プロセスIDキーセットを提供します。<br>
	 * @return プロセスIDキーセット
	 */
	public Set keySet() {
		return daemons.keySet();
	}

	/**
	 * デーモンサービスをレジストリに登録します。<br>
	 * また、対象のデーモンサービスが妥当な状態の場合に登録と同時にサービスが開始されます。<br>
	 * @param daemon デーモンサービスオブジェクト
	 * @return 登録されたデーモンサービスのプロセスID
	 */
	public String add(HttpReverseProxyDaemon daemon) {
		String pid = createProcessId();
		daemons.put(pid, daemon);
		if (daemon.isValidity() && !daemon.isProcessing() && !daemon.isHalted() && !daemon.isHaltSignaled() && !daemon.isStopSignaled()) {
			daemon.setRegistry(this);
			daemon.start();
		}
		log.debug("Server daemon service thread added (" + daemon.getName() + " , pid=" + pid + ")");
		return pid;
	}

	/**
	 * 指定されたプロセスIDのサービスデーモンオブジェクトを取得します。<br>
	 * @param pid デーモンサービスプロセスID
	 * @return サービスデーモンオブジェクト
	 */
	public HttpReverseProxyDaemon get(String pid) {
		return (HttpReverseProxyDaemon) daemons.get(pid);
	}

	/**
	 * 指定されたクラスで保持されているデーモンオブジェクトを全て取得します。<br>
	 * @param clazz デーモンクラス
	 * @return デーモンオブジェクト配列
	 */
	public HttpReverseProxyDaemon[] gets(Class clazz) {
		List list = new LinkedList();
		for (Iterator iterator = daemons.keySet().iterator(); iterator.hasNext();) {
			String pid = (String) iterator.next();
			HttpReverseProxyDaemon daemon = get(pid);
			if (daemon != null && clazz.isInstance(daemon)) {
				list.add(daemon);
			}
		}
		return (HttpReverseProxyDaemon[]) list.toArray(new HttpReverseProxyDaemon[list.size()]);
	}

	/**
	 * デーモンサービスレジストリ監視スレッドクラス<br>
	 * 
	 * @author Kitagawa<br>
	 * 
	 *<!--
	 * 更新日		更新者			更新内容
	 * 2008/12/05	Kitagawa		新規作成
	 *-->
	 */
	private final class RegistryWatchThread extends Thread {

		/**
		 * 登録済みデーモンサービスを監視します。<br>
		 * @see java.lang.Thread#run()
		 */
		public final void run() {
			while (true) {
				try {
					List removeList = new LinkedList();
					for (Iterator iterator = daemons.keySet().iterator(); iterator.hasNext();) {
						String pid = (String) iterator.next();
						HttpReverseProxyDaemon daemon = get(pid);
						if (daemon != null && daemon.isHalted()) {
							removeList.add(pid);
						}
					}
					for (Iterator iterator = removeList.iterator(); iterator.hasNext();) {
						String pid = (String) iterator.next();
						HttpReverseProxyDaemon daemon = (HttpReverseProxyDaemon) daemons.remove(pid);
						daemon.setRegistry(null);
						log.debug("Server daemon service thread removed (" + daemon.getName() + " , pid=" + pid + ")");
						daemon = null;
					}
					Thread.sleep(HttpReverseProxyGlobal.SERVER_THREAD_DELAY);
				} catch (Throwable e) {
					log.error("Server daemon service registry error", e);
				}
			}
		}
	}
}
