package org.phosphoresce.socket.proxy.http.daemon;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;

/**
 * リバースプロキシサーバーデーモン上位抽象クラス<br>
 * <br>
 * 当クラスは各種デーモンサービスクラスの上位に当たるクラスで、
 * デーモンレジストリクラスとの連携インターフェース及び、
 * デーモンに対する操作インタフェースを提供します。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/12/05	Kitagawa		新規作成
 *-->
 */
public abstract class HttpReverseProxyDaemon {

	/** ロギングオブジェクト */
	protected final Log log = LogFactory.getLog(getClass());

	/** 自身が所属するデーモンレジストリ */
	private HttpReverseProxyDaemonRegistry registry;

	/** デーモン名 */
	private String name;

	/** サービスデーモンループ処理時の間隔(ミリ秒) */
	private long loopDelay;

	/** デーモンサービスのループカウント(負数で永続的ループ) */
	private int loopCount;

	/** デーモンサービス処理中にエラーが発生した場合のリトライ回数(負数で永続的ループ) */
	private int retryCountOnError;

	/** 実行回数 */
	private int executeCount;

	/** 実行中発生例外(直前例外) */
	private Throwable exeception;

	/** 実行中例外発生回数 */
	private int exceptionCount;

	/** 実行中フラグ */
	private boolean processing;

	/** デーモン一時停止フラグ */
	private boolean stopSignaled;

	/** デーモン完全停止フラグ */
	private boolean haltSignaled;

	/**
	 * コンストラクタ<br>
	 * @param name デーモン名 
	 * @param loopDelay サービスデーモンループ処理時の間隔(ミリ秒)
	 * @param loopCount デーモンサービスのループカウント(負数で永続的ループ)
	 * @param retryCountOnError デーモンサービス処理中にエラーが発生した場合のリトライ回数(負数で永続的ループ)
	 */
	public HttpReverseProxyDaemon(String name, long loopDelay, int loopCount, int retryCountOnError) {
		super();
		this.name = name;
		this.loopDelay = loopDelay;
		this.loopCount = loopCount;
		this.retryCountOnError = retryCountOnError;
		this.executeCount = 0;
		this.exceptionCount = 0;
		this.exeception = null;
		this.processing = false;
		this.stopSignaled = false;
		this.haltSignaled = false;
	}

	/**
	 * 
	 * コンストラクタ<br>
	 */
	private HttpReverseProxyDaemon() {
		this(null, 1000, -1, -1);
	}

	/**
	 * サービスデーモンレジストリを設定します。<br>
	 * @param registry サービスデーモンレジストリ
	 */
	final void setRegistry(HttpReverseProxyDaemonRegistry registry) {
		this.registry = registry;
	}

	/**
	 * サーバーセッションオブジェクトを取得します。<br>
	 * @return サーバーセッションオブジェクト
	 */
	public final HttpReverseProxyServerSession getServerSession() {
		return registry == null ? null : registry.getServerSession();
	}

	/**
	 * デーモン名を取得します。<br>
	 * @return デーモン名
	 */
	public String getName() {
		return name;
	}

	/**
	 * デーモン名を設定します。<br>
	 * @param name デーモン名
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * サービスデーモンループ処理時の間隔(ミリ秒)を取得します。<br>
	 * @return サービスデーモンループ処理時の間隔(ミリ秒)
	 */
	public long getLoopDelay() {
		return loopDelay;
	}

	/**
	 * サービスデーモンループ処理時の間隔(ミリ秒)を設定します。<br>
	 * @param loopDelay サービスデーモンループ処理時の間隔(ミリ秒)
	 */
	public void setLoopDelay(long loopDelay) {
		this.loopDelay = loopDelay;
	}

	/**
	 * デーモンサービスのループカウント(負数で永続的ループ)を取得します。<br>
	 * @return デーモンサービスのループカウント(負数で永続的ループ)
	 */
	public int getLoopCount() {
		return loopCount;
	}

	/**
	 * デーモンサービスのループカウント(負数で永続的ループ)を設定します。<br>
	 * @param loopCount デーモンサービスのループカウント(負数で永続的ループ)
	 */
	public void setLoopCount(int loopCount) {
		this.loopCount = loopCount;
	}

	/**
	 * デーモンサービス処理中にエラーが発生した場合のリトライ回数(負数で永続的ループ)を取得します。<br>
	 * @return デーモンサービス処理中にエラーが発生した場合のリトライ回数(負数で永続的ループ)
	 */
	public int getRetryCountOnError() {
		return retryCountOnError;
	}

	/**
	 * デーモンサービス処理中にエラーが発生した場合のリトライ回数(負数で永続的ループ)を設定します。<br>
	 * @param retryCountOnError デーモンサービス処理中にエラーが発生した場合のリトライ回数(負数で永続的ループ)
	 */
	public void setRetryCountOnError(int retryCountOnError) {
		this.retryCountOnError = retryCountOnError;
	}

	/**
	 * 実行中例外発生回数を取得します。<br>
	 * @return 実行中例外発生回数
	 */
	public final int getExceptionCount() {
		return exceptionCount;
	}

	/**
	 * 実行中発生例外(直前例外)を取得します。<br>
	 * @return 実行中発生例外(直前例外)
	 */
	public final Throwable getExeception() {
		return exeception;
	}

	/**
	 * 実行回数を取得します。<br>
	 * @return 実行回数
	 */
	public final int getExecuteCount() {
		return executeCount;
	}

	/**
	 * デーモンサービスに対して一時停止指示が行われているか判定します。<br>
	 * 但し、一時停止指示が行われていた場合でも現在の処理が実行中である場合があることに留意してください。<br>
	 * @return デーモンサービスに対して一時停止指示が行われている場合にtrueを返却します
	 */
	public final boolean isStopSignaled() {
		return stopSignaled;
	}

	/**
	 * デーモン完全停止フラグを取得します。<br>
	 * 但し、完全停止指示が行われていた場合でも現在の処理が実行中である場合があることに留意してください。<br>
	 * @return デーモン完全停止フラグ
	 */
	public boolean isHaltSignaled() {
		return haltSignaled;
	}

	/**
	 * デーモンサービスが実行中であるか判定します。<br>
	 * 但し、現在の処理が実行中である場合でも一時停止指示が行われてい場合は現在の処理が完了したタイミングで一時停止状態となります。<br>
	 * @return デーモンサービスが実行中である場合にtrueを返却します
	 */
	public final boolean isProcessing() {
		return processing;
	}

	/**
	 * デーモンサービスが完全停止状態であるか判定します。<br>
	 * 判定条件はデーモンサービスが無効であり、かつ実行中でない場合、又は、
	 * 完全停止指示が行われ、かつ実行中でない場合に完全停止状態と判定します。<br>
	 * @return デーモンサービスが完全停止状態である場合にtrueを返却します
	 */
	public final boolean isHalted() {
		return (!isValidity() && !isProcessing()) || (haltSignaled && !isProcessing());
	}

	/**
	 * このデーモンサービスが有効であるか判定します。<br>
	 * ループカウント数の実行が終わっている場合、エラー発生数が限界値を超えた場合などに無効として判定します。<br>
	 * 但し、デーモン一時停止フラグ、完全停止フラグに関しては当メソッドでの判断条件とはなりません。<br>
	 * @return デーモンサービスが有効である場合にtrueを返却
	 */
	public final boolean isValidity() {
		boolean execute = true;
		// 実行回数チェック
		if (execute && loopCount >= 0 && executeCount >= loopCount) {
			execute = false;
		}
		// 実行中例外カウントチェック
		if (execute && retryCountOnError >= 0 && exceptionCount >= retryCountOnError) {
			execute = false;
		}
		return execute;
	}

	/**
	 * デーモンサービスを開始します。<br>
	 * 既にデーモンサービスが処理中である場合には新たなデーモンサービスが開始されません。<br>
	 * また、一時停止指示が行われていた場合、当メソッドを呼び出されたタイミングで指示フラグはクリアされます。<br>
	 * 完全停止指示が行われた場合、又は完全停止指示状況となったデーモンについても開始されません。<br>
	 */
	public final void start() {
		stopSignaled = false;
		if (isProcessing()) {
			log.trace("Server daemon service thread is already started [" + name + "]");
			return;
		}
		if (isHalted()) {
			log.trace("Server daemon service thread is halt signaled [" + name + "]");
			return;
		}
		new DaemonProcessThread().start();
	}

	/**
	 * デーモンサービスに一時停止指示を行います。<br>
	 * 当メソッドを実行したタイミングですぐにデーモンサービスが一時停止するとは限りません。<br>
	 * ここでは、停止フラグが設定され、現在の処理が完了したタイミングで停止します。<br>
	 * また、このメソッドで一時停止されたデーモンは再開することを前提としており、
	 * デーモンレジストリに登録された当オブジェクトは破棄対象とはなりません。<br>
	 */
	public final void stop() {
		if (stopSignaled) {
			log.trace("Server daemon service thread is already stop signaled [" + name + "]");
			return;
		}
		stopSignaled = true;
	}

	/**
	 * デーモンサービスに完全停止指示を行います。<br>
	 * 当メソッドを実行したタイミングですぐにデーモンサービスが完全停止するとは限りません。<br>
	 * ここでは、停止フラグが設定され、現在の処理が完了したタイミングで停止します。<br>
	 * 尚、当メソッドで指示したデーモンはデーモンレジストリに登録された当オブジェクトが
	 * 破棄対象となることを意味し、デーモンの復元は以降行われることはありません。<br>O
	 */
	public final void halt() {
		if (haltSignaled) {
			log.trace("Server daemon service thread is already halt signaled [" + name + "]");
			return;
		}
		haltSignaled = true;
	}

	/**
	 * デーモンサービス処理を実行します。<br>
	 * @throws Throwable 処理中に予期せぬエラーが発生した場合にスローされます
	 */
	public abstract void execute() throws Throwable;

	/**
	 * サービスデーモン実行スレッドクラス<br>
	 * 
	 * @author Kitagawa<br>
	 * 
	 *<!--
	 * 更新日		更新者			更新内容
	 * 2008/12/05	Kitagawa		新規作成
	 *-->
	 */
	private final class DaemonProcessThread extends Thread {

		/**
		 * デーモンサービス処理を実行します。<br>
		 * @see java.lang.Thread#run()
		 */
		public final void run() {
			processing = true;
			while (!stopSignaled && !haltSignaled && isValidity()) {
				if (registry == null || registry.getServerSession() == null) {
					//log.warn("Server daemon service is not registered");
					continue;
				}
				try {
					log.trace("Server daemon service thread process started [" + HttpReverseProxyDaemon.this.name + "]");
					execute();
					Thread.sleep(loopDelay);
				} catch (Throwable e) {
					exeception = e;
					exceptionCount++;
					log.error("Server daemon service process failed [" + HttpReverseProxyDaemon.this.name + "]", e);
				} finally {
					executeCount++;
					log.trace("Server daemon service thread process finished [" + HttpReverseProxyDaemon.this.name + "]");
				}
			}
			processing = false;
		}
	}
}
