package org.phosphoresce.webcore.ext.struts.action;

import java.io.Serializable;
import java.util.LinkedList;
import java.util.List;

import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.phosphoresce.webcore.ext.struts.StrutsConfigNames;
import org.phosphoresce.webcore.ext.struts.exception.StrutsSessionTimeoutException;
import org.slf4j.LoggerFactory;

/**
 * ウィンドウセッションタイムアウトモニタクラス<br>
 * <br>
 * ウィンドウ単位のセッションコンテナのタイムアウト状態を監視するスレッドクラスです。<br>
 * 当スレッドは開始されたタイミングから停止指示若しくは、セッション自体の破棄、監視対象ウィンドウセッションが存在しなくなるまで監視し続けます。<br>
 * 監視処理は不要なリソース圧迫を避けるため、監視対象がなくなったタイミングで終了するため、新たに監視対象が発生した場合は再度開始する必要があります。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/19	Kitagawa		新規作成
 *-->
 */
public final class WindowSessionTimeoutMonitor<L extends Serializable> implements Runnable {

	/** ロガーオブジェクト */
	private CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(getClass()));

	/** セッションコンテナオブジェクト */
	private SessionContainer<L> session;

	/** 実行中フラグ */
	private boolean processing;

	/** 停止指示 */
	private boolean haltIndicate;

	/**
	 * コンストラクタ<br>
	 * @param session セッションコンテナオブジェクト
	 */
	WindowSessionTimeoutMonitor(SessionContainer<L> session) {
		super();

		if (session == null) {
			throw new IllegalArgumentException("session is required");
		}

		this.session = session;
		this.processing = false;
		this.haltIndicate = false;
	}

	/**
	 * モニタに設定されているセッション情報が監視対象として有効であるか判定します。<br>
	 * @return 監視対象として有効なセッション情報が保持されている場合にtrueを返却
	 */
	public boolean isAliveSession() {
		return session != null && !session.isDisposed() && session.getWindowSessionCount() > 0;
	}

	/**
	 * チェックアウトモニタ処理が実行中であるか判定します。<br>
	 * @return チェックアウトモニタ処理が実行中である場合にtrueを返却
	 */
	public boolean isProcessing() {
		return processing;
	}

	/**
	 * チェックアウトモニタ処理をの停止指示を行います。<br>
	 * このメソッドを呼び出したタイミング直後にモニタ停止は行われず、次回チェック処理のタイミングで処理が停止することに注意してください。
	 */
	void halt() {
		haltIndicate = true;
	}

	/**
	 * ウィンドウセッションのタイムアウトを監視します。<br>
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
		try {
			/*
			 * セッションが既に破棄されている状態の場合はモニタスレッドは開始しない
			 */
			if (!isAliveSession()) {
				log.output("FSTR02001");
				return;
			}

			/*
			 * 監視対象セッションが存在し、停止指示が行われていない間はチェック処理をループします
			 */
			if (session != null) {
				log.output("FSTR02002", session.getSessionId());
			}
			while (isAliveSession() && !haltIndicate) {
				// 処理中フラグ設定
				processing = true;

				synchronized (session) {
					// 破棄対象ウィンドウセッション取得
					log.output("FSTR02004", session.getSessionId());
					List<String> destroyTargets = new LinkedList<String>();
					for (String windowId : session.getWindwosSessionIds()) {
						long lastAccessTime = session.getWindowSession(windowId).getLastAccessTime();
						long elapsedTime = (System.currentTimeMillis() - lastAccessTime);
						long timeoutGrace = Config.getLong(StrutsConfigNames.STRUTS_SESSION_WINDOW_TIMEOUT) - elapsedTime;
						if (timeoutGrace < 0) {
							destroyTargets.add(windowId);
						} else {
							log.output("FSTR02006", session.getSessionId(), windowId, timeoutGrace / 1000);
						}
					}

					// 破棄対象セッション破棄処理
					for (String windowId : destroyTargets) {
						session.removeWindowSession(windowId);
						log.output("FSTR02007", session.getSessionId(), windowId);
					}
					log.output("FSTR02005", session.getSessionId(), session.getWindowSessionCount());
				}

				/*
				 * チェック処理インターバル
				 */
				try {
					Thread.sleep(Config.getLong(StrutsConfigNames.STRTUS_SESSION_WINDOW_CHECK_INTERVAL));
				} catch (InterruptedException e) {
					// インターバル処理時に例外が発生した場合は不正なループによるリソース圧迫を避けるため停止指示を行う(例外スローはしない)
					log.output("FSTR02008", e);
					haltIndicate = true;
				}
			}

			if (session != null) {
				log.output("FSTR02003", session.getSessionId());
			}
		} catch (StrutsSessionTimeoutException e) {
			// 監視中にタイムアウト例外が発生した場合は監視中断を行う(例外扱いとはしない)
			log.output("FSTR02009");
		} finally {
			processing = false;
			haltIndicate = false;
			session = null;
		}
	}

	/**
	 * クラス情報を文字列で取得します。<br>
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("{sessionId=");
		builder.append(session == null ? "null" : session.getSessionId());
		builder.append(", processing=");
		builder.append(processing);
		builder.append(", haltIndicate=");
		builder.append(haltIndicate);
		builder.append("}");
		return builder.toString();
	}
}
