package org.phosphoresce.webcore.ext.struts.action;

import java.io.Serializable;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.phosphoresce.lib.commons.util.DateUtil;
import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.phosphoresce.webcore.ext.struts.StrutsConfigNames;
import org.phosphoresce.webcore.ext.struts.StrutsConstants;
import org.phosphoresce.webcore.ext.struts.exception.StrutsProcessInternalException;
import org.phosphoresce.webcore.ext.struts.exception.StrutsSessionTimeoutException;
import org.slf4j.LoggerFactory;

/**
 * ウィンドウセッションコンテナクラス<br>
 * <br>
 * ウィンドウ毎にユニークなセッションとして情報を管理する為のコンテナクラスです。<br>
 * 当クラスはセッションオブジェクト内にウィンドウ毎にユニークキーで保持されます。<br>
 * 尚、ウィンドウセッションコンテナクラスのライフサイクルは外部定義されたタイムアウト時刻に依存し、通常のセッションとは別に監視スレッドによってタイムアウトします。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/19	Kitagawa		新規作成
 *-->
 */
public final class WindowSessionContainer<L extends Serializable> implements Serializable, StrutsConstants {

	/** ロガーオブジェクト */
	private static transient CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(WindowSessionContainer.class));

	/** 破棄済みフラグ */
	private boolean disposed;

	/** セッションコンテナオブジェクト */
	private SessionContainer<L> session;

	/** ウィンドウID */
	private String windowId;

	/** トークン値 */
	private String tokenId;

	/** アクションフォームキャッシュ */
	private Map<String, AbstractForm<L>> forms;

	/** 画面セッションデータ */
	private Map<String, ScreenSession> screenData;

	/** 最終アクセス時刻 */
	private long lastAccessTime;

	/**
	 * コンストラクタ<br>
	 * @param session セッションコンテナオブジェクト
	 * @param windowId ウィンドウID
	 */
	private WindowSessionContainer(SessionContainer<L> session, String windowId) {
		super();

		if (session == null) {
			throw new IllegalArgumentException("session is required");
		}
		if (StringUtil.isEmpty(session.getSessionId())) {
			throw new IllegalArgumentException("sessionId is required");
		}
		if (StringUtil.isEmpty(windowId)) {
			throw new IllegalArgumentException("windowId is required");
		}

		this.disposed = false;
		this.session = session;
		this.windowId = windowId;
		this.tokenId = null;
		this.forms = new HashMap<String, AbstractForm<L>>();
		this.screenData = new HashMap<String, ScreenSession>();
		this.lastAccessTime = System.currentTimeMillis();
	}

	/**
	 * ロガーオブジェクトを取得します。<br>
	 * transientフィールドである為、nullとなった場合にインスタンスを生成して提供します。<br>
	 * @return ローがオブジェクト
	 */
	private static CodeConvertLogger getLog() {
		if (log == null) {
			log = new CodeConvertLogger(LoggerFactory.getLogger(WindowSessionContainer.class));
		}
		return log;
	}

	/**
	 * 新規にウィンドウセッションコンテナをセッションコンテナ上に生成します。<br>
	 * @param httpSession HTTPセッションオブジェクト
	 */
	static <L extends Serializable> WindowSessionContainer<L> createNewWindowSessionContainer(SessionContainer<L> session, String windowId) {
		WindowSessionContainer<L> windowSessionContainer = new WindowSessionContainer<L>(session, windowId);
		session.setWindowSession(windowSessionContainer);
		getLog().output("FSTR02010", session.getSessionId(), windowId);
		return windowSessionContainer;
	}

	/**
	 * 画面セッションデータオブジェクトを生成します。<br>
	 * @param screenDataClass 画面セッションデータクラス
	 * @return 画面セッションデータオブジェクト
	 */
	private <S extends ScreenSession> S createNewScreenData(Class<S> screenDataClass) {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (screenDataClass == null) {
			return null;
		}
		try {
			return screenDataClass.newInstance();
		} catch (Throwable e) {
			throw new StrutsProcessInternalException("FSTR01001", new Object[] { screenDataClass.getSimpleName() });
		}
	}

	/**
	 * ウィンドウセッション内容をすべて破棄します。<br>
	 * ウィンドウセッション破棄後は各種操作呼び出し時にタイムアウト例外が発生することに注意して下さい。<br>
	 */
	void dispose() {
		synchronized (this) {
			// 既に破棄済みである場合は処理中断
			if (disposed) {
				return;
			}

			// ロギング用情報取得
			String activeSessionId = session.getSessionId();
			String activeWindowId = windowId;

			// 各種フィールド初期化
			session = null;
			windowId = null;
			tokenId = null;
			forms = null;
			screenData = null;

			// 破棄済みフラグ設定
			disposed = true;

			getLog().output("FSTR01002", activeSessionId, activeWindowId);
		}
	}

	/**
	 * セッションが破棄済みであるか判定します。<br>
	 * @return セッションが破棄済みである場合にtrueを返却
	 */
	public boolean isDisposed() {
		return disposed || (session != null && session.isDisposed());
	}

	/**
	 * ログインユーザー情報を取得します。<br>
	 * @return ログインユーザー情報
	 */
	public LoginUser<L> getLoginUser() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		return session.getLoginUser();
	}

	/**
	 * ログインユーザー情報を設定します。<br>
	 * @param loginUser ログインユーザー情報
	 */
	public void setLoginUser(LoginUser<L> loginUser) {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		this.session.setLoginUser(loginUser);
	}

	/**
	 * ログイン済みであるか判定します。<br>
	 * @return ログイン済みの場合にtrueを返却
	 */
	public boolean isLogined() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		return session.isLogined();
	}

	/**
	 * セッションIDを取得します。<br>
	 * @return セッションID
	 */
	public String getSessionId() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		return session.getSessionId();
	}

	/**
	 * ウィンドウIDを取得します。<br>
	 * @return ウィンドウID
	 */
	public String getWindowId() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		return windowId;
	}

	/**
	 * トークン値を取得します。<br>
	 * @return トークン値
	 */
	public String getTokenId() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		return tokenId;
	}

	/**
	 * トークン値を設定します。<br>
	 * @param tokenId トークン値
	 */
	void setTokenId(String tokenId) {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		this.tokenId = tokenId;
	}

	/**
	 * 最終アクセス時刻を取得します。<br>
	 * @return 最終アクセス時刻
	 */
	public long getLastAccessTime() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		return lastAccessTime;
	}

	/**
	 * タイムアウト済みセッションであるか判定します。<br>
	 * @return タイムアウト済みセッションである場合にtrueを返却
	 */
	public boolean isTimeouted() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		return System.currentTimeMillis() - lastAccessTime > Config.getLong(StrutsConfigNames.STRUTS_SESSION_WINDOW_TIMEOUT);
	}

	/**
	 * 最終アクセス時刻を更新します。<br>
	 */
	void updateLastAccessTime() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		this.lastAccessTime = System.currentTimeMillis();
		getLog().output("FSTR01007", session.getSessionId(), windowId, DateUtil.format("yyyy/MM/dd HH:mm:ss.SSS", new Date(lastAccessTime)));
	}

	/**
	 * アクションフォームセッションキャッシュを設定します。<br>
	 * @param form アクションフォームオブジェクト
	 */
	<F extends AbstractForm<L>> void setActionForm(F form) {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		if (form == null) {
			return;
		}
		synchronized (forms) {
			if (forms == null) {
				return;
			}
			forms.put(form.getClass().getName(), form);
			getLog().output("FSTR01003", session.getSessionId(), windowId, form.getClass().getSimpleName());
		}
	}

	/**
	 * アクションフォームセッションキャッシュを取得します。<br>
	 * @param clazz アクションフォームクラス
	 * @return キャッシュされたアクションフォームオブジェクト
	 */
	<F extends AbstractForm<L>> F getActionForm(Class<F> clazz) {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		if (clazz == null) {
			return null;
		}
		synchronized (forms) {
			if (forms == null) {
				return null;
			}
			@SuppressWarnings("unchecked")
			F f = (F) forms.get(clazz.getName());
			return f;
		}
	}

	/**
	 * 画面セッション情報をMapオブジェクトとして全て取得します。<br>
	 * @return 画面セッション情報
	 */
	Map<String, ScreenSession> getScreenData() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		return screenData;
	}

	/**
	 * 画面セッションデータを取得します。<br>
	 * @param screenDataClass 画面セッションデータクラス
	 * @return 画面セッションオブジェクト
	 */
	public <S extends ScreenSession> S getScreenData(Class<S> screenDataClass) {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		if (screenDataClass == null) {
			return null;
		}
		synchronized (screenData) {
			if (screenData == null) {
				return null;
			}
			@SuppressWarnings("unchecked")
			S s = (S) screenData.get(screenDataClass.getSimpleName());
			if (s == null) {
				s = createNewScreenData(screenDataClass);
				screenData.put(s.getClass().getSimpleName(), s);
				getLog().output("FSTR00018", session.getSessionId(), windowId, s.getClass().getSimpleName());
			}
			return s;
		}
	}

	/**
	 * 画面セッションデータを削除します。<br>
	 * @param screenDataClass 画面セッションデータクラス
	 */
	<S extends ScreenSession> void removeScreenData(Class<S> screenDataClass) {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		if (screenDataClass == null) {
			return;
		}
		synchronized (screenData) {
			if (screenData == null) {
				return;
			}
			screenData.remove(screenDataClass.getSimpleName());
			getLog().output("FSTR01005", session.getSessionId(), windowId, screenDataClass.getSimpleName());
		}
	}

	/**
	 * 画面セッションデータをすべて破棄します。<br>
	 */
	void clearScreenData() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		synchronized (screenData) {
			if (screenData == null) {
				return;
			}
			screenData.clear();
			getLog().output("FSTR01006", session.getSessionId(), windowId);
		}
	}

	/**
	 * クラス情報を文字列で取得します。<br>
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("{lastAccessTime=");
		builder.append(lastAccessTime);
		builder.append(", sessionId=");
		builder.append(session.getSessionId());
		builder.append(", windowId=");
		builder.append(windowId);
		builder.append(", tokenId=");
		builder.append(tokenId);
		builder.append(", forms=");
		builder.append(forms);
		builder.append(", screenData=");
		builder.append(screenData);
		builder.append("}");
		return builder.toString();
	}
}
