package org.phosphoresce.webcore.ext.struts.action;

import java.io.BufferedWriter;
import java.io.OutputStreamWriter;
import java.io.Serializable;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletContext;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import net.arnx.jsonic.JSON;

import org.apache.struts.action.ActionMessage;
import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.ConfigName;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.phosphoresce.webcore.core.transaction.TransactionContainer;
import org.phosphoresce.webcore.ext.struts.StrutsConstants;
import org.phosphoresce.webcore.ext.struts.exception.StrutsProcessInternalException;
import org.phosphoresce.webcore.ext.struts.util.StrutsUtil;
import org.slf4j.LoggerFactory;

/**
 * サーブレットコンテナクラス<br>
 * <br>
 * リクエスト処理中に扱う各種情報を統括的に管理するクラスで、ライフサイクルはリクエスト単位となります。<br>
 * サーブレット実行時の管理情報の増減によってアクションクラスインタフェース変更を発生させないために設けられました。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/19	Kitagawa		新規作成
 * 2012/07/05	Kitagawa		全体的に再構築
 *-->
 */
public class ServletContainer<L extends Serializable> implements StrutsConstants {

	/** ロガーオブジェクト */
	private CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(getClass()));

	/** 処理実行日時 */
	private Date processDate;

	/** サーブレットコンテキストオブジェクト */
	private ServletContext context;

	/** リクエストオブジェクト */
	private HttpServletRequest request;

	/** レスポンスオブジェクト */
	private HttpServletResponse response;

	/** セッションコンテナオブジェクト */
	private SessionContainer<L> session;

	/** メッセージコンテナオブジェクト */
	private MessageContainer message;

	/** トランザクションコンテナオブジェクト */
	private TransactionContainer transaction;

	/** ストリームコンテナ */
	private StreamContainer streamContainer;

	/** 利用ロジックインスタンス */
	private List<AbstractLogic<L>> logics;

	/** リソースダウンロードID */
	private String downloadId;

	/** リソースダウンロード名 */
	private String downloadName;

	/**
	 * コンストラクタ<br>
	 * @param context サーブレットコンテキストオブジェクト
	 * @param request リクエストオブジェクト
	 * @param response レスポンスオブジェクト
	 * @param session セッションコンテナオブジェクト
	 * @param message メッセージコンテナオブジェクト
	 * @param transaction トランザクションコンテナオブジェクト
	 */
	ServletContainer(ServletContext context, HttpServletRequest request, HttpServletResponse response, SessionContainer<L> session, MessageContainer message, TransactionContainer transaction) {
		super();

		if (context == null) {
			throw new IllegalArgumentException("context is required");
		}
		if (request == null) {
			throw new IllegalArgumentException("request is required");
		}
		if (response == null) {
			throw new IllegalArgumentException("response is required");
		}
		if (session == null) {
			throw new IllegalArgumentException("session is required");
		}
		if (message == null) {
			throw new IllegalArgumentException("message is required");
		}
		if (transaction == null) {
			throw new IllegalArgumentException("transaction is required");
		}

		this.processDate = new Date();
		this.context = context;
		this.request = request;
		this.response = response;
		this.session = session;
		this.message = message;
		this.transaction = new TransactionContainer();
		this.logics = new LinkedList<AbstractLogic<L>>();
		this.streamContainer = new StreamContainer(request, response);
		this.downloadId = (String) request.getAttribute(REQUEST_ATTR_KEY_DOWNLOAD_ID);
		this.downloadName = (String) request.getAttribute(REQUEST_ATTR_KEY_DOWNLOAD_FILENAME);
	}

	/**
	 * サーブレットで利用したストリームオブジェクトを全てクローズします。<br>
	 */
	void closeStreams() {
		try {
			if (streamContainer != null) {
				streamContainer.close();
			}
			log.output("FSTR04001");
		} catch (Throwable e) {
			throw new StrutsProcessInternalException("FSTR04002", e);
		}
	}

	/**
	 * セッションを破棄します。<br>
	 */
	public void disposeSession() {
		session.dispose();
		HttpSession httpSession = request.getSession(true);
		httpSession.removeAttribute(SESSION_KEY_CONTAINER);
		httpSession.invalidate();
		log.output("FSTR04003");
	}

	/**
	 * 指定されたビーンオブジェクトをJSON形式でレスポンスします。<br>
	 * メッセージが設定されている場合は併せてレスポンスします。<br>
	 * @param bean ビーンオブジェクト
	 */
	public void writeJSON(Object bean) {
		try {
			Map<String, Object> map = new HashMap<String, Object>();

			/*
			 * アプリケーションメッセージ設定
			 */
			// エラーメッセージ
			List<String> errorMsgs = new LinkedList<String>();
			for (@SuppressWarnings("unchecked")
			Iterator<ActionMessage> iterator = message.getErrorActionMessages().get(); iterator.hasNext();) {
				ActionMessage message = iterator.next();
				errorMsgs.add(message.getValues()[0].toString());
			}
			map.put(JSON_KEY_ERROR_MESSAGES, errorMsgs);
			// 情報メッセージ
			List<String> informationMsgs = new LinkedList<String>();
			for (@SuppressWarnings("unchecked")
			Iterator<ActionMessage> iterator = message.getInformationActionMessages().get(); iterator.hasNext();) {
				ActionMessage message = iterator.next();
				informationMsgs.add(message.getValues()[0].toString());
			}
			map.put(JSON_KEY_INFO_MESSAGES, informationMsgs);
			// 警告メッセージ
			List<String> warningMsgs = new LinkedList<String>();
			for (@SuppressWarnings("unchecked")
			Iterator<ActionMessage> iterator = message.getWarningActionMessages().get(); iterator.hasNext();) {
				ActionMessage message = iterator.next();
				warningMsgs.add(message.getValues()[0].toString());
			}
			map.put(JSON_KEY_WARN_MESSAGES, warningMsgs);
			// 連絡メッセージ
			List<String> noticeMsgs = new LinkedList<String>();
			for (@SuppressWarnings("unchecked")
			Iterator<ActionMessage> iterator = message.getNoticeActionMessages().get(); iterator.hasNext();) {
				ActionMessage message = iterator.next();
				noticeMsgs.add(message.getValues()[0].toString());
			}
			map.put(JSON_KEY_NOTICE_MESSAGES, noticeMsgs);
			// デバッグメッセージ
			List<String> debugMsgs = new LinkedList<String>();
			for (@SuppressWarnings("unchecked")
			Iterator<ActionMessage> iterator = message.getDebugActionMessages().get(); iterator.hasNext();) {
				ActionMessage message = iterator.next();
				debugMsgs.add(message.getValues()[0].toString());
			}
			map.put(JSON_KEY_DEBUG_MESSAGES, debugMsgs);
			// トレースメッセージ
			List<String> traceMsgs = new LinkedList<String>();
			for (@SuppressWarnings("unchecked")
			Iterator<ActionMessage> iterator = message.getTraceActionMessages().get(); iterator.hasNext();) {
				ActionMessage message = iterator.next();
				traceMsgs.add(message.getValues()[0].toString());
			}
			map.put(JSON_KEY_TRACE_MESSAGES, traceMsgs);

			/*
			 * 指定ビーンオブジェクト設定
			 */
			if (bean == null) {
				map.put(JSON_KEY_RESPONSE_DATA, new HashMap<String, Object>());
			} else {
				map.put(JSON_KEY_RESPONSE_DATA, bean);
			}

			/*
			 * レスポンス処理
			 */
			//String data = JSON.encode(map, true);
			String data = JSON.encode(map);
			int contentLength = data.getBytes(Config.getString(ConfigName.COMMON_CHARSET_DEFAULT)).length;
			BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(streamContainer.getJSONStream(contentLength), Config.getString(ConfigName.COMMON_CHARSET_DEFAULT)));
			writer.write(data);
			writer.flush();

			log.output("FSTR04004", data);
		} catch (Throwable e) {
			throw new StrutsProcessInternalException("FSTR04005", e);
		}
	}

	/**
	 * 処理実行日時を取得します。<br>
	 * @return 処理実行日時
	 */
	public Date getProcessDate() {
		return processDate;
	}

	/**
	 * サーブレットが実行されたウィンドウIDを取得します。<br>
	 * @return ウィンドウID
	 */
	public String getSessionId() {
		return session.getSessionId();
	}

	/**
	 * サーブレットが実行されたウィンドウIDを取得します。<br>
	 * @return ウィンドウID
	 */
	public String getWindowId() {
		return StrutsUtil.getRequestWindowId(request, null);
	}

	/**
	 * ストリームコンテナを取得します。<br>
	 * @return ストリームコンテナ
	 */
	public StreamContainer getStreamContainer() {
		return streamContainer;
	}

	/**
	 * リクエストパラメータを取得します。<br>
	 * @param key リクエストパラメータキー
	 * @return リクエストパラメータ
	 */
	public String getParameter(String key) {
		return StringUtil.nvl(request.getParameter(key));
	}

	/**
	 * リクエストパラメータを取得します。<br>
	 * @param key リクエストパラメータキー
	 * @return リクエストパラメータ
	 */
	public String[] getParameters(String key) {
		return request.getParameterValues(key);
	}

	/**
	 * クッキー情報を設定します。<br>
	 * @param name クッキー保持名
	 * @param value クッキーオブジェクト
	 * @param expiry 有効日数
	 */
	public void setCookie(String name, String value, int expiry) {
		if (name == null) {
			return;
		}
		String path = request.getContextPath() + "/";
		Cookie cookie = new Cookie(name, StringUtil.encodeURL(value, Config.getString(ConfigName.COMMON_CHARSET_DEFAULT)));
		cookie.setPath(path);
		cookie.setMaxAge(expiry);
		response.addCookie(cookie);
	}

	/**
	 * クッキー情報を設定します。<br>
	 * @param name クッキー保持名
	 * @param value クッキーオブジェクト
	 */
	public void setCookie(String name, String value) {
		setCookie(name, value, Config.getInteger(ConfigName.COMMON_SERVLET_COOKIE_TIMEOUT));
	}

	/**
	 * クッキーに保持されている値を取得します。<br>
	 * @param name クッキー保持名
	 * @return 保持値
	 */
	public String getCookie(String name) {
		Cookie cookie = getCookieObject(name);
		if (cookie == null) {
			return null;
		}
		return StringUtil.decodeURL(cookie.getValue(), Config.getString(ConfigName.COMMON_CHARSET_DEFAULT));
	}

	/**
	 * 指定された名前のクッキーを削除します。<br>
	 * @param name クッキー保持名
	 */
	public void removeCookie(String name) {
		if (name == null) {
			return;
		}
		setCookie(name, EMPTY_STRING, 0);
	}

	/**
	 * 指定された名前のクッキーを取得します。<br>
	 * @param name クッキー保持名
	 * @return クッキーオブジェクト
	 */
	private Cookie getCookieObject(String name) {
		if (name == null) {
			return null;
		}
		if (request != null && request.getCookies() != null) {
			for (Cookie cookie : request.getCookies()) {
				if (name.equals(cookie.getName())) {
					return cookie;
				}
			}
		}
		return null;
	}

	/**
	 * サーブレットコンテキストオブジェクトを取得します。<br>
	 * @return サーブレットコンテキストオブジェクト
	 */
	ServletContext getContext() {
		return context;
	}

	/**
	 * サーブレットコンテキストオブジェクトを設定します。<br>
	 * @param context サーブレットコンテキストオブジェクト
	 */
	void setContext(ServletContext context) {
		this.context = context;
	}

	/**
	 * リクエストオブジェクトを取得します。<br>
	 * @return リクエストオブジェクト
	 */
	HttpServletRequest getRequest() {
		return request;
	}

	/**
	 * リクエストオブジェクトを設定します。<br>
	 * @param request リクエストオブジェクト
	 */
	void setRequest(HttpServletRequest request) {
		this.request = request;
	}

	/**
	 * レスポンスオブジェクトを取得します。<br>
	 * @return レスポンスオブジェクト
	 */
	HttpServletResponse getResponse() {
		return response;
	}

	/**
	 * レスポンスオブジェクトを設定します。<br>
	 * @param response レスポンスオブジェクト
	 */
	void setResponse(HttpServletResponse response) {
		this.response = response;
	}

	/**
	 * セッションコンテナオブジェクトを取得します。<br>
	 * @return セッションコンテナオブジェクト
	 */
	SessionContainer<L> getSession() {
		return session;
	}

	/**
	 * セッションコンテナオブジェクトを設定します。<br>
	 * @param session セッションコンテナオブジェクト
	 */
	void setSession(SessionContainer<L> session) {
		this.session = session;
	}

	/**
	 * メッセージコンテナオブジェクトを取得します。<br>
	 * @return メッセージコンテナオブジェクト
	 */
	MessageContainer getMessage() {
		return message;
	}

	/**
	 * メッセージコンテナオブジェクトを設定します。<br>
	 * @param message メッセージコンテナオブジェクト
	 */
	void setMessage(MessageContainer message) {
		this.message = message;
	}

	/**
	 * トランザクションコンテナオブジェクトを取得します。<br>
	 * @return トランザクションコンテナオブジェクト
	 */
	TransactionContainer getTransaction() {
		return transaction;
	}

	/**
	 * トランザクションコンテナオブジェクトを設定します。<br>
	 * @param transaction トランザクションコンテナオブジェクト
	 */
	void setTransaction(TransactionContainer transaction) {
		this.transaction = transaction;
	}

	/**
	 * 利用ロジックインスタンスを追加します。<br>
	 * @param logic ロジックインスタンス
	 */
	void addLogic(AbstractLogic<L> logic) {
		this.logics.add(logic);
	}

	/**
	 * 利用ロジックインスタンスを取得します。<br>
	 * @return 利用ロジックインスタンス
	 */
	List<AbstractLogic<L>> getLogics() {
		return logics;
	}

	/**
	 * リソースダウンロードIDを取得します。<br>
	 * @return リソースダウンロードID
	 */
	String getDownloadId() {
		return downloadId;
	}

	/**
	 * リソースダウンロードIDを設定します。<br>
	 * @param downloadId リソースダウンロードID
	 */
	void setDownloadId(String downloadId) {
		this.downloadId = downloadId;
	}

	/**
	 * リソースダウンロード名を取得します。<br>
	 * @return リソースダウンロード名
	 */
	String getDownloadName() {
		return downloadName;
	}

	/**
	 * リソースダウンロード名を設定します。<br>
	 * @param downloadName リソースダウンロード名
	 */
	void setDownloadName(String downloadName) {
		this.downloadName = downloadName;
	}
}
