package org.phosphoresce.webcore.ext.jod;

import java.util.HashMap;
import java.util.Map;

import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.lib.jod.JODOSType;
import org.phosphoresce.lib.jod.JODSOfficeProcess;
import org.phosphoresce.lib.jod.JODSOfficeProcessManager;
import org.phosphoresce.lib.jod.JODSetting;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.exception.WebcoreConfigurationException;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.slf4j.LoggerFactory;

/**
 * LibreOfficeドキュメントコンバートアドオン環境初期化クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2011/12/24	Kitagawa		新規作成
 *-->
 */
public class JODConfiguration implements JODConstants {

	/** ロガーオブジェクト */
	private static CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(JODConfiguration.class));

	/**
	 * コンストラクタ<br>
	 */
	private JODConfiguration() {
		super();
	}

	/**
	 * LibreOfficeドキュメントコンバートアドオンの初期化処理を行います。<br>
	 * 当初期化処理はアプリケーション共通初期化処理が完了されていることが前提となります。<br>
	 */
	public synchronized static void configure() {
		try {
			log.output("FJOD00001");

			/*
			 * 各種ドキュメント変換エンジン環境定義取得
			 */
			final String PLATFORM_OS = Config.getString(JODConfigName.JOD_SOFFICE_OS);
			final int CONNECT_TRY_COUNT = Config.getInteger(JODConfigName.JOD_SOFFICE_SOCKET_CONNECT_TEST_COUNT);
			final long CONNECT_TRY_DELAY = Config.getInteger(JODConfigName.JOD_SOFFICE_SOCKET_CONNECT_TEST_DELAY);
			final boolean CONNECT_KEEP = Config.getBoolean(JODConfigName.JOD_SOFFICE_SOCKET_PROCESS_KEEP);
			final String SOFFICE_PATH = Config.getString(JODConfigName.JOD_SOFFICE_PATH);
			final String TEMPORARY_PATH = Config.getString(JODConfigName.JOD_TEMPORARY_PATH);
			final boolean PROCESS_KILL_ON_STARTUP = Config.getBoolean(JODConfigName.JOD_SOFFICE_SOCKET_PROCESS_KILL_ON_STARTUP);
			final String PROCESS_KILL_COMMAND = Config.getString(JODConfigName.JOD_SOFFICE_KILL_COMMAND);
			final String SOCKET_HOST = Config.getString(JODConfigName.JOD_SOFFICE_SOCKET_HOST);
			final int SOCKET_PORT_START = Config.getInteger(JODConfigName.JOD_SOFFICE_SOCKET_PORT_START);
			final int SOCKET_PORT_END = Config.getInteger(JODConfigName.JOD_SOFFICE_SOCKET_PORT_END);

			/*
			 * ドキュメント変換エンジン環境設定初期化
			 */
			JODSetting setting = new JODSetting();
			setting.setOsType(JODOSType.valueOf(PLATFORM_OS.toUpperCase()));
			setting.setKeepConnect(CONNECT_KEEP);
			setting.setConnectTryCount(CONNECT_TRY_COUNT);
			setting.setConnectTryDelay(CONNECT_TRY_DELAY);
			setting.setSofficePath(SOFFICE_PATH);
			setting.setTemporaryPath(TEMPORARY_PATH);
			setting.setSofficeHost(SOCKET_HOST);
			setting.setSofficeReservePortStart(SOCKET_PORT_START);
			setting.setSofficeReservePortEnd(SOCKET_PORT_END);
			JODSOfficeProcessManager.initialize(setting);

			/*
			 * オペレーションシステム上のソケットサービスプロセスを強制終了
			 */
			if (PROCESS_KILL_ON_STARTUP && !StringUtil.isEmpty(PROCESS_KILL_COMMAND)) {
				log.output("FJOD00004");
				try {
					Runtime.getRuntime().exec(PROCESS_KILL_COMMAND);
				} catch (Throwable e) {
					log.output(e, "FJOD00011");
				}
			}

			log.output("FJOD00005", SOCKET_HOST, String.valueOf(SOCKET_PORT_START), String.valueOf(SOCKET_PORT_END));
			if (CONNECT_KEEP) {
				log.output("FJOD00006");
			} else {
				log.output("FJOD00007");
			}

			final Map<Integer, Boolean> validateMap = new HashMap<Integer, Boolean>();
			for (int i = SOCKET_PORT_START; i <= SOCKET_PORT_END; i++) {
				final int port = i;
				/*
				 * 有効可能ポートが大量に有る場合の接続試行が長時間を要する為、スレッドで接続試行を行う
				 */
				new Thread(new Runnable() {
					@Override
					public void run() {
						try {
							JODSOfficeProcess process = JODSOfficeProcessManager.instance().createProcess(port);
							process.open();
							if (CONNECT_KEEP) {
								process.close();
							} else {
								process.destroy();
							}
							log.output("FJOD00008", SOCKET_HOST, String.valueOf(port));
							validateMap.put(validateMap.size(), true);
						} catch (Throwable e) {
							log.output(e, "FJOD00009", SOCKET_HOST, String.valueOf(port));
							validateMap.put(validateMap.size(), false);
						}
					}
				}).start();
			}
			while (validateMap.size() < ((SOCKET_PORT_END - SOCKET_PORT_START) + 1)) {
				Thread.sleep(100);
			}

			log.output("FJOD00002");
		} catch (Throwable e) {
			throw new WebcoreConfigurationException("FJOD00010", e);
		}
	}

	/**
	 * LibreOfficeドキュメントコンバートアドオンの破棄処理を実行します。<br>
	 * @throws Throwable 終了処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void destroy() throws Throwable {
		JODSOfficeProcessManager.destroy();
		log.output("FJOD00003");
	}
}
