package org.phosphoresce.webcore.ext.jdbc.transaction;

import java.sql.Connection;

import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.transaction.TransactionManager;
import org.phosphoresce.webcore.ext.jdbc.JDBCConfigName;
import org.phosphoresce.webcore.ext.jdbc.JDBCConstants;
import org.phosphoresce.webcore.ext.jdbc.JDBCEnvironment;
import org.phosphoresce.webcore.ext.jdbc.exception.JDBCTransactionException;
import org.phosphoresce.webcore.ext.jdbc.factory.JDBCConnectionFactory;
import org.phosphoresce.webcore.ext.jdbc.factory.JDBCDatasourceConnectionFactory;
import org.phosphoresce.webcore.ext.jdbc.factory.JDBCDriverConnectionFactory;

/**
 * JDBCトランザクション管理クラス<br>
 * <br>
 * JDBC経由のデータベースへの接続確立、切断、トランザクションのコミット、ロールバック制御を提供するクラスです。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2012/07/05	Kitagawa		新規作成
 *-->
 */
public class JDBCTransactionManager extends TransactionManager implements JDBCConstants {

	/** データベースコネクション */
	private Connection connection;

	/** コミット待ち状態フラグ */
	private boolean waitCommit;

	/**
	 * コンストラクタ<br>
	 * @param getNamespace()  接続定義名前空間
	 */
	public JDBCTransactionManager(String namespace) {
		super(namespace);
	}

	/**
	 * コンストラクタ<br>
	 */
	public JDBCTransactionManager() {
		super();
	}

	/**
	 * データベースコネクションを取得します。<br>
	 * @return データベースコネクション
	 */
	public Connection getConnection() {
		return connection;
	}

	/**
	 * コミット待ち状態フラグを設定します。<br>
	 * @param waitCommit コミット待ち状態フラグ
	 */
	public void setWaitCommit(boolean waitCommit) {
		this.waitCommit = waitCommit;
	}

	/**
	 * トランザクション開始処理を実行します。<br>
	 * @see org.phosphoresce.webcore.core.transaction.TransactionManager#openProcess()
	 */
	@Override
	protected void openProcess() {
		if (!JDBCEnvironment.isEnabledDatabase()) {
			log.output("FJDB00021");
			return;
		}
		if (isOpened()) {
			throw new JDBCTransactionException("FJDB00011");
		}
		try {
			String suffix = "";
			if (!StringUtil.isEmpty(getNamespace())) {
				suffix += "." + getNamespace();
			}

			boolean useDatasource = false;
			JDBCConnectionFactory factory = null;
			if (CONNECTION_TYPE_JDBC.equals(Config.getString(JDBCConfigName.JDBC_CONNECTION_TYPE.suffixedName(suffix)))) {
				factory = new JDBCDriverConnectionFactory( //
						Config.getString(JDBCConfigName.JDBC_DRIVER.suffixedName(suffix)), //
						Config.getString(JDBCConfigName.JDBC_URL.suffixedName(suffix)), //
						Config.getString(JDBCConfigName.JDBC_USER.suffixedName(suffix)), //
						Config.getString(JDBCConfigName.JDBC_PASSWORD.suffixedName(suffix)), //
						Config.getBoolean(JDBCConfigName.JDBC_AUTOCOMMIT.suffixedName(suffix)) //
				);
			} else if (CONNECTION_TYPE_DATASOURCE.equals(Config.getString(JDBCConfigName.JDBC_CONNECTION_TYPE.suffixedName(suffix)))) {
				factory = new JDBCDatasourceConnectionFactory( //
						Config.getString(JDBCConfigName.JDCB_DATASOURCE_NAME.suffixedName(suffix)), //
						Config.getBoolean(JDBCConfigName.JDBC_AUTOCOMMIT.suffixedName(suffix)) //
				);
				useDatasource = true;
			}
			connection = factory.createConnection();

			// トレースログ
			if (useDatasource) {
				if (StringUtil.isEmpty(getNamespace())) {
					log.output("FJDB00002", getNamespace());
				} else {
					log.output("FJDB00006", getNamespace());
				}
			} else {
				if (StringUtil.isEmpty(getNamespace())) {
					log.output("FJDB00003", getNamespace());
				} else {
					log.output("FJDB00007", getNamespace());
				}
			}

			log.output("FJDB00026");
		} catch (Throwable e) {
			throw new JDBCTransactionException("FJDB00012", e);
		}
	}

	/**
	 * トランザクションを終了処理を実行します。<br>
	 * @see org.phosphoresce.webcore.core.transaction.TransactionManager#closeProcess()
	 */
	@Override
	protected void closeProcess() {
		if (!JDBCEnvironment.isEnabledDatabase()) {
			log.output("FJDB00022");
			return;
		}
		if (!isOpened()) {
			throw new JDBCTransactionException("FJDB00013");
		}
		try {
			if (waitCommit) {
				throw new JDBCTransactionException("FJDB00014");
			}
			connection.rollback();
			connection.close();
			connection = null;
			log.output("FJDB00027");
		} catch (Throwable e) {
			throw new JDBCTransactionException("FJDB00015", e);
		}
	}

	/**
	 * トランザクションをコミット処理を実行します。<br>
	 * @see org.phosphoresce.webcore.core.transaction.TransactionManager#commitProcess()
	 */
	@Override
	protected void commitProcess() {
		if (!JDBCEnvironment.isEnabledDatabase()) {
			log.output("FJDB00023");
			return;
		}
		if (!isOpened()) {
			throw new JDBCTransactionException("FJDB00013");
		}
		try {
			connection.commit();
			waitCommit = false;
			log.output("FJDB00028");
		} catch (Throwable e) {
			throw new JDBCTransactionException("FJDB00016", e);
		}
	}

	/**
	 * トランザクションをロールバック処理を実行します。<br>
	 * @see org.phosphoresce.webcore.core.transaction.TransactionManager#rollbackProcess()
	 */
	@Override
	protected void rollbackProcess() {
		if (!JDBCEnvironment.isEnabledDatabase()) {
			log.output("FJDB00024");
			return;
		}
		if (!isOpened()) {
			throw new JDBCTransactionException("FJDB00013");
		}
		try {
			connection.rollback();
			waitCommit = false;
			log.output("FJDB00029");
		} catch (Throwable e) {
			throw new JDBCTransactionException("FJDB00017", e);
		}
	}
}
