package org.phosphoresce.webcore.ext.jasper.export;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.print.attribute.HashPrintServiceAttributeSet;

import net.sf.jasperreports.engine.JRExporter;
import net.sf.jasperreports.engine.JRExporterParameter;
import net.sf.jasperreports.engine.export.JExcelApiExporter;
import net.sf.jasperreports.engine.export.JRCsvExporter;
import net.sf.jasperreports.engine.export.JRHtmlExporter;
import net.sf.jasperreports.engine.export.JRHtmlExporterParameter;
import net.sf.jasperreports.engine.export.JRPdfExporter;
import net.sf.jasperreports.engine.export.JRPrintServiceExporter;
import net.sf.jasperreports.engine.export.JRPrintServiceExporterParameter;
import net.sf.jasperreports.engine.export.JRRtfExporter;
import net.sf.jasperreports.engine.export.JRXlsExporter;
import net.sf.jasperreports.engine.export.JRXmlExporter;
import net.sf.jasperreports.engine.export.JRXmlExporterParameter;
import net.sf.jasperreports.engine.export.oasis.JROdtExporter;

import org.phosphoresce.webcore.core.Configuration;
import org.phosphoresce.webcore.core.Environment;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.phosphoresce.webcore.ext.jasper.JasperConfigName;
import org.phosphoresce.webcore.ext.jasper.JasperConstants;
import org.slf4j.LoggerFactory;

/**
 * JasperReport出力時のExporterオブジェクトファクトリクラス。<br>
 * 汎用的なJRExporterオブジェクトについてはstaticメソッドより直接生成可能ですが、
 * 明示的にパラメータを指定して生成したい場合は、ファクトリクラスインスタンスに対して
 * パラメータを設定し、それに依存した各種JRExporterインスタンスを生成します。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2007/08/10	Kitagawa		新規作成
 *-->
 */
public final class JasperExporterFactory implements JasperConstants {

	/** ロガーオブジェクト */
	private static CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(Configuration.class));

	/** 出力タイプ */
	private JasperExportType type;

	/** 拡張パラメータ */
	private Map<JRExporterParameter, Object> parameter;

	/** プリンタ名 */
	private String printerName;

	/**
	 * コンストラクタ<br>
	 */
	private JasperExporterFactory() {
		super();
	}

	/**
	 * コンストラクタ<br>
	 * @param type 出力タイプ
	 */
	public JasperExporterFactory(JasperExportType type) {
		super();
		this.type = type;
	}

	/**
	 * コンストラクタ<br>
	 * @param type 出力タイプ
	 * @param parameter パラメータ
	 */
	public JasperExporterFactory(JasperExportType type, Map<JRExporterParameter, Object> parameter) {
		super();
		this.type = type;
		this.parameter = parameter;
	}

	/**
	 * プリンタ名を取得します。<br>
	 * @return プリンタ名
	 */
	public String getPrinterName() {
		return printerName;
	}

	/**
	 * プリンタ名を設定します。<br>
	 * @param printerName プリンタ名
	 */
	public void setPrinterName(String printerName) {
		this.printerName = printerName;
	}

	/**
	 * パラメータを設定します。<br>
	 * @param key パラメータキー
	 * @param value パラメータ
	 */
	public void setParameter(JRExporterParameter key, Object value) {
		if (parameter == null) {
			parameter = new HashMap<JRExporterParameter, Object>();
		}
		parameter.put(key, value);
	}

	/**
	 * JRExporterオブジェクトを生成します。<br>
	 * @return JRExporterオブジェクト
	 */
	public JRExporter create() {
		JRExporter exporter = null;
		if (JasperExportType.PDF.equals(type)) {
			exporter = createDefaultPDFExporter();
		} else if (JasperExportType.XML.equals(type)) {
			exporter = createDefaultXMLExporter();
		} else if (JasperExportType.CSV.equals(type)) {
			exporter = createDefaultCSVExporter();
		} else if (JasperExportType.HTML.equals(type)) {
			exporter = createDefaultHTMLExporter();
		} else if (JasperExportType.RTF.equals(type)) {
			exporter = createDefaultRTFExporter();
		} else if (JasperExportType.ODF.equals(type)) {
			exporter = createDefaultODFExporter();
		} else if (JasperExportType.JXL.equals(type)) {
			exporter = createDefaultXLSJXLExporter();
		} else if (JasperExportType.POI.equals(type)) {
			exporter = createDefaultXLSPOIExporter();
		} else if (JasperExportType.PRINT.equals(type)) {
			if (printerName != null && printerName.trim().length() > 0) {
				exporter = createNamedPrintServiceExporter(printerName);
			} else {
				exporter = createDefaultPrintServiceExporter();
			}
		}
		fillParameter(exporter, parameter);

		log.output("FJPR00011", String.valueOf(type));

		return exporter;
	}

	/**
	 * Exporterに対して指定されたマップが保持するパラメータを設定します。<br>
	 * @param exporter JRExporterオブジェクト
	 * @param parameter パラメータマップ
	 */
	private void fillParameter(JRExporter exporter, Map<JRExporterParameter, Object> parameter) {
		if (exporter != null && parameter != null) {
			for (Iterator<JRExporterParameter> iterator = parameter.keySet().iterator(); iterator.hasNext();) {
				JRExporterParameter key = iterator.next();
				Object value = parameter.get(key);
				exporter.setParameter(key, value);
			}
		}
	}

	/**
	 * ディフォルトのPDF生成用Exporterオブジェクトを生成します。<br>
	 * ディフォルトのExporterとしてのパラメータは以下の通りです。<br>
	 * <pre>
	 * ・印刷を許可
	 * </pre>
	 * @return JRExporterオブジェクト
	 */
	public static JRExporter createDefaultPDFExporter() {
		JRPdfExporter exporter = new JRPdfExporter();
		//exporter.setParameter(JRPdfExporterParameter.IS_ENCRYPTED, Boolean.TRUE);
		//exporter.setParameter(JRPdfExporterParameter.USER_PASSWORD, password);
		return exporter;
	}

	/**
	 * ディフォルトのXML生成用Exporterオブジェクトを生成します。<br>
	 * ディフォルトのExporterとしてのパラメータは以下の通りです。<br>
	 * <pre>
	 * ・イメージを埋め込み
	 * </pre>
	 * @return JRExporterオブジェクト
	 */
	public static JRExporter createDefaultXMLExporter() {
		JRXmlExporter exporter = new JRXmlExporter();
		exporter.setParameter(JRXmlExporterParameter.IS_EMBEDDING_IMAGES, Boolean.TRUE);
		return exporter;
	}

	/**
	 * ディフォルトのCSV生成用Exporterオブジェクトを生成します。<br>
	 * ディフォルトのExporterとしてのパラメータは以下の通りです。<br>
	 * <pre>
	 * </pre>
	 * @return JRExporterオブジェクト
	 */
	public static JRExporter createDefaultCSVExporter() {
		JRCsvExporter exporter = new JRCsvExporter();
		//exporter.setParameter(JRCsvExporterParameter.FIELD_DELIMITER, ",");
		//exporter.setParameter(JRCsvExporterParameter.RECORD_DELIMITER, "\n");
		return exporter;
	}

	/**
	 * ディフォルトのHTML生成用Exporterオブジェクトを生成します。<br>
	 * ディフォルトのExporterとしてのパラメータは以下の通りです。<br>
	 * <pre>
	 * </pre>
	 * @return JRExporterオブジェクト
	 */
	public static JRExporter createDefaultHTMLExporter() {
		JRHtmlExporter exporter = new JRHtmlExporter();
		//exporter.setParameter(JRHtmlExporterParameter.HTML_HEADER, "html");
		exporter.setParameter(JRHtmlExporterParameter.IMAGES_URI, Environment.getContextPath() + Config.getString(JasperConfigName.JASPER_IMAGE_SERVLETPATH));
		return exporter;
	}

	/**
	 * ディフォルトのRTF生成用Exporterオブジェクトを生成します。<br>
	 * ディフォルトのExporterとしてのパラメータは以下の通りです。<br>
	 * <pre>
	 * </pre>
	 * @return JRExporterオブジェクト
	 */
	public static JRExporter createDefaultRTFExporter() {
		JRRtfExporter exporter = new JRRtfExporter();
		//exporter.setParameter(JRExporterParameter.JASPER_PRINT, print);
		return exporter;
	}

	/**
	 * ディフォルトのODF生成用Exporterオブジェクトを生成します。<br>
	 * ディフォルトのExporterとしてのパラメータは以下の通りです。<br>
	 * <pre>
	 * </pre>
	 * @return JRExporterオブジェクト
	 */
	public static JRExporter createDefaultODFExporter() {
		JROdtExporter exporter = new JROdtExporter();
		//exporter.setParameter(JRExporterParameter.JASPER_PRINT, print);
		return exporter;
	}

	/**
	 * ディフォルトのXLS(JExcepAPI)生成用Exporterオブジェクトを生成します。<br>
	 * ディフォルトのExporterとしてのパラメータは以下の通りです。<br>
	 * <pre>
	 * </pre>
	 * @return JRExporterオブジェクト
	 */
	public static JRExporter createDefaultXLSJXLExporter() {
		JExcelApiExporter exporter = new JExcelApiExporter();
		//exporter.setParameter(JExcelApiExporterParameter.JASPER_PRINT, print);
		return exporter;
	}

	/**
	 * ディフォルトのXLS(POI)生成用Exporterオブジェクトを生成します。<br>
	 * ディフォルトのExporterとしてのパラメータは以下の通りです。<br>
	 * <pre>
	 * </pre>
	 * @return JRExporterオブジェクト
	 */
	public static JRExporter createDefaultXLSPOIExporter() {
		JRXlsExporter exporter = new JRXlsExporter();
		//exporter.setParameter(JRXlsExporterParameter.SHEET_NAMES, "JReport");
		return exporter;
	}

	/**
	 * ディフォルトの直接印刷Exporterオブジェクトを生成します。<br>
	 * ディフォルトのExporterとしてのパラメータは以下の通りです。<br>
	 * <pre>
	 * ・プリンタ選択ダイアログ表示
	 * </pre>
	 * @return JRExporterオブジェクト
	 */
	public static JRExporter createDefaultPrintServiceExporter() {
		JRPrintServiceExporter exporter = new JRPrintServiceExporter();
		//HashPrintServiceAttributeSet attributes = new HashPrintServiceAttributeSet();
		//attributes.add(new PrinterName(JasperPrinterUtil.getPrinterNames()[0], Locale.getDefault()));
		//attributes.add(new PrinterName("PrimoPDF", Locale.getDefault()));
		//exporter.setParameter(JRPrintServiceExporterParameter.PRINT_SERVICE_ATTRIBUTE_SET, attributes);
		exporter.setParameter(JRPrintServiceExporterParameter.DISPLAY_PRINT_DIALOG, Boolean.TRUE);
		return exporter;
	}

	/**
	 * 指定されたプリンタ名の直接印刷Exporterオブジェクトを生成します。<br>
	 * @param printerName 直接印刷対象プリンタ名
	 * @return JRExporterオブジェクト
	 */
	public static JRExporter createNamedPrintServiceExporter(String printerName) {
		HashPrintServiceAttributeSet attribute = JasperPrinterUtil.createPrintServiceAttributeSet(printerName);
		JRPrintServiceExporter exporter = new JRPrintServiceExporter();
		exporter.setParameter(JRPrintServiceExporterParameter.DISPLAY_PRINT_DIALOG, Boolean.FALSE);
		exporter.setParameter(JRPrintServiceExporterParameter.PRINT_SERVICE_ATTRIBUTE_SET, attribute);
		return exporter;
	}
}
