package org.phosphoresce.webcore.ext.hibernate4.transaction;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.phosphoresce.webcore.ext.hibernate4.exception.HibernateTransactionException;

/**
 * トランザクション制御コンテナクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2013/02/01	Kitagawa		新規作成
 *-->
 */
public class TransactionContainer {

	/** トランザクション開始済みフラグ */
	private boolean opened;

	/** セッションファクトリオブジェクト */
	private SessionFactory factory;

	/** セッションオブジェクト */
	private Session session;

	/** トランザクションオブジェクト */
	private Transaction transaction;

	/**
	 * コンストラクタ<br>
	 */
	TransactionContainer() {
		super();
		this.opened = false;
		this.factory = TransactionFactory.instance().createSessionFactory();
	}

	/**
	 * コンストラクタ<br>
	 * @param namespace 名前空間
	 */
	TransactionContainer(String namespace) {
		super();
		this.opened = false;
		this.factory = TransactionFactory.instance().createSessionFactory(namespace);
	}

	/**
	 * Hibernateセッションオブジェクトを取得します。<br>
	 * @return session Hibernateセッションオブジェクト
	 */
	Session getSession() {
		if (!opened) {
			throw new HibernateTransactionException("FHBN00013");
		}
		return session;
	}

	/**
	 * トランザクションを開始します。<br>
	 */
	void open() {
		if (opened) {
			throw new HibernateTransactionException("FHBN00014");
		}
		try {
			session = factory.openSession();
			transaction = session.beginTransaction();
			opened = true;
		} catch (Throwable e) {
			throw new HibernateTransactionException("FHBN00015");
		}
	}

	/**
	 * トランザクションを終了します。<br>
	 */
	void close() {
		if (!opened) {
			throw new HibernateTransactionException("FHBN00013");
		}
		try {
			if (!(transaction.wasCommitted() || transaction.wasRolledBack())) {
				transaction.rollback();
			}
			session.close();
			factory.close();
			opened = false;
		} catch (Throwable e) {
			throw new HibernateTransactionException("FHBN00016");
		}
	}

	/**
	 * トランザクションをコミットします。<br>
	 */
	void commit() {
		if (!opened) {
			throw new HibernateTransactionException("FHBN00013");
		}
		transaction.commit();
	}

	/**
	 * トランザクションをロールバックします。<br>
	 */
	void rollback() {
		if (!opened) {
			throw new HibernateTransactionException("FHBN00013");
		}
		transaction.rollback();
	}
}
