package org.phosphoresce.webcore.ext.hibernate4.transaction;

import java.io.Serializable;
import java.util.Date;

import javax.persistence.Column;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.MappedSuperclass;
import javax.persistence.OneToOne;
import javax.persistence.Version;

import org.phosphoresce.webcore.ext.hibernate4.model.SMUser;
import org.phosphoresce.webcore.ext.hibernate4.model.enums.DeleteType;

/**
 * データベーステーブルエンティティ上位抽象クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2013/02/01	Kitagawa		新規作成
 *-->
 */
@MappedSuperclass
public abstract class AbstractHibernateEntity implements Serializable {

	/** レコードID */
	@Id
	@GeneratedValue(strategy = GenerationType.AUTO)
	@Column(name = "id", unique = true, nullable = false)
	private Integer id = null;

	/** バージョン番号 */
	@Version
	@Column(name = "version_lock", nullable = false)
	private Integer versionLocked = null;

	/** 登録日時 */
	@Column(name = "entry_date", nullable = false)
	private Date enrollDate = DAOUtil.getMinimumDate();

	/** 登録ユーザ */
	@OneToOne
	@JoinColumn(name = "entry_user", nullable = true)
	private SMUser enrollUser = null;

	/** 更新日時 */
	@Column(name = "modify_date", nullable = false)
	private Date modifyDate = DAOUtil.getMinimumDate();

	/** 更新ユーザ */
	@OneToOne
	@JoinColumn(name = "modify_user", nullable = true)
	private SMUser modifyUser = null;

	/** 削除日時 */
	@Column(name = "delete_date", nullable = false)
	private Date deleteDate = DAOUtil.getMinimumDate();

	/** 削除ユーザ */
	@OneToOne
	@JoinColumn(name = "delete_user", nullable = true)
	private SMUser deleteUser = null;

	/** 削除状態 */
	@Column(name = "delete_type", nullable = false, length = 32)
	@Enumerated(EnumType.STRING)
	private DeleteType deleteType = DeleteType.Validate;

	/**
	 * データベース登録処理前のエンティティ内容の整合処理を実施します。<br>
	 */
	void validate() {
		validateEntity();
	}

	/**
	 * データベース登録処理前のエンティティ内容の整合処理を実施します。<br>
	 * 当処理はエンティティ毎に処理が異なります。<br>
	 */
	protected abstract void validateEntity();

	/**
	 * レコードIDを取得します。<br>
	 * @return レコードID
	 */
	public final Integer getId() {
		return id;
	}

	/**
	 * レコードIDを設定します。<br>
	 * @param id レコードID
	 */
	final void setId(Integer id) {
		this.id = id;
	}

	/**
	 * バージョン番号を取得します。<br>
	 * @return バージョン番号
	 */
	public final Integer getVersionLocked() {
		return versionLocked;
	}

	/**
	 * バージョン番号を設定します。<br>
	 * @param versionLocked バージョン番号
	 */
	final void setVersionLocked(Integer versionLocked) {
		this.versionLocked = versionLocked;
	}

	/**
	 * 登録日時を取得します。<br>
	 * @return 登録日時
	 */
	public final Date getEnrollDate() {
		if (enrollDate == null) {
			enrollDate = DAOUtil.getMinimumDate();
		}
		return enrollDate;
	}

	/**
	 * 登録日時を設定します。<br>
	 * @param enrollDate 登録日時
	 */
	final void setEnrollDate(Date enrollDate) {
		if (enrollDate == null) {
			this.enrollDate = DAOUtil.getMinimumDate();
		} else {
			this.enrollDate = enrollDate;
		}
	}

	/**
	 * 登録ユーザを取得します。<br>
	 * @return 登録ユーザ
	 */
	public final SMUser getEnrollUser() {
		return enrollUser;
	}

	/**
	 * 登録ユーザを設定します。<br>
	 * @param enrollUser 登録ユーザ
	 */
	final void setEnrollUser(SMUser enrollUser) {
		this.enrollUser = enrollUser;
	}

	/**
	 * 更新日時を取得します。<br>
	 * @return 更新日時
	 */
	public final Date getModifyDate() {
		if (modifyDate == null) {
			modifyDate = DAOUtil.getMinimumDate();
		}
		return modifyDate;
	}

	/**
	 * 更新日時を設定します。<br>
	 * @param modifyDate 更新日時
	 */
	final void setModifyDate(Date modifyDate) {
		if (modifyDate == null) {
			this.modifyDate = DAOUtil.getMinimumDate();
		} else {
			this.modifyDate = modifyDate;
		}
	}

	/**
	 * 更新ユーザを取得します。<br>
	 * @return 更新ユーザ
	 */
	public final SMUser getModifyUser() {
		return modifyUser;
	}

	/**
	 * 更新ユーザを設定します。<br>
	 * @param modifyUser 更新ユーザ
	 */
	final void setModifyUser(SMUser modifyUser) {
		this.modifyUser = modifyUser;
	}

	/**
	 * 削除日時を取得します。<br>
	 * @return 削除日時
	 */
	public final Date getDeleteDate() {
		if (deleteDate == null) {
			deleteDate = DAOUtil.getMinimumDate();
		}
		return deleteDate;
	}

	/**
	 * 削除日時を設定します。<br>
	 * @param deleteDate 削除日時
	 */
	final void setDeleteDate(Date deleteDate) {
		if (deleteDate == null) {
			this.deleteDate = DAOUtil.getMinimumDate();
		} else {
			this.deleteDate = deleteDate;
		}
		this.deleteDate = deleteDate;
	}

	/**
	 * 削除ユーザを取得します。<br>
	 * @return 削除ユーザ
	 */
	public final SMUser getDeleteUser() {
		return deleteUser;
	}

	/**
	 * 削除ユーザを設定します。<br>
	 * @param deleteUser 削除ユーザ
	 */
	final void setDeleteUser(SMUser deleteUser) {
		this.deleteUser = deleteUser;
	}

	/**
	 * 削除状態を取得します。<br>
	 * @return 削除状態
	 */
	public final DeleteType getDeleteType() {
		if (deleteType == null) {
			deleteType = DeleteType.Validate;
		}
		return deleteType;
	}

	/**
	 * 削除状態を設定します。<br>
	 * @param deleteType 削除状態
	 */
	final void setDeleteType(DeleteType deleteType) {
		if (deleteType == null) {
			this.deleteType = DeleteType.Validate;
		} else {
			this.deleteType = deleteType;
		}
	}
}
