package org.phosphoresce.webcore.core.taglib;

import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.ConfigName;
import org.phosphoresce.webcore.core.Environment;
import org.phosphoresce.webcore.core.config.Config;

/**
 * 関数処理カスタムタグライブラリクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2012/07/04	Kitagawa		新規作成
 *-->
 */
public class Functions {

	/**
	 * 指定されたオブジェクトを文字列として提供します。<br>
	 * オブジェクトにnullが指定された場合は空文字列が返却されます。<br>
	 * @param object オブジェクト
	 * @return オブジェクトから編集された文字列
	 */
	private static String toString(Object object) {
		if (object == null) {
			return "";
		}
		String string = "";
		if (object.getClass().isArray()) {
			Class<?> clazz = object.getClass().getComponentType();
			if (clazz == String.class) {
				string = Arrays.toString((Object[]) object);
			} else if (clazz == boolean.class) {
				string = Arrays.toString((boolean[]) object);
			} else if (clazz == int.class) {
				string = Arrays.toString((int[]) object);
			} else if (clazz == long.class) {
				string = Arrays.toString((long[]) object);
			} else if (clazz == byte.class) {
				string = Arrays.toString((byte[]) object);
			} else if (clazz == short.class) {
				string = Arrays.toString((short[]) object);
			} else if (clazz == float.class) {
				string = Arrays.toString((float[]) object);
			} else if (clazz == double.class) {
				string = Arrays.toString((double[]) object);
			} else if (clazz == char.class) {
				string = Arrays.toString((char[]) object);
			} else {
				string = Arrays.toString((Object[]) object);
			}
		} else {
			string = object.toString();
		}
		return string;
	}

	/**
	 * 指定されたコンテキストパス配下のURLリソースパスを提供します。<br>
	 * サーバー起動時にキャッシュを無効化するために提供するURLはタイムスタンプが付与されます。<br>
	 * @param input コンテキストパスから見たリソースパス
	 * @return リソースパス
	 */
	public static String contextResource(String input) {
		return Environment.getContextPath() + toString(input) + "?" + Environment.getStartedTime();
	}

	/**
	 * 指定されたキーの環境定義を提供します。<br>
	 * @param input 環境定義キー
	 * @return 環境定義値
	 */
	public static String config(String input) {
		return Config.getString(ConfigName.get(toString(input)));
	}

	/**
	 * 指定されたプロパティキーの起動環境変数を提供します。<br>
	 * @param input 環境定義キー
	 * @return 環境定義値
	 */
	public static Object environment(String input) {
		return Environment.getMap().get(toString(input));
	}

	/**
	 * オブジェクトをエスケープして提供します。<br>
	 * オブジェクトはStringが想定されますが、それ以外のオブジェクトが指定された場合はStringに変換されます。<br>
	 * 属性値に利用する文字列を想定したメソッドでスペース、タブ、改行は変換されません。<br>
	 * @param input 入力文字列
	 * @return エスケープ後文字列
	 */
	public static String escapeString(Object input) {
		return StringUtil.toHTMLAttribute(toString(input));
	}

	/**
	 * オブジェクトをエスケープして提供します。<br>
	 * オブジェクトはStringが想定されますが、それ以外のオブジェクトが指定された場合はStringに変換されます。<br>
	 * ドキュメントで利用する文字列を想定したメソッドでスペース、タブ、改行も変換します。<br>
	 * @param input 入力文字列
	 * @return エスケープ後文字列
	 */
	public static String escapeStringForHTML(Object input) {
		return StringUtil.toHTML(toString(input));
	}

	/**
	 * オブジェクトをフォーマットして提供します。<br>
	 * オブジェクトはLong又はDoubleが想定されますが、それ以外のオブジェクトが指定された場合はパースされて処理されます。<br>
	 * @param input 入力数値
	 * @param pattern フォーマットパターン文字列
	 * @return フォーマット済み文字列
	 */
	public static String formatDecimal(Object input, String pattern) {
		if (input == null) {
			return "";
		}
		DecimalFormat format = new DecimalFormat(pattern);
		try {
			return format.format(Double.parseDouble(toString(input)));
		} catch (Throwable e) {
			return toString(input);
		}
	}

	/**
	 * 日付オブジェクトをフォーマットして提供します。<br>
	 * @param input 入力日付
	 * @param pattern フォーマットパターン文字列
	 * @return フォーマット済み文字列
	 */
	public static String formatDate(Date input, String pattern) {
		if (input == null) {
			return "";
		}
		try {
			return new SimpleDateFormat(pattern).format(input);
		} catch (Throwable e) {
			return toString(input);
		}
	}

	/**
	 * 指定されたオブジェクトリストに含まれるオブジェクトであるか判定します。<br>
	 * @param objects オブジェクトリスト
	 * @param object オブジェクト
	 * @return 配列に含まれる場合にtrueを返却
	 */
	public static boolean contains(List<Object> objects, Object object) {
		if (objects == null) {
			return false;
		}
		for (Object o : objects) {
			if (o == null) {
				if (object == null) {
					return true;
				} else {
					if (object.equals(o)) {
						return true;
					}
				}
			} else {
				if (o.equals(object)) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * 指定されたオブジェクト配列に含まれるオブジェクトであるか判定します。<br>
	 * @param objects オブジェクト配列
	 * @param object オブジェクト
	 * @return 配列に含まれる場合にtrueを返却
	 */
	public static boolean contains(Object[] objects, Object object) {
		if (objects == null) {
			return false;
		}
		return contains(Arrays.asList(objects), object);
	}
}
