package org.phosphoresce.lib.poi.adapter;

import java.util.LinkedList;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.ErrorConstants;
import org.apache.poi.ss.usermodel.FormulaEvaluator;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.phosphoresce.lib.poi.PoiGlobal;

/**
 * 内部基本ユーティリティクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日       更新者          更新内容
 * 2005/11/21	Kitagawa		新規作成
 * 2012/07/09	Kitagawa		バージョン2にアップデート
 *-->
 */
class PoiBaseUtil implements PoiGlobal {

	/**
	 * セルに指定されている値を基本型で取得します。<br>
	 * @param cell セルオブジェクト
	 * @return 真偽値、数値、文字列、エラーのいずれかで取得します。<br>
	 */
	public static Object getBaseValue(Cell cell) {
		if (cell == null) {
			return EMPTY;
		}
		try {
			// 真偽値取得
			return cell.getBooleanCellValue();
		} catch (IllegalStateException e1) {
			try {
				// 数値取得
				double d = cell.getNumericCellValue();
				if ((double) ((long) d) == d) {
					return (long) d;
				} else {
					return d;
				}
			} catch (IllegalStateException e2) {
				try {
					// 文字列取得
					return cell.getStringCellValue();
				} catch (IllegalStateException e3) {
					try {
						// エラー取得
						//return cell.getErrorCellValue();
						return ErrorConstants.getText(cell.getErrorCellValue());
					} catch (IllegalStateException e4) {
						return EMPTY;
					}
				}
			}
		}
	}

	/**
	 * 指定されたセルの計算結果を更新します。<br>
	 * @param workbook ワークブックオブジェクト
	 * @param cell セルオブジェクト
	 */
	public static void evaluateCell(Workbook workbook, Cell cell) {
		if (workbook == null) {
			return;
		}
		if (cell == null) {
			return;
		}
		if (cell.getCellType() != Cell.CELL_TYPE_FORMULA) {
			return;
		}
		FormulaEvaluator evaluator = workbook.getCreationHelper().createFormulaEvaluator();
		try {
			evaluator.evaluateFormulaCell(cell);
		} catch (Throwable e) {
			// 非対応自動計算部は無視する
		}
	}

	/**
	 * 行を再計算します。<br>
	 * @param workbook ワークブックオブジェクト
	 * @param row 行オブジェクト
	 */
	public static void evaluateRow(Workbook workbook, Row row) {
		if (workbook == null) {
			return;
		}
		if (row == null) {
			return;
		}
		for (Cell cell : row) {
			evaluateCell(workbook, cell);
		}
	}

	/**
	 * シートを再計算します。<br>
	 * @param workbook ワークブックオブジェクト
	 * @param sheet シートオブジェクト
	 */
	public static void evaluateSheet(Workbook workbook, Sheet sheet) {
		if (workbook == null) {
			return;
		}
		if (sheet == null) {
			return;
		}
		for (Row row : sheet) {
			evaluateRow(workbook, row);
		}
	}

	/**
	 * ワークブックを再計算します。<br>
	 * @param workbook ワークブックオブジェクト
	 */
	public static void evaluateWorkbook(Workbook workbook) {
		if (workbook == null) {
			return;
		}
		//FormulaEvaluator evaluator = workbook.getCreationHelper().createFormulaEvaluator();
		//evaluator.evaluateAll();
		for (int i = 0; i < workbook.getNumberOfSheets(); i++) {
			Sheet sheet = workbook.getSheetAt(i);
			evaluateSheet(workbook, sheet);
		}
	}

	/**
	 * 指定された範囲情報が同一の領域であるかを判定します。<br>
	 * オブジェクトハッシュが同一でない場合でも領域情報が同じ場合は同一とみなします。<br>
	 * @param region1 比較元Regionオブジェクト
	 * @param region2 比較先Regionオブジェクト
	 * @return 同一の領域を持つ場合にtrueを返却
	 */
	public static boolean isEqualsRegion(CellRangeAddress region1, CellRangeAddress region2) {
		if (region1 == null && region2 == null) {
			return true;
		} else if (region1 == null || region2 == null) {
			return false;
		}
		return region1.getFirstColumn() == region2.getFirstColumn() //
				&& region1.getLastColumn() == region2.getLastColumn() //
				&& region1.getFirstRow() == region2.getFirstRow() //
				&& region1.getLastRow() == region1.getLastRow() //
		;
	}

	/**
	 * 指定された位置の結合セルオブジェクトを取得します。<br>
	 * @param sheet 対象のSheetオブジェクト
	 * @param rowIndex 行位置(0～)
	 * @param colIndex 列位置(0～)
	 * @return 指定位置に存在する結合セルオブジェクト
	 */
	public static CellRangeAddress getMargedRegion(Sheet sheet, int rowIndex, int colIndex) {
		for (int i = 0; i <= sheet.getNumMergedRegions() - 1; i++) {
			CellRangeAddress region = sheet.getMergedRegion(i);
			if (rowIndex >= region.getFirstRow() //
					&& rowIndex <= region.getLastRow() //
					&& colIndex >= region.getFirstColumn() //
					&& colIndex <= region.getLastColumn()) {
				return region;
			}
		}
		return null;
	}

	/**
	 * 指定された範囲に含まれる結合セルオブジェクトを全て取得します。<br>
	 * 範囲に結合セルの一部でも含まれる場合はその結合セルも取得対象とします。<br>
	 * @param sheet 対象のSheetオブジェクト
	 * @param range 取得範囲PoiRangeオブジェクト
	 * @return 範囲に含まれる結合情報
	 */
	public static LinkedList<CellRangeAddress> getMargedRegions(Sheet sheet, PoiRange range) {
		LinkedList<CellRangeAddress> regions = new LinkedList<CellRangeAddress>();
		for (int rowIndex = range.getRowIndexFrom(); rowIndex <= range.getRowIndexTo(); rowIndex++) {
			for (int colIndex = range.getColIndexFrom(); colIndex <= range.getColIndexTo(); colIndex++) {
				CellRangeAddress region = getMargedRegion(sheet, rowIndex, colIndex);
				if (region != null) {
					boolean isContain = false;
					for (int i = 0; i <= regions.size() - 1; i++) {
						CellRangeAddress target = (CellRangeAddress) regions.get(i);
						if (isEqualsRegion(target, region)) {
							isContain = true;
						}
					}
					if (!isContain) {
						regions.add(region);
					}
				}
			}
		}
		return regions;
	}
}
