/**
 * サブミット操作クラス<br>
 * <br>
 * ページ内のサブミット操作に関するインタフェースが提供されます。<br>
 *
 * @author Kitagawa<br>
 *
 *<!--
 * 更新日		更新者				更新内容
 * 2012/07/02	Kitagawa			新規作成
 *-->
 */

/**
 * コンストラクタ<br>
 */
var Submit = function() {
};

/** サブミット中フラグ */
Submit.submitted = new Array();

/** 画面項目編集済みフラグ */
Submit.edited = false;

/**
 * 編集済みフラグを設定します。<br>
 * @returns {Boolean} 正常終了した場合はtrueを返却
 */
Submit.setEdited = function() {
	Submit.edited = true;
	return true;
};

/**
 * 編集済みであるか判定します。<br>
 * @returns {Boolean} 編集済みである場合にtrueを返却
 */
Submit.setEdited = function() {
	return Submit.edited;
};

/**
 * 編集済みフラグをクリアします。<br>
 * @returns {Boolean} 正常終了した場合はtrueを返却
 */
Submit.clearEdited = function() {
	Submit.edited = false;
	return true;
};

/**
 * フォームサブミット処理を実施します。<br>
 * @param {Object} options 動作オプション<br>
 * <ul>
 * <li>method : サブミット方法(ディフォルト:"post")</li>
 * <li>form : サブミット対象フォームセレクタ/jQueryオブジェクト(ディフォルト:form[0])</li>
 * <li>url : サブミット対象URL(コンテンツパスからの相対パス)(ディフォルト:"/")</li>
 * <li>editedConfirm : 編集済み確認(ディフォルト:false)</li>
 * <li>editedClear : 編集済みクリア(ディフォルト:false)</li>
 * <li>forbidDoubleSubmit : ダブルサブミット禁止フラグ(ディフォルト:false)</li>
 * <li>lockContent : コンテンツロックフラグ(ディフォルト:true)</li>
 * </ul>
 * @returns {Boolean} 正常終了した場合はtrueを返却
 */
Submit.submit = function(options) {
	var defaults = {
		method : "post",
		form : $("form:first"),
		url : "/",
		editedConfirm : false,
		editedClear : false,
		forbidDoubleSubmit : false,
		lockContent : true
	};
	var mergedOptions = $.extend({}, defaults, options);
	var uid = UniqueID.generate();
	var lockLayer = new LockLayer({
		excludeEvents : ["submit"]
	});
	var $disabled = null;

	try {
		// ダブルサブミットチェック
		if (mergedOptions.forbidDoubleSubmit && Submit.submitted.length > 0) {
			alert(Message.get("FCRE00036"));
			return false;
		}

		// 編集済み確認
		if (mergedOptions.editedConfirm && Submit.edited) {
			if (!window.confirm(Message.get("FCRE00037"))) {
				return false;
			}
		}

		// ダブルサブミット制御フラグ設定
		Submit.submitted[Submit.submitted.length] = uid;

		// 非活性項目の活性化(非活性フィールド強制サブミット対応)
		$disabled = $("input:disabled, textarea:disabled, select:disabled").removeAttr("disabled");

		// コンテンツロック
		if (mergedOptions.lockContent) {
			lockLayer.lock();
		}

		// サブミット処理
		var $targetForm = $(mergedOptions.form);
		$targetForm.attr("action", Environment.REQUEST_CONTEXT_PATH + mergedOptions.url);
		$targetForm.attr("method", mergedOptions.method);
		$targetForm.submit();
	} catch (e) {
		// コンテンツアンロック
		lockLayer.unlock();
		throw e;
	} finally {
		if ($disabled) {
			// 活性化した非活性フィールドを元に戻す
			$disabled.attr("disabled", "disabled");
		}
	}

	return true;
};

/**
 * POST処理を実施します。<br>
 * @param {Object} options 動作オプション(初期値はSubmit.submit参照)<br>
 * @returns {Boolean} 正常終了した場合はtrueを返却
 */
Submit.post = function(options) {
	var defaults = {
		method : "post"
	};
	var mergedOptions = $.extend({}, defaults, options);
	return Submit.submit(mergedOptions);
};

/**
 * GET処理を実施します。<br>
 * @param {Object} options 動作オプション(初期値はSubmit.submit参照)<br>
 * @returns {Boolean} 正常終了した場合はtrueを返却
 */
Submit.get = function(options) {
	var defaults = {
		method : "get"
	};
	var mergedOptions = $.extend({}, defaults, options);
	return Submit.submit(mergedOptions);
};

/**
 * 非同期JSONを利用したフォームサブミット処理を実施します。<br>
 * @param {Object} options 動作オプション<br>
 * <ul>
 * <li>method : サブミット方法(ディフォルト:"post")</li>
 * <li>form : サブミット対象フォームセレクタ/jQueryオブジェクト(ディフォルト:form[0])</li>
 * <li>url : サブミット対象URL(コンテンツパスからの相対パス)(ディフォルト:"/")</li>
 * <li>editedConfirm : 編集済み確認(ディフォルト:false)</li>
 * <li>editedClear : 編集済みクリア(ディフォルト:false)</li>
 * <li>forbidDoubleSubmit : ダブルサブミット禁止フラグ(ディフォルト:false)</li>
 * <li>lockTarget : ロック対象エレメントセレクタ(ディフォルト:null)</li>
 * <li>beforeFunction : サブミット前実行関数(ディフォルト:null)</li>
 * <li>successFunction : サブミット前実行関数(ディフォルト:null)</li>
 * </ul>
 * @returns {Boolean} 正常終了した場合はtrueを返却
 */
Submit.submitAsynJson = function(options) {
	var defaults = {
		method : "post",
		form : $("form:first"),
		url : "/",
		editedConfirm : false,
		editedClear : false,
		forbidDoubleSubmit : false,
		lockTarget : null,
		beforeFunction : null,
		successFunction : null
	};
	var mergedOptions = $.extend({}, defaults, options);
	var uid = UniqueID.generate();
	var lockLayer = new LockLayer({
		target : mergedOptions.lockTarget,
		excludeEvents : ["submit"]
	});

	var $disabled = null;

	try {
		// ダブルサブミットチェック
		if (mergedOptions.forbidDoubleSubmit && Submit.submitted.length > 0) {
			alert(Message.get("FCRE00036"));
			return false;
		}

		// 編集済み確認
		if (mergedOptions.editedConfirm && Submit.edited) {
			if (!window.confirm(Message.get("FCRE00037"))) {
				return false;
			}
		}

		// ダブルサブミット制御フラグ設定
		Submit.submitted[Submit.submitted.length] = uid;

		// 非活性項目の活性化(非活性フィールド強制サブミット対応)
		$disabled = $("input:disabled, textarea:disabled, select:disabled").removeAttr("disabled");

		// フォーム情報のシリアライズ
		var data = $(mergedOptions.form).serialize();

		// 活性化した非活性フィールドを元に戻す
		$disabled.attr("disabled", "disabled");

		// 配列プロパティは除去
		data = data.replace(/%5B/g, "[");
		data = data.replace(/%5D/g, "]");

		// コンテンツロック
		lockLayer.lock();

		// サブミット処理
		$.ajax({
			type : mergedOptions.method,
			url : mergedOptions.url,
			async : true,
			cache : false,
			dataType : "json",
			data : data,
			beforeSend : function(XMLHttpRequest) {
				if (mergedOptions.beforeFunction) {
					mergedOptions.beforeFunction();
				}
				return true;
			},
			error : function(XMLHttpRequest, textStatus, errorThrown) {
				return false;
			},
			success : function(data, dataType) {
				if (mergedOptions.suncessFunction) {
					mergedOptions.suncessFunction();
				}
				return true;
			},
			complete : function(XMLHttpRequest, textStatus) {
				// コンテンツアンロック
				lockLayer.unlock();
				return true;
			}
		});
	} catch (e) {
		// コンテンツアンロック
		lockLayer.unlock();
		throw e;
	} finally {
		if ($disabled) {
			// 活性化した非活性フィールドを元に戻す
			$disabled.attr("disabled", "disabled");
		}
	}

	return true;
};

/**
 * 非同期JSONを利用したPOST処理を実施します。<br>
 * @param {Object} options 動作オプション(初期値はSubmit.submit参照)<br>
 * @returns {Boolean} 正常終了した場合はtrueを返却
 */
Submit.postAsynJson = function(options) {
	var defaults = {
		method : "post"
	};
	var mergedOptions = $.extend({}, defaults, options);
	return Submit.submitAsynJson(mergedOptions);
};

/**
 * 非同期JSONを利用したGET処理を実施します。<br>
 * @param {Object} options 動作オプション(初期値はSubmit.submit参照)<br>
 * @returns {Boolean} 正常終了した場合はtrueを返却
 */
Submit.getAsynJson = function(options) {
	var defaults = {
		method : "get"
	};
	var mergedOptions = $.extend({}, defaults, options);
	return Submit.submitAsynJson(mergedOptions);
};
