package org.phosphoresce.lib.commons.util;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 文字列に対する操作を行うメソッドを提供するクラス<br>
 * <br>
 * 当メソッドが提供するインタフェースの多くはnullオブジェクトが指定された場合でも
 * 空文字列として扱い、NullPointerExceptionがスローされることはありません。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2005/07/02	Kitagawa		新規作成
 * 2006/05/10	Kitagawa		isAscii、バイト操作関係メソッドインタフェースにエンコード方式パラメータ追加
 * 2006/07/10	Kitagawa		無駄なStringインスタンス生成を除去
 * 2007/01/19	Kitagawa		半角記号文字配列定義を追加
 * 2007/02/01	Kitagawa		toHTML追加
 * 2007/02/24	Kitagawa		escapeFilename追加
 * 2008/05/17	Kitagawa		equals追加
 * 2008/10/08	Kitagawa		native2ascii、ascii2native追加
 * 2008/10/08	Kitagawa		render*メソッドをto*メソッドにリファクタリング
 * 2008/11/13	Kitagawa		equalsIgnoreCase追加
 * 2008/11/14	Kitagawa		contains、containsIgnoreCase追加
 * 2008/11/21	Kitagawa		nullsafe追加
 * 2008/12/10	Kitagawa		native2ascii、ascii2nativeメソッド名変更(nativeToAscii、asciiToNative)
 * 2008/12/10	Kitagawa		nativeToBase64、base64ToNativeメソッド追加
 * 2009/11/20	Kitagawa		toLowerCamelCase、toUpperCamelCase、toLowerSnakeCase、toUpperSnakeCaseメソッド追加
 * 2009/12/15	Kitagawa		encode(String, String, String)、convert(String, String)、space(int)追加
 * 2009/12/16	Kitagawa		supplementByte**メソッドをsupplyByte**に名称変更
 * 2009/12/16	Kitagawa		各種メソッドにおいてNull不許可なパラメータは予め例外をスローするように修正
 * 2009/12/16	Kitagawa		冗長的なコードを汎用メソッドに委譲するように修正
 * 2009/12/16	Kitagawa		isValidCharacers追加、isInLenght***→is***LengthIn名称変更
 * 2009/12/17	Kitagawa		nativeToAscii、asciiToNativeをNative2AsciiUtilに委譲するように修正
 * 2009/12/18	Kitagawa		paddingChar、paddingCharHead、paddingCharFoot追加
 * 2010/07/14	Kitagawa		splitメソッド追加
 * 2010/10/04	Kitagawa		isValidityDateFormatメソッド追加
 * 2011/04/22	Kitagawa		charAtb、asByteArrayByHexメソッド追加
 * 2011/05/10	Kitagawa		String bind(String, Map)メソッド追加
 * 2011/05/11	Kitagawa		decodeKEIS83Char、decodeKEIS83メソッド追加
 * 2011/05/13	Kitagawa		get/setUtilJavaCharset、get/setUtilOutputCharsetメソッド追加
 * 2011/05/17	Kitagawa		decodeEBCDIKChar、decodeEBCDICCharメソッド追加
 * 2011/05/17	Kitagawa		EBCDICType列挙型インナークラス追加
 * 2012/06/30	Kitagawa		JDK1.5以上を前提としたユーティリティに修正、各種メソッド名修正
 *-->
 */
public class StringUtil {

	/** 空文字列 */
	public static final String EMPTY = "";

	/** 半角スペース */
	public static final char ASCII_SPACE = ' ';

	/** ヘキサ文字列 */
	private static final String[] HEX_STRINGS = { "0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "a", "b", "c", "d", "e", "f" };

	/** BASE64変換文字列 */
	public static final String BASE64_CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";

	/** ディフォルト改行コード */
	public static final String NEWLINE = System.getProperty("line.separator");

	/** 改行コード(CR+LF) */
	public static final String NEWLINE_CRLF = new String(new byte[] { 0x0D, 0x0A });

	/** 改行コード(CR) */
	public static final String NEWLINE_CR = new String(new byte[] { 0x0D });

	/** 改行コード(LF) */
	public static final String NEWLINE_LF = new String(new byte[] { 0x0A });

	/** ディフォルトJavaキャラクタセット */
	public static final String DEFAULT_JAVA_CHARSET = "MS932";

	/** ディフォルト出力ファイルキャラクタセット */
	public static final String DEFAULT_OUTPUTFILE_CHARSET = "8859_1";

	/** ユーティリティ内利用Javaキャラクタセット */
	private static String utilJavaCharset = DEFAULT_JAVA_CHARSET;

	/** ユーティリティ内利用ファイルキャラクタセット */
	private static String utilOutputCharset = DEFAULT_OUTPUTFILE_CHARSET;

	/** ランダム文字列生成時付加文字 */
	private static final List<String> RANDOM_SEED_CHARS;
	static {
		String CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!\"#$%&'()=~|-^\\[]{}@:`*/.?><,";
		RANDOM_SEED_CHARS = new LinkedList<String>();
		for (int i = 0; i <= CHARS.length() - 1; i++) {
			RANDOM_SEED_CHARS.add(String.valueOf(CHARS.charAt(i)));
		}
	}

	/** ランダム文字列生成時付加文字インデックス */
	private static Integer randomSeedCharIndex = 0;

	/**
	 * EBCDICコードタイプ列挙型クラス<br>
	 * 
	 * @author Kitagawa<br>
	 * 
	 *<!--
	 * 更新日		更新者			更新内容
	 * 2011/05/17	Kitagawa		新規作成
	 *-->
	 */
	public static class EBCDICType {

		/** EBCDIC */
		public static final EBCDICType EBCDIC = new EBCDICType("EBCDIC");

		/** EBCDIK */
		public static final EBCDICType EBCDIK = new EBCDICType("EBCDIK");

		/** タイプ */
		private String type;

		/**
		 * コンストラクタ<br>
		 * @param type タイプ
		 */
		private EBCDICType(String type) {
			super();
			this.type = type;
		}

		/**
		 * コンストラクタ<br>
		 */
		private EBCDICType() {
			this("");
		}

		/**
		 * 指定されたEBCDICコードタイプ文字列からコードタイプインスタンスを取得します。<br>
		 * @param type タイプ文字列(EBCDIC/EBCDIK)
		 * @return コードタイプインスタンス
		 */
		public static EBCDICType valueOf(String type) {
			if ("EBCDIC".equalsIgnoreCase(type)) {
				return EBCDIC;
			} else if ("EBCDIK".equalsIgnoreCase(type)) {
				return EBCDIK;
			} else {
				throw new IllegalArgumentException("type");
			}
		}
	}

	/**
	 * コンストラクタ<br>
	 */
	private StringUtil() {
		super();
	}

	/**
	 * ユーティリティ内利用Javaキャラクタセットを取得します。<br>
	 * @return ユーティリティ内利用Javaキャラクタセット
	 */
	public static String getUtilJavaCharset() {
		return utilJavaCharset;
	}

	/**
	 * ユーティリティ内利用Javaキャラクタセットを設定します。<br>
	 * Javaキャラクタセットを指定するメソッドにおいてキャラクタセットパラメータが省略された際に利用されます。<br>
	 * @param utilJavaCharset ディフォルト使用Javaキャラクタセット
	 */
	public static void setUtilJavaCharset(String utilJavaCharset) {
		try {
			new String(new byte[] { 0x00 }, utilJavaCharset);
		} catch (UnsupportedEncodingException e) {
			throw new IllegalArgumentException("unsupported charset (" + utilJavaCharset + ")");
		}
		StringUtil.utilJavaCharset = utilJavaCharset;
	}

	/**
	 * ユーティリティ内利用ファイルキャラクタセットを取得します。<br>
	 * @return ユーティリティ内利用ファイルキャラクタセット
	 */
	public static String getUtilOutputCharset() {
		return utilOutputCharset;
	}

	/**
	 * ユーティリティ内利用ファイルキャラクタセットを設定します。<br>
	 * 出力キャラクタセットを指定するメソッドにおいてキャラクタセットパラメータが省略された際に利用されます。<br>
	 * @param utilOutputCharset ユーティリティ内利用ファイルキャラクタセット
	 */
	public static void setUtilOutputCharset(String utilOutputCharset) {
		try {
			new String(new byte[] { 0x00 }, utilOutputCharset);
		} catch (UnsupportedEncodingException e) {
			throw new IllegalArgumentException("unsupported charset (" + utilOutputCharset + ")");
		}
		StringUtil.utilOutputCharset = utilOutputCharset;
	}

	/**
	 * ランダムなハッシュ文字列を提供します。<br>
	 * @return ランダムなハッシュ文字列
	 */
	public static String createRandomHash() {
		synchronized (randomSeedCharIndex) {
			if (RANDOM_SEED_CHARS.size() - 1 >= randomSeedCharIndex + 1) {
				randomSeedCharIndex++;
			} else {
				randomSeedCharIndex = 0;
			}
			long time = System.currentTimeMillis();
			long rval = new Random(time).nextLong();
			return StringUtil.toMD5Hash(String.valueOf(rval) + RANDOM_SEED_CHARS.get(randomSeedCharIndex));
		}
	}

	/**
	 * 指定された文字列がnull又は空文字列か判定します。<br>
	 * @param value 対象文字列
	 * @return 文字列がnull又は空文字列の場合にtrueを返却
	 */
	public static boolean isEmpty(String value) {
		return value == null || value.length() == 0;
	}

	/**
	 * 指定された文字列同士が同一であるか判定します。<br>
	 * 文字列が両者ともnullの場合も同一として判定します。<br>
	 * 当メソッドでは大文字小文字の判断を行います。<br>
	 * @param value1 対象文字列
	 * @param value2 対象文字列
	 * @return 同一の文字列の場合にtrueを返却
	 */
	public static boolean equals(String value1, String value2) {
		if (value1 == null && value2 == null) {
			return true;
		} else if (value1 != null) {
			return value1.equals(value2);
		} else {
			return value2.equals(value1);
		}
	}

	/**
	 * 指定された文字列同士が同一であるか判定します。<br>
	 * 文字列が両者ともnullの場合も同一として判定します。<br>
	 * また、当メソッドでは大文字小文字の区別は行いません。<br>
	 * @param value1 対象文字列
	 * @param value2 対象文字列
	 * @return 同一の文字列の場合にtrueを返却
	 */
	public static boolean equalsIgnoreCase(String value1, String value2) {
		if (value1 == null && value2 == null) {
			return true;
		} else if (value1 != null) {
			return value1.equalsIgnoreCase(value2);
		} else {
			return value2.equalsIgnoreCase(value1);
		}
	}

	/**
	 * 指定された文字列が空の場合にnullを返却します。<br>
	 * @param value 対象文字列
	 * @return 編集された文字列
	 */
	public static String empty2null(String value) {
		return isEmpty(value) ? null : value;
	}

	/**
	 * 指定された文字列がnullの場合に別の文字列に置換して提供します。<br>
	 * @param value 対象文字列
	 * @param render null時置換文字列
	 * @return 編集された文字列
	 */
	public static String nvl(String value, String render) {
		return value == null ? render : value;
	}

	/**
	 * 指定された文字列が空の場合に別の文字列に置換して提供します。<br>
	 * @param value 対象文字列
	 * @param render null時置換文字列
	 * @return 編集された文字列
	 */
	public static String evl(String value, String render) {
		return StringUtil.isEmpty(value) ? render : value;
	}

	/**
	 * 指定された文字列がnullの場合に空文字列に置換して提供します。<br>
	 * @param value 対象文字列
	 * @return 編集された文字列
	 */
	public static String nvl(String value) {
		return nvl(value, EMPTY);
	}

	/**
	 * 指定された数の半角スペース文字列を生成します。<br>
	 * @param count 生成するスペース数
	 * @return 半角スペース文字列
	 */
	public static String space(int count) {
		StringBuilder builder = new StringBuilder();
		for (int i = 0; i <= count - 1; i++) {
			builder.append(ASCII_SPACE);
		}
		return builder.toString();
	}

	/**
	 * 指定された文字列の文字長を取得します。<br>
	 * @param value 対象文字列
	 * @return 文字長
	 */
	public static int len(String value) {
		return nvl(value).length();
	}

	/**
	 * 指定された文字列のバイト長を取得します。<br>
	 * @param value 対象文字列
	 * @param charset キャラクタセット
	 * @return 文字列バイト長
	 */
	public static int lenb(String value, String charset) {
		try {
			if (charset == null) {
				return nvl(value).getBytes(utilJavaCharset).length;
			} else {
				return nvl(value).getBytes(charset).length;
			}
		} catch (UnsupportedEncodingException e) {
			throw new IllegalArgumentException("unsupported charset (" + charset + ")");
		}
	}

	/**
	 * 指定された文字列のバイト長を取得します。<br>
	 * @param value 対象文字列
	 * @return 文字列バイト長
	 */
	public static int lenb(String value) {
		return lenb(value, null);
	}

	/**
	 * 指定した文字数範囲で文字列をカットして提供します。<br>
	 * @param value 対象文字列
	 * @param position 開始位置
	 * @param length 取得文字数
	 * @return 編集を行った文字列
	 */
	public static String substr(String value, int position, int length) {
		if (value == null || length <= 0 || position > len(value)) {
			return EMPTY;
		}
		if (position < 0) {
			position = 0;
		}
		if (length < 0) {
			length = 0;
		}
		if (len(value) < position + length) {
			length = len(value) - position;
		}
		return value.substring(position, position + length);
	}

	/**
	 * 指定したバイト長で文字列をカットして提供します。<br>
	 * @param value 対象文字列
	 * @param position 開始位置
	 * @param length 取得バイト数
	 * @param charset キャラクタセット
	 * @return 編集を行った文字列
	 */
	public static String substrb(String value, int position, int length, String charset) {
		if (value == null || length <= 0 || position > lenb(value, charset) - 1) {
			return EMPTY;
		}
		if (position < 0) {
			position = 0;
		}
		if (length < 0) {
			length = 0;
		}
		if (lenb(value, charset) < position + length) {
			length = lenb(value, charset) - position;
		}

		StringBuilder header = new StringBuilder();
		for (int i = 0; true; i++) {
			if (lenb(header.toString(), charset) >= position) {
				break;
			}
			header.append(value.substring(i, i + 1));
		}
		String space = space(lenb(header.toString(), charset) - position);
		String buffer = new StringBuffer(space).append(value.substring(header.toString().length())).toString();
		while (lenb(buffer, charset) > length) {
			buffer = buffer.substring(0, buffer.length() - 1);
		}

		return new StringBuilder(buffer).append(space(length - lenb(buffer, charset))).toString();
	}

	/**
	 * 指定したバイト長で文字列をカットして提供します。<br>
	 * @param value 対象文字列
	 * @param position 開始位置
	 * @param length 取得バイト数
	 * @return 編集を行った文字列
	 */
	public static String substrb(String value, int position, int length) {
		return substrb(value, position, length, null);
	}

	/**
	 * 指定された文字列を左から指定された長さで抽出して取得します。<br>
	 * @param value 対象文字列
	 * @param length 文字列長
	 * @return 編集後文字列
	 */
	public static String left(String value, int length) {
		return substr(value, 0, length);
	}

	/**
	 * 指定された文字列を左から指定されたバイト長で抽出して取得します。<br>
	 * @param value 対象文字列
	 * @param length バイト長
	 * @param charset キャラクタセット
	 * @return 編集後文字列
	 */
	public static String leftb(String value, int length, String charset) {
		return substrb(value, 0, length, charset);
	}

	/**
	 * 指定された文字列を左から指定されたバイト長で抽出して取得します。<br>
	 * @param value 対象文字列
	 * @param length 文字長
	 * @return 編集後文字列
	 */
	public static String leftb(String value, int length) {
		return leftb(value, length, null);
	}

	/**
	 * 指定された文字列を右から指定された長さで抽出して取得します。<br>
	 * @param value 対象文字列
	 * @param length 文字長
	 * @return 編集後文字列
	 */
	public static String right(String value, int length) {
		return substr(value, len(value) - length, length);
	}

	/**
	 * 指定された文字列を右から指定されたバイト長で抽出して取得します。<br>
	 * @param value 対象文字列
	 * @param length バイト長
	 * @param charset キャラクタセット
	 * @return 編集後文字列
	 */
	public static String rightb(String value, int length, String charset) {
		return substrb(value, lenb(value) - length, length, charset);
	}

	/**
	 * 指定された文字列を右から指定されたバイト長で抽出して取得します。<br>
	 * @param value 対象文字列
	 * @param length 文字長
	 * @return 編集後文字列
	 */
	public static String rightb(String value, int length) {
		return rightb(value, length, null);
	}

	/**
	 * 指定された文字列を指定されたインデックスの開始、終了の範囲で抽出して取得します。<br>
	 * @param value 対象文字列
	 * @param start 開始インデックス
	 * @param end 終了インデックス
	 * @return 編集後文字列
	 */
	public static String mid(String value, int start, int end) {
		String buffer = right(value, len(value) - start);
		return left(buffer, end - start + 1);
	}

	/**
	 * 指定された文字列を指定されたバイトインデックスの開始、終了の範囲で抽出して取得します。<br>
	 * @param value 対象文字列
	 * @param start 開始バイトインデックス
	 * @param end 終了バイトインデックス
	 * @param charset キャラクタセット
	 * @return 指定された文字列を指定されたインデックスの開始、終了の範囲で抽出した文字列
	 */
	public static String midb(String value, int start, int end, String charset) {
		String buffer = rightb(value, lenb(value, charset) - start, charset);
		return leftb(buffer, end - start + 1, charset);
	}

	/**
	 * 指定された文字列を指定されたバイトインデックスの開始、終了の範囲で抽出して取得します。<br>
	 * @param value 対象文字列
	 * @param start 開始バイトインデックス
	 * @param end 終了バイトインデックス
	 * @return 指定された文字列を指定されたインデックスの開始、終了の範囲で抽出した文字列
	 */
	public static String midb(String value, int start, int end) {
		return midb(value, start, end, null);
	}

	/**
	 * 指定された文字列の置換を行い、その結果を取得します。<br>
	 * 当メソッドは正規表現は使用しません。<br>
	 * @param source 処理の対象の文字列
	 * @param before 置換前の文字列
	 * @param after 置換後の文字列
	 * @return 置換処理後の文字列
	 */
	public static String replace(String source, String before, String after) {
		if (isEmpty(source) || source.indexOf(before) == -1) {
			return source;
		}
		StringBuilder builder = new StringBuilder();
		int index = source.indexOf(before);
		builder.append(source.substring(0, index) + after);
		if (index + before.length() < source.length()) {
			String rest = source.substring(index + before.length(), source.length());
			builder.append(replace(rest, before, after));
		}
		return builder.toString();
	}

	/**
	 * 指定された文字列を全て逆順に並べなおして提供します。<br>
	 * @param source 並べ替える対象の文字列
	 * @return 指定された文字列を全て逆順に並べなおした文字列
	 */
	public static String reverse(String source) {
		return new StringBuilder(source).reverse().toString();
	}

	/**
	 * 指定された文字列内に含まれる指定文字の数を取得します。<br>
	 * @param source 検索対象文字列
	 * @param c 検索文字
	 * @return 文字列内に含まれる指定文字の数
	 */
	public static int count(String source, char c) {
		if (isEmpty(source)) {
			return 0;
		}
		int count = 0;
		for (int i = 0; i <= source.length() - 1; i++) {
			if (source.charAt(i) == c) {
				count++;
			}
		}
		return count;
	}

	/**
	 * 指定された文字列を指定バイト長となるまで文字を追加して提供します。<br>
	 * @param value 元文字列
	 * @param lenb 調整バイト長
	 * @param charset キャラクタセット
	 * @param c 補正文字
	 * @param left 文字列左側に補完する場合はtrueを指定
	 * @return 補完された文字列
	 */
	public static String padding(String value, int lenb, String charset, char c, boolean left) {
		charset = charset == null ? utilJavaCharset : charset;
		if (lenb <= lenb(nvl(value), charset)) {
			return nvl(value);
		}
		StringBuilder builder = new StringBuilder(nvl(value));
		while (lenb(builder.toString(), charset) < lenb) {
			if (left) {
				builder.insert(0, c);
			} else {
				builder.append(c);
			}
		}
		return leftb(builder.toString(), lenb, charset);
	}

	/**
	 * 指定された文字列を指定バイト長となるまで文字を追加して提供します。<br>
	 * @param value 元文字列
	 * @param lenb 調整バイト長
	 * @param c 補正文字
	 * @param left 文字列左側に補完する場合はtrueを指定
	 * @return 補完された文字列
	 */
	public static String padding(String value, int lenb, char c, boolean left) {
		return padding(value, lenb, null, c, left);
	}

	/**
	 * 指定された文字列を指定バイト長となるまで先頭にスペースを追加して提供します。<br>
	 * @param value 元文字列
	 * @param lenb 調整バイト長
	 * @param charset キャラクタセット
	 * @return 補完された文字列
	 */
	public static String paddinglSpace(String value, int lenb, String charset) {
		return padding(value, lenb, charset, ASCII_SPACE, true);
	}

	/**
	 * 指定された文字列を指定バイト長となるまで先頭にスペースを追加して提供します。<br>
	 * @param value 元文字列
	 * @param lenb 調整バイト長
	 * @return 補完された文字列
	 */
	public static String paddinglSpace(String value, int lenb) {
		return padding(value, lenb, null, ASCII_SPACE, true);
	}

	/**
	 * 指定された文字列を指定バイト長となるまで末尾にスペースを追加して提供します。<br>
	 * @param value 元文字列
	 * @param lenb 調整バイト長
	 * @param charset キャラクタセット
	 * @return 補完された文字列
	 */
	public static String paddingrSpace(String value, int lenb, String charset) {
		return padding(value, lenb, charset, ASCII_SPACE, false);
	}

	/**
	 * 指定された文字列を指定バイト長となるまで末尾にスペースを追加して提供します。<br>
	 * @param value 元文字列
	 * @param lenb 調整バイト長
	 * @return 補完された文字列
	 */
	public static String paddingrSpace(String value, int lenb) {
		return padding(value, lenb, null, ASCII_SPACE, false);
	}

	/**
	 * 指定された文字列を指定されたキャラクタセットにエンコードします。<br>
	 * @param value 変換対象文字列
	 * @param charsetFrom 変換前キャラクタセット
	 * @param charsetTo 変換後キャラクタセット
	 * @return キャラクタエンコードされた文字列
	 */
	public static String encode(String value, String charsetFrom, String charsetTo) {
		if (value == null) {
			return null;
		} else {
			try {
				charsetFrom = charsetFrom == null ? utilJavaCharset : charsetFrom;
				charsetTo = charsetTo == null ? utilJavaCharset : charsetTo;
				return new String(value.getBytes(charsetFrom), charsetTo);
			} catch (UnsupportedEncodingException e) {
				throw new IllegalArgumentException("unsupported charset (" + charsetFrom + ")");
			}
		}
	}

	/**
	 * 指定された文字列(MS932を想定)を指定されたキャラクタセットにエンコードします。<br>
	 * @param value 変換対象文字列
	 * @param charset キャラクタセット
	 * @return キャラクタエンコードされた文字列
	 */
	public static String encode(String value, String charset) {
		return encode(value, null, charset);
	}

	/**
	 * 指定された文字列をURL文字列ににエンコードします。<br>
	 * @param value 変換対象文字列
	 * @param charset キャラクタセット
	 * @return エンコードされたURL文字列
	 */
	public static String encodeURL(String value, String charset) {
		try {
			return URLEncoder.encode(nvl(value), charset);
		} catch (UnsupportedEncodingException e) {
			throw new IllegalArgumentException("unsupported charset (" + charset + ")");
		}
	}

	/**
	 * 指定されたエンコード済みURL文字列をデコードします。<br>
	 * @param value 変換対象文字列
	 * @param charset キャラクタセット
	 * @return エンコードされたURL文字列
	 */
	public static String decodeURL(String value, String charset) {
		try {
			return URLDecoder.decode(nvl(value), charset);
		} catch (UnsupportedEncodingException e) {
			throw new IllegalArgumentException("unsupported charset (" + charset + ")");
		}
	}

	/**
	 * 指定された文字列を指定されたキャラクタセットに変換してそのバイト情報を提供します。<br>
	 * @param value 変換対象文字列
	 * @param charset 変換対象キャラクタセット
	 * @return 変換された文字列のバイト情報
	 */
	public static byte[] convert(String value, String charset) {
		try {
			CharsetEncoder encoder = Charset.forName(charset).newEncoder();

			value = value.replaceAll("\uff5e", "\u301c"); //～
			value = value.replaceAll("\uff0d", "\u2212"); //－

			ByteBuffer encoded = encoder.encode(CharBuffer.wrap(value.toCharArray()));
			byte[] encodedArray = encoded.array();

			List<Byte> buffer = new LinkedList<Byte>();
			for (int i = 0; i < encoded.limit(); i++) {
				buffer.add(new Byte(encodedArray[i]));
			}

			byte[] result = result = new byte[buffer.size()];
			for (int i = 0; i < result.length; i++) {
				result[i] = ((Byte) buffer.get(i)).byteValue();
			}

			return result;
		} catch (CharacterCodingException e) {
			throw new IllegalArgumentException("unsupported charset (" + charset + ")");
		}
	}

	/**
	 * 指定された文字列内の{#}に対してパラメータStringオブジェクト配列順に文字列を挿入して提供します(MessageFormatは利用しません)。<br>
	 * パラメータStringオブジェクトがnullの場合は空文字を挿入します。<br>
	 * @param source バインド対象ベース文字列
	 * @param params バインドパラメータStringオブジェクト配列
	 * @return バインド編集後文字列
	 */
	public static String bind(String source, String... params) {
		if (params == null || params.length == 0 || source == null || source.length() == 0) {
			return nvl(source);
		}
		String render = source;
		for (int i = 0; i <= params.length - 1; i++) {
			String value = nvl(params[i]);
			render = replace(render, "{" + i + "}", value);
		}
		return render;
	}

	/**
	 * 指定された文字列内の{#}に対してパラメータStringオブジェクト配列順に文字列を挿入して提供します(MessageFormatは利用しません)。<br>
	 * パラメータStringオブジェクトがnullの場合は空文字を挿入します。<br>
	 * @param source バインド対象ベース文字列
	 * @param params バインドパラメータObjectオブジェクト配列
	 * @return バインド編集後文字列
	 */
	public static String bind(String source, Object... params) {
		List<String> list = new LinkedList<String>();
		if (params != null) {
			for (Object object : params) {
				list.add(object == null ? "" : object.toString());
			}
		}
		return bind(source, list.toArray(new String[0]));
	}

	/**
	 * 指定された文字列内の${****}に対してマップキーセットのオブジェクトを挿入した文字列を提供します。<br>
	 * パラメータStringオブジェクトがnullの場合は空文字を挿入します。<br>
	 * @param source バインド対象ベース文字列
	 * @param map バインドパラメータマップ
	 * @return バインド編集後文字列
	 */
	public static String bind(String source, Map<String, String> map) {
		if (map == null || map.size() == 0 || source == null || source.length() == 0) {
			return nvl(source);
		}
		String render = source;
		for (Iterator<String> iterator = map.keySet().iterator(); iterator.hasNext();) {
			String key = iterator.next();
			String value = nvl(map.get(key));
			render = replace(render, "${" + key + "}", value);
		}
		return render;
	}

	/**
	 * 指定された'83版KEISの16進コードから文字列に変換します。<br>
	 * @param hex '83版KEISの16進コード
	 * @param type EBCDICコードタイプ
	 * @return 変換後文字列
	 */
	public static String decodeKEIS83(String hex, EBCDICType type) {
		StringBuffer buffer = new StringBuffer();
		for (int i = 0; i <= hex.length() - 1; i += 4) {
			String value = substrb(hex, i, 4);
			String valueH = substrb(value, 0, 2).trim();
			if (valueH.length() == 0) {
				valueH = "00";
			}
			String valueL = substrb(value, 2, 2).trim();
			if (valueL.length() == 0) {
				valueL = "00";
			}

			// 機能キャラクタ範囲内処理(無変換)
			if ("0A41".equals(value) || "0A42".equals(value)) {
				i += 4;
				// 末尾が機能キャラクタの場合は処理終了
				if (i > hex.length() - 1) {
					break;
				}
				while (true) {
					value = substrb(hex, i, 4);
					if ("0A41".equals(value) || "0A42".equals(value)) {
						break;
					}
					value = substrb(hex, i, 2);
					if (EBCDICType.EBCDIC.equals(type)) {
						buffer.append(decodeEBCDICChar(value));
					} else if (EBCDICType.EBCDIK.equals(type)) {
						buffer.append(decodeEBCDIKChar(value));
					} else {
						buffer.append("?");
					}
					i += 2;
					if (i > hex.length() - 1) {
						break;
					}
				}
				continue;
			}

			// 1バイトコード(EBCDIKコード/EBCDICコード)判別(上位バイトが0x00～0x40)
			if (Integer.decode("0x" + valueH).intValue() >= 0x00 //
					&& Integer.decode("0x" + valueH).intValue() <= 0x40) {
				if (EBCDICType.EBCDIC.equals(type)) {
					buffer.append(decodeEBCDICChar(valueH));
				} else if (EBCDICType.EBCDIK.equals(type)) {
					buffer.append(decodeEBCDIKChar(valueH));
				} else {
					buffer.append("?");
				}
				i -= 2;
				continue;
			}

			// SJIS外字コード(上位バイトが0x41～0xA0かつ下位バイトが0xA1～FE)
			if (Integer.decode("0x" + valueH).intValue() >= 0x41 //
					&& Integer.decode("0x" + valueH).intValue() <= 0xA0 //
					&& Integer.decode("0x" + valueL).intValue() >= 0xA1 //
					&& Integer.decode("0x" + valueL).intValue() <= 0xFE) {
				buffer.append("〓");
				continue;
			}

			// SJIS未対応コード(上位バイトが0x41～0xFEかつ下位バイトが0x00～0x0A、または上位バイトが0xFF、または下位バイトが0xFF)
			if ((Integer.decode("0x" + valueH).intValue() >= 0x41 && Integer.decode("0x" + valueH).intValue() <= 0xFE && Integer.decode("0x" + valueL).intValue() >= 0x00 && Integer.decode("0x" + valueL).intValue() <= 0x0A) //
					|| (Integer.decode("0x" + valueH).intValue() == 0xFF || Integer.decode("0x" + valueL).intValue() == 0xFF)) {
				buffer.append("〓");
				continue;
			}

			// 全角文字変換
			buffer.append(decodeKEIS83Char(value));
		}

		return buffer.toString();
	}

	/**
	 * 指定されたEBCDIK16進コードをデコードした文字を取得します。<br>
	 * @param hex EBCDIK16進コード表記文字列
	 * @return デコードした文字
	 */
	public static char decodeEBCDIKChar(String hex) {
		// EBCDIK→SJISコードマッピング配列(0x00～0xFF)
		final int EBCDIK_MAPPING[] = new int[] { //
		//
				0, 1, 2, 3, 156, 9, 134, 127, 151, 141, 142, 11, 12, 13, 14, 15, //
				16, 17, 18, 19, 157, 10, 8, 135, 24, 25, 146, 143, 28, 29, 30, 31, //
				128, 129, 130, 131, 132, 133, 23, 27, 136, 137, 138, 139, 140, 5, 6, 7, //
				144, 145, 22, 147, 148, 149, 150, 4, 152, 153, 154, 155, 20, 21, 158, 26, //
				32, 161, 162, 163, 164, 165, 166, 167, 168, 169, 91, 46, 60, 40, 43, 33, //
				38, 170, 171, 172, 173, 174, 175, 160, 176, 97, 93, 92, 42, 41, 59, 94, //
				45, 47, 98, 99, 100, 101, 102, 103, 104, 105, 124, 44, 37, 95, 62, 63, //
				106, 107, 108, 109, 110, 111, 112, 113, 114, 96, 58, 35, 64, 39, 61, 34, //
				115, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 116, 187, 188, 189, 190, //
				191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 117, 118, 202, 203, 204, //
				119, 126, 205, 206, 207, 208, 209, 210, 211, 212, 213, 120, 214, 215, 216, 217, //
				121, 122, 224, 225, 226, 227, 228, 229, 230, 231, 218, 219, 220, 221, 222, 223, //
				123, 65, 66, 67, 68, 69, 70, 71, 72, 73, 232, 233, 234, 235, 236, 237, //
				125, 74, 75, 76, 77, 78, 79, 80, 81, 82, 238, 239, 240, 241, 242, 243, //
				36, 159, 83, 84, 85, 86, 87, 88, 89, 250, 244, 245, 246, 247, 248, 249, //
				48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 250, 251, 252, 253, 254, 255, //
		//
		};
		if (!hex.matches("[A-Za-z0-9]+")) {
			throw new IllegalArgumentException(hex + " not [A-Za-z0-9]+");
		}
		int code = Integer.decode("0x" + hex).intValue();
		if (code > EBCDIK_MAPPING.length - 1) {
			throw new IllegalArgumentException("0x" + hex + " > 255");
		}
		if (code < 0) {
			throw new IllegalArgumentException("0x" + hex + " < 0");
		}
		try {
			return new String(new byte[] { (byte) EBCDIK_MAPPING[code] }, "SJIS").charAt(0);
		} catch (UnsupportedEncodingException e) {
			// SJISコード未サポート例外がスローされることはない
			return 0x00;
		}
	}

	/**
	 * 指定されたEBCDIC16進コードをデコードした文字を取得します。<br>
	 * @param hex EBCDIC16進コード表記文字列
	 * @return デコードした文字
	 */
	public static char decodeEBCDICChar(String hex) {
		// EBCDIK→SJISコードマッピング配列(0x00～0xFF)
		final int EBCDIK_MAPPING[] = new int[] { //
		//
				0, 1, 2, 3, 156, 9, 134, 127, 151, 141, 142, 11, 12, 13, 14, 15, //
				16, 17, 18, 19, 157, 10, 8, 135, 24, 25, 146, 143, 28, 29, 30, 31, //
				128, 129, 130, 131, 132, 133, 23, 27, 136, 137, 138, 139, 140, 5, 6, 7, //
				144, 145, 22, 147, 148, 149, 150, 4, 152, 153, 154, 155, 20, 21, 158, 26, //
				32, 160, 161, 162, 163, 164, 165, 166, 167, 168, 91, 46, 60, 40, 43, 33, //
				38, 169, 170, 171, 172, 173, 174, 175, 176, 177, 93, 92, 42, 41, 59, 94, //
				45, 47, 178, 179, 180, 181, 182, 183, 184, 185, 124, 44, 37, 95, 62, 63, //
				186, 187, 188, 189, 190, 191, 192, 193, 194, 96, 58, 35, 64, 39, 61, 34, //
				195, 97, 98, 99, 100, 101, 102, 103, 104, 105, 196, 197, 198, 199, 200, 201, //
				202, 106, 107, 108, 109, 110, 111, 112, 113, 114, 203, 204, 205, 206, 207, 208, //
				209, 126, 115, 116, 117, 118, 119, 120, 121, 122, 210, 211, 212, 213, 214, 215, //
				216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, //
				123, 65, 66, 67, 68, 69, 70, 71, 72, 73, 232, 233, 234, 235, 236, 237, //
				125, 74, 75, 76, 77, 78, 79, 80, 81, 82, 238, 239, 240, 241, 242, 243, //
				36, 32, 83, 84, 85, 86, 87, 88, 89, 90, 244, 245, 246, 247, 248, 249, //
				48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 250, 251, 252, 253, 254, 255, //
		//
		};
		if (!hex.matches("[A-Za-z0-9]+")) {
			throw new IllegalArgumentException(hex + " not [A-Za-z0-9]+");
		}
		int code = Integer.decode("0x" + hex).intValue();
		if (code > EBCDIK_MAPPING.length - 1) {
			throw new IllegalArgumentException("0x" + hex + " > 255");
		}
		if (code < 0) {
			throw new IllegalArgumentException("0x" + hex + " < 0");
		}
		try {
			return new String(new byte[] { (byte) EBCDIK_MAPPING[code] }, "SJIS").charAt(0);
		} catch (UnsupportedEncodingException e) {
			// SJISコード未サポート例外がスローされることはない
			return 0x00;
		}
	}

	/**
	 * 指定された全角一文字分の'83版KEISの16進コードから文字列に変換します。<br>
	 * @param hex '83版KEISの16進コード(一文字分)
	 * @return 変換後文字列
	 */
	private static String decodeKEIS83Char(String hex) {
		try {
			// 4文字byte文字列でない場合は半角スペースを返却
			if (hex.length() != 4) {
				//throw new IllegalArgumentException("hex.length != 4");
				return " ";
			}

			/*
			 * 上位バイトSJIS変換先コード範囲定義(0x81～0x9F、0xE0～0xEF)
			 */
			final int SJIS_HI_BYTES[] = new int[] { //
			//
					129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, // 0x81～
					145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, // ～0x9F
					224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, // 0xE0～0xEF
			//
			};
			/*
			 * 下位バイトSJIS変換先コード範囲定義(0x40～0x7E、0x80～0xFC)
			 */
			final int SJIS_LO_BYTES[] = new int[] { //
			//
					64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, // 0x40～
					80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, //
					96, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, //
					112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, // ～0x7E 
					128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, // 0x80～
					145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 160, 161, //
					162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, //
					179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, //
					196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, //
					213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, //
					230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, //
					247, 248, 249, 250, 251, 252, // ～0xFC
			//
			};

			/*
			 * 特殊コード対応
			 */
			// 0x4040の場合は全角スペース変換
			if ("4040".equals(hex)) {
				return "　";
			}
			// 0xA1A1の場合は全角スペース変換
			if ("A1A1".equals(hex)) {
				return "　";
			}

			/*
			 * KEIS上位/下位バイト分割
			 */
			int keis_hi_byte = Integer.decode("0x" + hex.substring(0, 2)).intValue();
			int keis_lo_byte = Integer.decode("0x" + hex.substring(2)).intValue();

			/*
			 * 上位/下位バイトの開始位置からのオフセットインデックス算出
			 * (KEIS上位/下位バイト範囲:A1(161)～FE(254) -> 00(0)～5D(93))
			 */
			int index_hb = keis_hi_byte - 161;
			int index_lb = keis_lo_byte - 161;
			int index = (index_hb * 94) + index_lb;

			// 1byte目変換
			int rhb = 0;
			//int ihb = keis_hi_byte - 161;
			rhb += SJIS_HI_BYTES[index / SJIS_LO_BYTES.length];

			// 2byte目変換
			int rlb = 0;
			rlb += SJIS_LO_BYTES[index - ((index / SJIS_LO_BYTES.length) * SJIS_LO_BYTES.length)];

			// 文字列変換
			byte[] bs = new byte[2];
			bs[0] = (byte) rhb;
			bs[1] = (byte) rlb;
			String value = "";
			try {
				value = new String(bs, "SJIS");
			} catch (UnsupportedEncodingException e) {
				// SJISコード未サポート例外がスローされることはない
				value = "";
			}

			return value;
		} catch (Throwable e) {
			throw new IllegalArgumentException(hex);
		}
	}

	/**
	 * 指定された文字列をnative2asciiエンコード後の文字列として提供します。<br>
	 * @param value エンコード対象文字列
	 * @return native2asciiエンコード後の文字列
	 */
	public static String nativeToAscii(String value) {
		return Native2AsciiUtil.native2ascii(value);
	}

	/**
	 * 指定されたnative2asciiエンコード文字列を通常の文字列に変換します。<br>
	 * @param value native2asciiエンコード文字列
	 * @return 通常文字列
	 */
	public static String asciiToNative(String value) {
		try {
			return Native2AsciiUtil.ascii2native(value);
		} catch (IOException e) {
			throw new IllegalArgumentException("value");
		}
	}

	/**
	 * 指定されたbyteを16進数で提供します。<br>
	 * @param b 16進数編集対象byte
	 * @return 16進数編集をしたbyte
	 */
	public static String toHex(byte b) {
		int lbs = b & 0xf;
		int sbs = (b & 0xf0) >> 4;
		return HEX_STRINGS[sbs] + HEX_STRINGS[lbs];
	}

	/**
	 * 指定されたbyte配列を16進数で提供します。<br>
	 * @param b 16進数編集対象byte配列
	 * @return 16進数編集をしたbyte配列
	 */
	public static String toHex(byte[] b) {
		String result = EMPTY;
		for (int i = 0; i < b.length; i++) {
			result += toHex(b[i]);
		}
		return result;
	}

	/**
	 * 指定された文字列のMD5ハッシュコードを提供します。<br>
	 * @param value MD5ハッシュコード算出対象文字列
	 * @return 指定された文字列のハッシュコード
	 */
	public static String toMD5Hash(String value) {
		MessageDigest md = null;

		try {
			md = MessageDigest.getInstance("MD5");
		} catch (NoSuchAlgorithmException e) {
			//e.printStackTrace();
			return EMPTY;
		}

		md.update(value == null ? EMPTY.getBytes() : value.getBytes());

		return toHex(md.digest());
	}

	/**
	 * 指定されたBASE64エンコードバイトデータの元のバイトデータ値を提供します。<br>
	 * @param data BASE64エンコードバイトデータ
	 * @return 元のバイトデータ値
	 */
	private static byte base64ToNative(byte data) {
		final byte[] BASE64_CHARS_BYTES = BASE64_CHARS.getBytes();
		if (data == '=') {
			return 0;
		}
		for (int j = 0; j < BASE64_CHARS_BYTES.length; j++) {
			if (data == BASE64_CHARS_BYTES[j]) {
				return (byte) j;
			}
		}
		return 0;
	}

	/**
	 * 指定されたBASE64エンコード文字列を元のバイトデータに変換して提供します。<br>
	 * @param value デコード対象BASE64文字列
	 * @param start デコード対象とする開始バイトインデックス
	 * @param length デコード対象のバイト長
	 * @return デコード後のバイトデータ
	 */
	public static byte[] base64ToNative(String value, int start, int length) {
		byte[] bytes = value.getBytes();
		byte[] buffer = new byte[length];
		int j = 0;
		for (int i = start; i < start + length; i += 4) {
			buffer[j] = (byte) ((base64ToNative(bytes[i]) << 2) | ((base64ToNative(bytes[i + 1]) & 0x30) >>> 4));
			if (bytes[i + 2] == (byte) '=') {
				j++;
				break;
			}
			buffer[j + 1] = (byte) (((base64ToNative(bytes[i + 1]) & 0x0f) << 4) | ((base64ToNative(bytes[i + 2]) & 0x3c) >>> 2));
			if (bytes[i + 3] == (byte) '=') {
				j += 2;
				break;
			}
			buffer[j + 2] = (byte) (((base64ToNative(bytes[i + 2]) & 0x03) << 6) | (base64ToNative(bytes[i + 3]) & 0x3f));
			j += 3;
		}

		byte[] result = new byte[j];
		System.arraycopy(buffer, 0, result, 0, j);

		return result;
	}

	/**
	 * 指定されたBASE64エンコード文字列を元のバイトデータに変換して提供します。<br>
	 * @param value デコード対象BASE64文字列
	 * @return デコード後のバイトデータ
	 */
	public static byte[] base64ToNative(String value) {
		return base64ToNative(value, 0, value.getBytes().length);
	}

	/**
	 * 指定されたバイトデータをBASE64エンコード文字列に変換して提供します。<br>
	 * @param bytes バイトデータ
	 * @param start 変換対象のバイトインデックス
	 * @param length 変換を行うバイト長
	 * @return BASE64エンコード文字列
	 */
	public static String nativeToBase64(byte[] bytes, int start, int length) {
		final byte[] BASE64_CHARS_BYTES = BASE64_CHARS.getBytes();
		byte[] buffer = new byte[length * 2];
		int i, j, k;

		int index = (length / 3) * 3 + start;
		i = 0;
		for (j = start; j < index; j += 3) {
			k = (bytes[j] >>> 2) & 0x3f;
			buffer[i++] = BASE64_CHARS_BYTES[k];
			k = (bytes[j] & 0x03) << 4 | (bytes[j + 1] >>> 4) & 0x0f;
			buffer[i++] = BASE64_CHARS_BYTES[k];
			k = (bytes[j + 1] & 0x0f) << 2 | (bytes[j + 2] >>> 6) & 0x03;
			buffer[i++] = BASE64_CHARS_BYTES[k];
			k = bytes[j + 2] & 0x3f;
			buffer[i++] = BASE64_CHARS_BYTES[k];
		}

		index = (start + length) - index;
		if (index == 1) {
			k = (bytes[j] >>> 2) & 0x3f;
			buffer[i++] = BASE64_CHARS_BYTES[k];
			k = ((bytes[j] & 0x03) << 4) & 0x3f;
			buffer[i++] = BASE64_CHARS_BYTES[k];
			buffer[i++] = (byte) '=';
			buffer[i++] = (byte) '=';
		} else if (index == 2) {
			k = (bytes[j] >>> 2) & 0x3f;
			buffer[i++] = BASE64_CHARS_BYTES[k];
			k = (bytes[j] & 0x03) << 4 | (bytes[j + 1] >>> 4) & 0x0f;
			buffer[i++] = BASE64_CHARS_BYTES[k];
			k = ((bytes[j + 1] & 0x0f) << 2) & 0x3f;
			buffer[i++] = BASE64_CHARS_BYTES[k];
			buffer[i++] = (byte) '=';
		}

		byte[] result = new byte[i];
		System.arraycopy(buffer, 0, result, 0, i);

		return new String(result);
	}

	/**
	 * 指定されたバイトデータをBASE64エンコード文字列に変換して提供します。<br>
	 * @param value エンコード対象文字列
	 * @return BASE64エンコード文字列
	 */
	public static String nativeToBase64(String value) {
		byte[] bytes = value.getBytes();
		return nativeToBase64(bytes, 0, bytes.length);
	}

	/**
	 * 指定された通常文字列をHTML形式に合わせた文字列として提供します。<br>
	 * 属性値に利用する文字列を想定したメソッドでスペース、タブ、改行は変換されません。<br>
	 * @param value 変換対象文字列
	 * @return HTML形式に合わせた文字列
	 */
	public static String toHTMLAttribute(String value) {
		String buffer = value;
		buffer = StringUtil.replace(buffer, "&", "&amp;");
		buffer = StringUtil.replace(buffer, "<", "&lt;");
		buffer = StringUtil.replace(buffer, ">", "&gt;");
		buffer = StringUtil.replace(buffer, "\"", "&#034;");
		buffer = StringUtil.replace(buffer, "'", "&#039;");
		return buffer;
	}

	/**
	 * 指定された通常文字列をHTML形式に合わせた文字列として提供します。<br>
	 * ドキュメントで利用する文字列を想定したメソッドでスペース、タブ、改行も変換します。<br>
	 * @param value 変換対象文字列
	 * @return HTML形式に合わせた文字列
	 */
	public static String toHTML(String value) {
		String buffer = value;
		buffer = StringUtil.toHTMLAttribute(value);
		buffer = StringUtil.replace(buffer, " ", "&nbsp;");
		buffer = StringUtil.replace(buffer, "\t", "&nbsp;&nbsp;&nbsp;&nbsp;");
		buffer = StringUtil.replace(buffer, "\n", "<br>");
		return buffer;
	}

	/**
	 * スネークケース(snake_case)から小文字開始のキャメルケース(camelCase)に変換して提供します。<br>
	 * @param value 変換対象文字列
	 * @return 小文字開始のキャメルケース(camelCase)文字列
	 */
	public static String toLowerCamelCase(String value) {
		if (value == null) {
			return null;
		}
		Matcher matcher = Pattern.compile("(_)([a-z0-9])").matcher(value.toLowerCase());
		StringBuffer buffer = new StringBuffer();
		while (matcher.find()) {
			matcher.appendReplacement(buffer, matcher.group(2).toUpperCase());
		}
		matcher.appendTail(buffer);
		return buffer.toString();
	}

	/**
	 * スネークケース(snake_case)から大文字開始のキャメルケース(CamelCase)に変換して提供します。<br>
	 * @param value 変換対象文字列
	 * @return 大文字開始のキャメルケース(CamelCase)文字列
	 */
	public static String toUpperCamelCase(String value) {
		if (value == null) {
			return null;
		}
		String buffer = toLowerCamelCase(value);
		if (buffer.length() == 0) {
			return EMPTY;
		} else if (buffer.length() == 1) {
			return value.toUpperCase();
		}
		return buffer.substring(0, 1).toUpperCase() + buffer.substring(1);
	}

	/**
	 * キャメルケース(camelCase)から小文字のスネークケース(snake_case)に変換します。<br>
	 * @param value 変換対象文字列
	 * @return 小文字のスネークケース(snake_case)文字列
	 */
	public static String toLowerSnakeCase(String value) {
		if (value == null) {
			return null;
		}
		return value.replaceAll("([A-Z0-9]+)([A-Z0-9][a-z0-9])", "$1_$2").replaceAll("([a-z])([A-Z])", "$1_$2").toLowerCase();
	}

	/**
	 * キャメルケース(camelCase)から大文字のスネークケース(SNAKE_CASE)に変換します。<br>
	 * @param value 変換対象文字列
	 * @return 大文字のスネークケース(SNAKE_CASE)文字列
	 */
	public static String toUpperSnakeCase(String value) {
		if (value == null) {
			return null;
		}
		return toLowerSnakeCase(value).toUpperCase();
	}

	/**
	 * 指定されたバイト位置にある文字を取得します。<br>
	 * 該当位置に文字が存在しない場合は\0を返却します。
	 * @param value 対象文字列
	 * @param index バイトインデックス
	 * @return 指定されたバイト位置にある文字
	 */
	public static char charAtb(String value, int index) {
		String result = substrb(value, index, 1);
		return result.length() == 0 ? '\0' : result.toCharArray()[0];
	}

	/**
	 * 指定された16進数表記文字列をバイト配列に変換します。<br>
	 * @param hex 16進数表記文字列
	 * @return バイト配列
	 */
	public static byte[] asByteArrayByHex(String hex) {
		byte[] bytes = new byte[hex.length() / 2];
		for (int index = 0; index < bytes.length; index++) {
			bytes[index] = (byte) Integer.parseInt(hex.substring(index * 2, (index + 1) * 2), 16);
		}
		return bytes;
	}

	/**
	 * 指定された文字が半角文字か判定します。<br>
	 * @param c 判定対象文字
	 * @return 半角文字の場合はtrueを返却
	 */
	public static boolean isAscii(char c) {
		return isByteLengthIn(String.valueOf(c), 1);
	}

	/**
	 * 指定された文字列が全て半角文字で構成されているか判定します。<br>
	 * nullが指定された場合は無条件に半角文字であると判断します。<br>
	 * @param value 判定対象文字列
	 * @return 全て半角文字で構成されている場合、trueを返却
	 */
	public static boolean isAscii(String value) {
		if (isEmpty(value)) {
			return true;
		}
		//return value.length() == lenb(value);
		char[] cs = value.toCharArray();
		for (int i = 0; i <= cs.length - 1; i++) {
			if (!isAscii(cs[i])) {
				return false;
			}
		}
		return true;
	}

	/**
	 * 指定された文字列が数値として扱えるか判定します。<br>
	 * このメソッドは空文字列は不正な数値として判定します。<br>
	 * @param value 判定対象文字列
	 * @return 数字として扱える場合はtrueを返却
	 */
	public static boolean isDigit(String value) {
		try {
			//Double.valueOf(value);
			new BigDecimal(value);
			return true;
		} catch (Exception e) {
			return false;
		}
	}

	/**
	 * 指定された文字列が整数値として扱えるか判定します。<br>
	 * このメソッドは空文字列は不正な数値として判定します。<br>
	 * @param value 判定対象文字列
	 * @return 整数として扱える場合はtrueを返却
	 */
	public static boolean isDigitInteger(String value) {
		try {
			//return Double.valueOf(new BigDecimal(value).longValue()).doubleValue() == new BigDecimal(value).doubleValue();
			return isDigit(value) && value.indexOf(".") < 0;
		} catch (Exception e) {
			return false;
		}
	}

	/**
	 * 指定された文字列が許可された数字のみで構成された文字列か判定します。<br>
	 * 当メソッドは単純に文字列が数字のみであるかの判定を行い、数値として扱えるかは判断しません。<br>
	 * 即ち"0.1"のような数字文字以外を含む数値はfalseを返却します。<br>
	 * 数値として扱えるかの反映を行う場合は{@link #isDigit(String)}を使用して下さい。<br>
	 * @param value 判定対象文字列
	 * @return 文字列が数字のみで構成された文字列の場合はtrueを返却
	 */
	public static boolean isNumeric(String value) {
		if (isEmpty(value)) {
			return true;
		} else {
			for (int i = 0; i <= value.length() - 1; i++) {
				char c = value.charAt(i);
				if (!(c >= '0' && c <= '9')) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * 指定された文字列が許可された数字のみで構成された文字列か判定します。<br>
	 * 当メソッドは単純に文字列が数字のみであるかの判定を行い、数値として扱えるかは判断しません。<br>
	 * 即ち"0.1"のような数字文字以外を含む数値はfalseを返却します。<br>
	 * 数値として扱えるかの反映を行う場合は{@link #isDigit(String)}を使用して下さい。<br>
	 * @param value 判定対象文字列
	 * @param exclude 許可文字配列
	 * @return 文字列が数字のみで構成された文字列の場合はtrueを返却
	 */
	public static boolean isNumeric(String value, char[] exclude) {
		if (isEmpty(value)) {
			return true;
		} else {
			for (int i = 0; i <= value.length() - 1; i++) {
				char c = value.charAt(i);
				if (!(c >= '0' && c <= '9')) {
					boolean skip = false;
					for (int j = 0; j <= exclude.length - 1; j++) {
						if (c == exclude[j]) {
							skip = true;
						}
					}
					if (!skip) {
						return false;
					}
				}
			}
		}
		return true;
	}

	/**
	 * 指定された文字列が許可された英字のみで構成された文字列か判定します。<br>
	 * @param value 判定対象文字列
	 * @return 文字列が英字のみで構成された文字列の場合はtrueを返却
	 */
	public static boolean isAlphabet(String value) {
		if (isEmpty(value)) {
			return true;
		} else {
			for (int i = 0; i <= value.length() - 1; i++) {
				char c = value.charAt(i);
				if (!((c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z'))) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * 指定された文字列が許可された英字のみで構成された文字列か判定します。<br>
	 * @param value 判定対象文字列
	 * @param exclude 許可文字配列
	 * @return 文字列が英字のみで構成された文字列の場合はtrueを返却
	 */
	public static boolean isAlphabet(String value, char[] exclude) {
		if (isEmpty(value)) {
			return true;
		} else {
			for (int i = 0; i <= value.length() - 1; i++) {
				char c = value.charAt(i);
				if (!((c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z'))) {
					boolean skip = false;
					for (int j = 0; j <= exclude.length - 1; j++) {
						if (c == exclude[j]) {
							skip = true;
						}
					}
					if (!skip) {
						return false;
					}
				}
			}
		}
		return true;
	}

	/**
	 * 指定された文字列が許可された英数字のみで構成された文字列か判定します。<br>
	 * @param value 判定対象文字列
	 * @return 文字列が英数字のみで構成された文字列の場合はtrueを返却
	 */
	public static boolean isAlphaNumeric(String value) {
		if (isEmpty(value)) {
			return true;
		} else {
			for (int i = 0; i <= value.length() - 1; i++) {
				char c = value.charAt(i);
				if (!((c >= '0' && c <= '9') || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z'))) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * 指定された文字列が許可された英数字のみで構成された文字列か判定します。<br>
	 * @param value 判定対象文字列
	 * @param exclude 許可文字配列
	 * @return 文字列が英数字のみで構成された文字列の場合はtrueを返却
	 */
	public static boolean isAlphaNumeric(String value, char[] exclude) {
		if (isEmpty(value)) {
			return true;
		} else {
			for (int i = 0; i <= value.length() - 1; i++) {
				char c = value.charAt(i);
				if (!((c >= '0' && c <= '9') || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z'))) {
					boolean skip = false;
					for (int j = 0; j <= exclude.length - 1; j++) {
						if (c == exclude[j]) {
							skip = true;
						}
					}
					if (!skip) {
						return false;
					}
				}
			}
		}
		return true;
	}

	/**
	 * 指定された文字列が期待する文字のみで構成された文字列であるか判定します。<br>
	 * @param value 判定対象文字列
	 * @param chars 期待文字配列
	 * @return 文字列が期待する文字のみで構成された文字列である場合にtrueを返却
	 */
	public static boolean isValidCharacers(String value, char[] chars) {
		if (isEmpty(value)) {
			return true;
		} else {
			for (int i = 0; i <= value.length() - 1; i++) {
				char c = value.charAt(i);
				boolean valid = false;
				for (int j = 0; j <= chars.length - 1; j++) {
					if (c == chars[j]) {
						valid = true;
					}
				}
				if (!valid) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * 指定された文字列が指定されたバイト長範囲内にあるか判定します。<br>
	 * @param value 判定対象文字列
	 * @param length チェック範囲長(byte長)
	 * @param encode エンコード方式
	 * @return 文字列が指定されたバイト長範囲内に収まっている場合はtrueを返却
	 */
	public static boolean isByteLengthIn(String value, int length, String encode) {
		if (isEmpty(value)) {
			return true;
		} else {
			return lenb(value, encode) > length ? false : true;
		}
	}

	/**
	 * 指定された文字列が指定されたバイト長範囲内にあるか判定します。<br>
	 * @param value 判定対象文字列
	 * @param length チェック範囲長(byte長)
	 * @return 文字列が指定されたバイト長範囲内に収まっている場合はtrueを返却
	 */
	public static boolean isByteLengthIn(String value, int length) {
		return isByteLengthIn(value, length, null);
	}

	/**
	 * 指定された文字列が指定された文字長(文字数)内にあるか判定します。<br>
	 * @param value 判定対象文字列
	 * @param length チェック範囲長(文字数)
	 * @return 文字列が指定された文字長(文字数)内に収まっている場合はtrueを返却
	 */
	public static boolean isLengthIn(String value, int length) {
		if (isEmpty(value)) {
			return true;
		} else {
			return len(value) > length ? false : true;
		}
	}
}
