package org.phosphoresce.lib.commons.util;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

/**
 * リフレクションユーティリティ<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2012/10/02	Kitagawa		新規作成
 * 2013/07/10	Kitagawa		getInterfaces(Class<?>)追加
 *-->
 */
public class ReflectionUtil {

	/**
	 * コンストラクタ<br>
	 */
	private ReflectionUtil() {
		super();
	}

	/**
	 * 指定されたクラスインタフェースを親クラスまで再帰的に検索して取得します。<br>
	 * @param target ターゲットクラス
	 * @return インタフェースリスト
	 */
	public static List<Class<?>> getInterfaces(Class<?> target) {
		List<Class<?>> list = new LinkedList<Class<?>>();
		for (Class<?> clazz = target; clazz != Object.class; clazz = clazz.getSuperclass()) {
			try {
				list.addAll(Arrays.asList(clazz.getInterfaces()));
			} catch (SecurityException ex) {
			}
		}
		return list;
	}

	/**
	 * 指定されたクラスフィールドを親クラスまで再帰的に検索して取得します。<br>
	 * @param target ターゲットクラス
	 * @param name フィールド名
	 * @return フィールドオブジェクト
	 */
	public static Field getField(Class<?> target, String name) {
		for (Class<?> clazz = target; clazz != Object.class; clazz = clazz.getSuperclass()) {
			try {
				Field field = clazz.getDeclaredField(name);
				field.setAccessible(true);
				return field;
			} catch (NoSuchFieldException ex) {
			} catch (SecurityException ex) {
			}
		}
		return null;
	}

	/**
	 * 指定されたクラスの親クラスを含めたフィールドを取得します。<br>
	 * @param target 対象クラス
	 * @return フィールド
	 */
	public static List<Field> getFields(Class<?> target) {
		List<Field> list = new LinkedList<Field>();
		for (Class<?> clazz = target; clazz != Object.class; clazz = clazz.getSuperclass()) {
			try {
				list.addAll(Arrays.asList(clazz.getDeclaredFields()));
			} catch (SecurityException ex) {
			}
		}
		return list;
	}

	/**
	 * 指定されたクラスメソッドを親クラスまで再帰的に検索して取得します。<br>
	 * @param target ターゲットクラス
	 * @param name メソッド名
	 * @param parameterTypes パラメータクラス
	 * @return メソッドオブジェクト
	 */
	public static Method getMethod(Class<?> target, String name, Class<?>... parameterTypes) {
		try {
			Method method = target.getDeclaredMethod(name, parameterTypes);
			return method;
		} catch (NoSuchMethodException ex) {
		} catch (SecurityException ex) {
		}
		for (Class<?> clazz = target; clazz != Object.class; clazz = clazz.getSuperclass()) {
			try {
				//Method method = clazz.getDeclaredMethod(name, parameterTypes);
				//method.setAccessible(true);
				//return method;
				for (Method method : clazz.getDeclaredMethods()) {
					method.setAccessible(true);
					if (!name.equals(method.getName())) {
						continue;
					}
					if (Modifier.isAbstract(method.getModifiers())) {
						continue;
					}
					if (parameterTypes == null && method.getParameterTypes() != null) {
						continue;
					}
					if (parameterTypes != null && method.getParameterTypes() == null) {
						continue;
					}
					if (parameterTypes == null && method.getParameterTypes() == null) {
						return method;
					}
					if (parameterTypes.length != method.getParameterTypes().length) {
						continue;
					}
					for (int i = 0; i <= method.getParameterTypes().length - 1; i++) {
						if (!method.getParameterTypes()[i].isInstance(parameterTypes[i])) {
							continue;
						}
					}
					return method;
				}
				continue;
				//} catch (NoSuchMethodException ex) {
			} catch (SecurityException ex) {
			}
		}
		return null;
	}

	/**
	 * 指定されたクラスの親クラスを含めたメソッドを取得します。<br>
	 * @param target 対象クラス
	 * @return メソッド
	 */
	public static List<Method> getMethods(Class<?> target) {
		List<Method> list = new LinkedList<Method>();
		for (Class<?> clazz = target; clazz != Object.class; clazz = clazz.getSuperclass()) {
			try {
				list.addAll(Arrays.asList(clazz.getDeclaredMethods()));
			} catch (SecurityException ex) {
			}
		}
		return list;
	}
}
