package org.phosphoresce.lib.commons.util;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.dom4j.DocumentException;
import org.dom4j.io.DOMReader;
import org.dom4j.io.DOMWriter;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.XMLWriter;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * XMLドキュメントオブジェクト操作ユーティリティクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/03	Kitagawa		新規作成
 *-->
 */
public class DOM4JUtil {

	/**
	 * コンストラクタ<br>
	 */
	private DOM4JUtil() {
		super();
	}

	/**
	 * XMLソースをW3Cドキュメントオブジェクトとして提供します。<br>
	 * @param source XMLソース文字列
	 * @return W3Cドキュメントオブジェクト
	 * @throws ParserConfigurationException SAXパーサーの初期化に失敗した場合にスローされます
	 * @throws SAXException 指定されたXMLソースのパースに失敗した場合にスローされます
	 * @throws IOException XMLソースの読み込みに失敗した場合にスローされます
	 */
	public static org.w3c.dom.Document toW3CDocument(String source) throws ParserConfigurationException, SAXException, IOException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setValidating(false); // default 'false'
		factory.setNamespaceAware(true); // default 'false'
		factory.setIgnoringElementContentWhitespace(false); // default 'false'
		factory.setExpandEntityReferences(true); // default 'true'
		factory.setIgnoringComments(false); // default 'false'
		factory.setCoalescing(false); // default 'false'

		DocumentBuilder builder = factory.newDocumentBuilder();

		return builder.parse(new InputSource(new StringReader(source)));
	}

	/**
	 * XMLソースをDOM4Jドキュメントオブジェクトとして提供します。<br>
	 * @param source XMLソース文字列
	 * @return DOM4Jドキュメントオブジェクト
	 * @throws ParserConfigurationException SAXパーサーの初期化に失敗した場合にスローされます
	 * @throws SAXException 指定されたXMLソースのパースに失敗した場合にスローされます
	 * @throws IOException XMLソースの読み込みに失敗した場合にスローされます
	 */
	public static org.dom4j.Document toDOM4JDocument(String source) throws ParserConfigurationException, SAXException, IOException {
		return convertDOM4J(toW3CDocument(source));
	}

	/**
	 * W3CドキュメントオブジェクトしてXMLファイルを読み込みます。<br>
	 * @param xmlFile XMLファイルパス
	 * @return W3Cドキュメントオブジェクト
	 * @throws ParserConfigurationException SAXパーサーの初期化に失敗した場合にスローされます
	 * @throws SAXException 指定されたXMLファイルのパースに失敗した場合にスローされます
	 * @throws IOException XMLファイルの読み込みに失敗した場合にスローされます
	 */
	public static org.w3c.dom.Document loadByW3C(String xmlFile) throws ParserConfigurationException, SAXException, IOException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setValidating(false); // default 'false'
		factory.setNamespaceAware(true); // default 'false'
		factory.setIgnoringElementContentWhitespace(false); // default 'false'
		factory.setExpandEntityReferences(true); // default 'true'
		factory.setIgnoringComments(false); // default 'false'
		factory.setCoalescing(false); // default 'false'

		DocumentBuilder builder = factory.newDocumentBuilder();

		return builder.parse(ResourceUtil.getInputStream(xmlFile));
	}

	/**
	 * DOM4JドキュメントオブジェクトとしてXMLファイルを読み込みます。<br>
	 * @param xmlFile XMLファイルパス
	 * @return DOM4Jドキュメントオブジェクト
	 * @throws ParserConfigurationException SAXパーサーの初期化に失敗した場合にスローされます
	 * @throws SAXException 指定されたXMLファイルのパースに失敗した場合にスローされます
	 * @throws IOException XMLファイルの読み込みに失敗した場合にスローされます
	 */
	public static org.dom4j.Document loadByDOM4J(String xmlFile) throws ParserConfigurationException, SAXException, IOException {
		return convertDOM4J(loadByW3C(xmlFile));
	}

	/**
	 * 指定されたDOM4JドキュメントをW3Cドキュメントに変換します。<br>
	 * @param document DOM4Jドキュメントオブジェクト
	 * @return W3Cドキュメントオブジェクト 
	 * @throws DocumentException ドキュメントオブジェクトの変換に失敗した場合にスローされます
	 */
	public static org.w3c.dom.Document convertW3C(org.dom4j.Document document) throws DocumentException {
		DOMWriter writer = new DOMWriter();
		return writer.write(document);
	}

	/**
	 * 指定されたW3CドキュメントをDOM4Jドキュメントに変換します。<br>
	 * @param document W3Cドキュメントオブジェクト
	 * @return DOM4Jドキュメントオブジェクト 
	 */
	public static org.dom4j.Document convertDOM4J(org.w3c.dom.Document document) {
		DOMReader reader = new DOMReader();
		org.dom4j.Document dom4jDoc = reader.read(document);
		dom4jDoc.setXMLEncoding(document.getXmlEncoding()); // DOM4jパース後encodingが消えてしまう為
		return dom4jDoc;
	}

	/**
	 * 指定されたドキュメントオブジェクトをXML文字列として提供します。<br>
	 * @param document ドキュメントオブジェクト
	 * @param charset キャラクタセット
	 * @return XML文字列
	 * @throws IOException XMLリソースの出力時に入出力例外が発生した場合にスローされます
	 */
	public static String toXMLString(org.dom4j.Document document, String charset) throws IOException {
		if (charset == null) {
			charset = document.getXMLEncoding();
		}
		if (charset == null) {
			charset = "UTF-8";
		}
		StringWriter writer = new StringWriter();
		new XMLWriter(writer, new OutputFormat("  ", true, charset)).write(document);
		return writer.toString().replaceAll("\n[ ]*\n", "\n");
	}

	/**
	 * 指定されたドキュメントオブジェクトをXML文字列として提供します。<br>
	 * @param document ドキュメントオブジェクト
	 * @return XML文字列
	 * @throws IOException XMLリソースの出力時に入出力例外が発生した場合にスローされます
	 */
	public static String toXMLString(org.dom4j.Document document) throws IOException {
		return toXMLString(document, null);
	}

	/**
	 * 指定されたドキュメントオブジェクトをXML文字列として提供します。<br>
	 * @param document ドキュメントオブジェクト
	 * @return XML文字列
	 * @throws IOException XMLリソースの出力時に入出力例外が発生した場合にスローされます
	 */
	public static String toXMLString(org.w3c.dom.Document document) throws IOException {
		return toXMLString(convertDOM4J(document), null);
	}

	/**
	 * 指定されたパスのエレメントを取得します。<br>
	 * @param document ドキュメントオブジェクト
	 * @param path 検索パス
	 * @return 検索対象エレメント
	 */
	public static org.dom4j.Element findElement(org.dom4j.Document document, String path) {
		if (document == null || path == null || path.length() == 0) {
			return null;
		}
		org.dom4j.Element root = document.getRootElement();
		if (root == null) {
			return null;
		}
		org.dom4j.Element element = root;
		for (String name : path.split("/")) {
			element = element.element(name);
			if (element == null) {
				return null;
			}
		}
		return element;
	}

	/**
	 * 指定されたパスのエレメントを取得します。<br>
	 * @param element ドキュメントオブジェクト
	 * @param path 検索パス
	 * @return 検索対象エレメント
	 */
	public static org.dom4j.Element findElement(org.dom4j.Element element, String path) {
		if (element == null || path == null || path.length() == 0) {
			return null;
		}
		org.dom4j.Element target = element;
		for (String name : path.split("/")) {
			target = target.element(name);
			if (target == null) {
				return null;
			}
		}
		return target;
	}

	/**
	 * エレメントに対して名前空間情報を追加します。<br>
	 * @param element エレメントオブジェクト
	 * @param namespaces 名前空間オブジェクト配列
	 */
	public static void addNamespaces(org.dom4j.Element element, org.dom4j.Namespace... namespaces) {
		if (element == null || namespaces == null || namespaces.length == 0) {
			return;
		}
		for (org.dom4j.Namespace namespace : namespaces) {
			if (namespace != null) {
				element.addNamespace(namespace.getPrefix(), namespace.getURI());
			}
		}
	}
}
