package org.phosphoresce.commons.wpoi.dyna;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.phosphoresce.commons.wpoi.PoiGlobal;
import org.phosphoresce.commons.wpoi.adapter.PoiSheet;
import org.phosphoresce.commons.wpoi.adapter.PoiWorkbook;
import org.phosphoresce.commons.wpoi.exception.PoiException;
import org.phosphoresce.commons.wpoi.print.PoiPrintConfig;

/**
 * 動的バインド出力クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2005/12/07	Kitagawa		新規作成
 * 2006/01/18	Kitagawa		シート生成時にシートが未選択となるバグを修正
 * 2006/01/25	Kitagawa		シート生成時に複数シートとしてのExcelファイルとして生成可能に変更(I/F変更)
 * 2012/07/09	Kitagawa		バージョン2にアップデート(PoiDynaSheetから移管)
 *-->
 */
public class PoiDynaExporter implements PoiGlobal {

	/** テンプレートワークブック */
	private PoiWorkbook template;

	/** 出力情報コンテナリスト */
	private List<PoiDynaExportContainer> containers;

	/** テンプレートファイルシート全削除フラグ */
	private boolean baseSheetDeleteOnExport;

	/**
	 * コンストラクタ<br>
	 * @param template テンプレートワークブック
	 */
	public PoiDynaExporter(PoiWorkbook template) {
		super();
		this.template = template;
		this.containers = new LinkedList<PoiDynaExportContainer>();
		this.baseSheetDeleteOnExport = false;
	}

	/**
	 * コンストラクタ<br>
	 * @param template テンプレートワークブック
	 * @param baseSheetDeleteOnExport テンプレートファイルシート全削除フラグ
	 */
	public PoiDynaExporter(PoiWorkbook template, boolean baseSheetDeleteOnExport) {
		super();
		this.template = template;
		this.containers = new LinkedList<PoiDynaExportContainer>();
		this.baseSheetDeleteOnExport = baseSheetDeleteOnExport;
	}

	/**
	 * 動的生成シート情報を追加します。<br>
	 * @param templateSheetIndex テンプレートシートインデックス
	 * @param sheetName 生成対象シート名
	 * @param dynaMap 動的値マップオブジェクト
	 * @param printConfig 印刷環境設定オブジェクト
	 */
	public void addExportSheet(int templateSheetIndex, String sheetName, PoiDynaMap dynaMap, PoiPrintConfig printConfig) {
		if (sheetName == null || sheetName.length() == 0) {
			throw new PoiException("sheetName is empty");
		}
		for (PoiDynaExportContainer container : containers) {
			if (sheetName.equals(container.getSheetName())) {
				throw new PoiException(sheetName + " is duplicate sheet name");
			}
		}
		containers.add(new PoiDynaExportContainer(templateSheetIndex, sheetName, dynaMap, printConfig));
	}

	/**
	 * 動的生成シート情報を追加します。<br>
	 * @param templateSheetIndex テンプレートシートインデックス
	 * @param sheetName 生成対象シート名
	 * @param dynaMap 動的値マップオブジェクト
	 */
	public void addExportSheet(int templateSheetIndex, String sheetName, PoiDynaMap dynaMap) {
		addExportSheet(templateSheetIndex, sheetName, dynaMap, null);
	}

	/**
	 * 動的生成シート情報を削除します。<br>
	 */
	public void clearExportSheet() {
		containers.clear();
	}

	/**
	 * クラスに設定されている動的バインド情報を利用してテンプレートワークブックを基に動的バインド処理を行います。<br>
	 * テンプレートとして設定されたワークブックオブジェクトがディープコピーされて処理が行われる為、元のワークブックへの影響はありません。<br>
	 */
	private PoiWorkbook processDynamicBind() {
		// ワークブックオブジェクトクローン
		PoiWorkbook workbook = template.clone();

		for (PoiDynaExportContainer container : containers) {
			// テンプレートシートコピー
			PoiSheet template = workbook.getSheet(container.getTemplateSheetIndex());
			workbook.getPoiWorkbook().cloneSheet(container.getTemplateSheetIndex());
			workbook.getPoiWorkbook().setActiveSheet(workbook.getPoiWorkbook().getNumberOfSheets() - 1);
			workbook.getPoiWorkbook().setSelectedTab(workbook.getPoiWorkbook().getNumberOfSheets() - 1);
			PoiSheet sheet = workbook.getSheet(workbook.getPoiWorkbook().getNumberOfSheets() - 1);

			// シート名の設定
			sheet.setSheetName(container.getSheetName());

			// 再計算設定
			if (sheet.getPoiSheet() instanceof HSSFSheet) {
				((HSSFSheet) sheet.getPoiSheet()).setForceFormulaRecalculation(true);
			} else {
				// XSSF時
			}

			// 印刷環境設定オブジェクトの適用
			if (container.getPrintConfig() != null) {
				container.getPrintConfig().apply(sheet);
			} else {
				PoiPrintConfig.create(template.getPoiSheet()).apply(sheet);
			}

			//			// 印刷範囲指定
			//			String printArea = workbook.getPoiWorkbook().getPrintArea(template.getSheetIndex());
			//			Name templatePrintArea = workbook.getPoiWorkbook().getName("Print_Area");
			//			if (templatePrintArea != null) {
			//				Name renderPrintArea = workbook.getPoiWorkbook().createName();
			//				renderPrintArea.setSheetIndex(sheet.getSheetIndex());
			//				renderPrintArea.setNameName("Print_Area");
			//				String printAreaFormula = templatePrintArea.getRefersToFormula();
			//				printAreaFormula = StringUtil.replace(printAreaFormula, template.getPoiSheet().getSheetName(), container.getSheetName());
			//				renderPrintArea.setRefersToFormula(printAreaFormula);
			//			}
			//
			//			// タイトル設定
			//			Name templatePrintTitles = workbook.getPoiWorkbook().getName("Print_Titles");
			//			if (templatePrintTitles != null) {
			//				Name renderPrintTitles = workbook.getPoiWorkbook().createName();
			//				renderPrintTitles.setSheetIndex(sheet.getSheetIndex());
			//				renderPrintTitles.setNameName("Print_Titles");
			//				String printTitlesFormula = templatePrintTitles.getRefersToFormula();
			//				printTitlesFormula = StringUtil.replace(printTitlesFormula, template.getPoiSheet().getSheetName(), container.getSheetName());
			//				renderPrintTitles.setRefersToFormula(printTitlesFormula);
			//			}

			// 動的値バインド処理
			new PoiDynaValueSetter(sheet, container.getDynaMap()).execute();
		}

		// ワークブック再計算
		workbook.evaluateAll();

		return workbook;
	}

	/**
	 * バインド処理を行ったワークブックオブジェクトを出力します。<br>
	 * @param afterProcess 動的出力後のワークブック処理オブジェクト
	 */
	public PoiWorkbook export(PoiDynaAfterProcess afterProcess) {
		PoiWorkbook workbook = processDynamicBind();

		if (baseSheetDeleteOnExport) {
			// 動的出力したシート以外を全て削除
			for (int i = 0; i <= template.getPoiWorkbook().getNumberOfSheets() - 1; i++) {
				workbook.getPoiWorkbook().removeSheetAt(0);
			}
		} else {
			// 変換したシートのみを削除(ディフォルト動作仕様)
			List<Integer> templateIndex = new LinkedList<Integer>();
			for (PoiDynaExportContainer container : containers) {
				int index = container.getTemplateSheetIndex();
				if (!templateIndex.contains(index)) {
					templateIndex.add(index);
				}
			}
			Collections.sort(templateIndex);

			int offset = 0;
			for (int index : templateIndex) {
				workbook.getPoiWorkbook().removeSheetAt(index - offset);
				offset++;
			}
		}

		if (afterProcess != null) {
			afterProcess.afterProcess(workbook);
		}

		return workbook;
	}

	/**
	 * バインド処理を行ったワークブックオブジェクトを出力します。<br>
	 */
	public PoiWorkbook export() {
		return export((PoiDynaAfterProcess) null);
	}

	/**
	 * 指定されたストリームオブジェクトに対してワークブックオブジェクトを出力します。<br>
	 * @param stream 出力対象ストリームオブジェクト
	 * @param afterProcess 動的出力後のワークブック処理オブジェクト
	 * @throws IOException 正常に出力処理が行えなかった場合にスローされます
	 */
	public void export(OutputStream stream, PoiDynaAfterProcess afterProcess) throws IOException {
		PoiWorkbook workbook = export(afterProcess);
		workbook.save(stream);
	}

	/**
	 * 指定されたストリームオブジェクトに対してワークブックオブジェクトを出力します。<br>
	 * @param stream 出力対象ストリームオブジェクト
	 * @throws IOException 正常に出力処理が行えなかった場合にスローされます
	 */
	public void export(OutputStream stream) throws IOException {
		export(stream, null);
	}

	/**
	 * 指定されたファイルパスにワークブックオブジェクトを保存します。<br>
	 * @param path ファイルパス
	 * @param afterProcess 動的出力後のワークブック処理オブジェクト
	 * @throws IOException 正常に保存が行えなかった場合にスローされます
	*/
	public void export(String path, PoiDynaAfterProcess afterProcess) throws IOException {
		FileOutputStream stream = null;
		try {
			stream = new FileOutputStream(new File(path));
			export(stream, afterProcess);
		} finally {
			if (stream != null) {
				stream.close();
			}
		}
	}

	/**
	 * 指定されたファイルパスにワークブックオブジェクトを保存します。<br>
	 * @param path ファイルパス
	 * @throws IOException 正常に保存が行えなかった場合にスローされます
	*/
	public void export(String path) throws IOException {
		export(path, null);
	}
}
