package org.phosphoresce.commons.socket.http.io;

import java.io.IOException;
import java.net.Socket;

import org.phosphoresce.commons.socket.core.io.SocketRelayStream;
import org.phosphoresce.commons.socket.core.io.SocketStream;
import org.phosphoresce.commons.socket.http.container.HttpSocketRequest;
import org.phosphoresce.commons.socket.http.container.HttpSocketResponse;

/**
 * ソケットストリーム中継ストリームクラス<br>
 * <br>
 * 当クラスはHTTPプロトコルソケットの中継を行うストリームクラスで、入力、出力を
 * 個別に提供しません。あくまでも入力元、出力先へのデータ転送を請け負うクラスとなります。<br>
 * <br>
 * 当クラスではデータを転送するためのインタフェースとして、汎用的な転送機能のほかに、
 * HTTPプロトコルに特化した転送機能インタフェースを提供します。<br>
 * 例えばChunk形式での動的なデータストリームを転送するインタフェースを提供しています。<br>
 * 
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/30	Kitagawa		新規作成
 *-->
 */
public final class HttpSocketRelayStream extends SocketRelayStream {

	/**
	 * コンストラクタ<br>
	 */
	public HttpSocketRelayStream() {
		super();
	}

	/**
	 * コンストラクタ<br>
	 * @param source 入力元ソケット
	 * @param destination 出力先ソケット
	 * @throws IOException 正常にソケットストリームを生成できなかった場合にスローされます
	 */
	public HttpSocketRelayStream(Socket source, Socket destination) throws IOException {
		super(source, destination);
	}

	/**
	 * コンストラクタ<br>
	 * @param source 入力元ソケットストリーム
	 * @param destination 出力先ソケットストリーム
	 */
	public HttpSocketRelayStream(SocketStream source, SocketStream destination) {
		super(source, destination);
	}

	/**
	 * チャンク形式データ(ヘッダでTransfer-Encoding: chunkedが定義されているデータ)をバッファサイズごとに転送します。<br>
	 * 当メソッドでは転送するチャンクデータ形式は以下を想定しています。<br>
	 * <pre>
	 * [次のバイト数16進表記]\r\n
	 * [....バイトデータ....]
	 * \r\n
	 * [次のバイト数16進表記]\r\n
	 * [....バイトデータ....]
	 * \r\n
	 *   .
	 *   .
	 *   .
	 * [0]\r\n
	 * [フッタ情報]\r\n
	 *   .
	 *   .
	 * \r\n
	 * </pre>
	 * @param bufferSize バッファサイズ
	 * @return 転送したバイト数
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public int forwardChunk(int bufferSize) throws IOException {
		streaming = true;
		try {
			int forwardedSize = 0;
			while (true) {
				String chunkIdentifier = forwardLine();
				String chunkSizeBlock = chunkIdentifier.indexOf(";") >= 0 ? chunkIdentifier.substring(0, chunkIdentifier.indexOf(";")) : chunkIdentifier;
				int chunkSize = Integer.parseInt(chunkSizeBlock.trim(), 16);
				forwardedSize += chunkSize;
				if (chunkSize == 0) {
					break;
				}
				forward(chunkSize, bufferSize);
				forwardLine(); // 復帰行フォワード
			}
			while (true) {
				String footer = forwardLine();
				if (footer.length() == 0) {
					break;
				}
			}
			return forwardedSize;
		} finally {
			streaming = false;
		}
	}

	/**
	 * リクエスト基本情報を指定されたリクエストオブジェクトに読み込みます。<br>
	 * @param request リクエスト基本情報オブジェクト
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public void readRequest(HttpSocketRequest request) throws IOException {
		streaming = true;
		try {
			// リクエスト識別情報取得
			request.setIdentifier(source.readLine());

			// ヘッダ情報取得
			while (true) {
				String header = source.readLine();
				if (header.length() == 0) {
					break;
				}
				request.addHeader(header);
			}
		} finally {
			streaming = false;
		}
	}

	/**
	 * リクエスト基本情報を書き込みます。<br>
	 * @param request リクエスト基本情報オブジェクト
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public void writeRequest(HttpSocketRequest request) throws IOException {
		streaming = true;
		try {
			// リクエスト識別情報送信
			destination.writeLine(request.getIdentifier());

			// ヘッダ情報送信
			String[] headerStrings = request.getHeaderStrings();
			for (int i = 0; i <= headerStrings.length - 1; i++) {
				destination.writeLine(headerStrings[i]);
			}
			destination.writeSeparate();
		} finally {
			streaming = false;
		}
	}

	/**
	 * リクエスト基本情報を転送します。<br>
	 * 尚、当メソッドでは転送処理間でのリクエスト情報の操作は行わずに転送されます。<br>
	 * @return 転送されたリクエスト情報
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public void forwardRequest() throws IOException {
		streaming = true;
		try {
			HttpSocketRequest request = new HttpSocketRequest();
			readRequest(request);
			writeRequest(request);
		} finally {
			streaming = false;
		}
	}

	/**
	 * レスポンス基本情報を指定されたレスポンス情報に読み込みます。<br>
	 * @return レスポンス基本情報オブジェクト
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public void readResponse(HttpSocketResponse response) throws IOException {
		streaming = true;
		try {
			// レスポンス識別情報取得
			response.setIdentifier(source.readLine());

			// ヘッダ情報取得
			while (true) {
				String header = source.readLine();
				if (header.length() == 0) {
					break;
				}
				response.addHeader(header);
			}
		} finally {
			streaming = false;
		}
	}

	/**
	 * レスポンス基本情報を書き込みます。<br>
	 * @param response レスポンス基本情報オブジェクト
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public void writeResponse(HttpSocketResponse response) throws IOException {
		streaming = true;
		try {
			// レスポンス識別情報送信
			destination.writeLine(response.getIdentifier());

			// ヘッダ情報送信
			String[] headerStrings = response.getHeaderStrings();
			for (int i = 0; i <= headerStrings.length - 1; i++) {
				destination.writeLine(headerStrings[i]);
			}
			destination.writeSeparate();
		} finally {
			streaming = false;
		}
	}

	/**
	 * レスポンス基本情報を転送します。<br>
	 * @return 転送されたレスポンス情報
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public void forwardResponse() throws IOException {
		streaming = true;
		try {
			HttpSocketResponse response = new HttpSocketResponse();
			readResponse(response);
			writeResponse(response);
		} finally {
			streaming = false;
		}
	}

	/**
	 * 指定されたヘッダ情報を元にコンテンツボディデータを転送します。<br>
	 * ヘッダ情報からはチャンク等のデータ形式、データ長等の情報を取得します。<br>
	 * @param request ソケットリクエスト情報
	 * @param bufferSize バッファサイズ
	 * @return 転送したバイト数
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public int forwardRequsetContent(HttpSocketRequest request, int bufferSize) throws IOException {
		streaming = true;
		try {
			// コンテンツボディサイズ取得
			int contentLength = request.getHeader("Content-Length").getValueOfInteger();

			int forwardedSize = 0;
			if ("chunked".equalsIgnoreCase(request.getHeader("Transfer-Encoding").getHeaderString())) {
				// チャンク形式データ転送
				forwardedSize = forwardChunk(bufferSize);
			} else {
				if (contentLength == -1) {
					// コンテンツサイズが不明な場合は1バイトずつ転送
					forwardedSize = forwardEOS(bufferSize);
				} else if (contentLength > 0) {
					// コンテンツサイズが特定できている場合はバッファサイズごとに転送
					forwardedSize = forward(contentLength, bufferSize);
				} else {
					// コンテンツサイズが0の場合
				}
			}

			return forwardedSize;
		} finally {
			streaming = false;
		}
	}
}
