package org.phosphoresce.commons.socket.core.io;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.Socket;

import org.phosphoresce.commons.io.ReplacableOutputStream;
import org.phosphoresce.commons.util.ByteFIFOStack;
import org.phosphoresce.commons.util.ByteUtil;

/**
 * ソケットリレーストリームリーダークラス<br>
 * <br>
 * java.net.SocketInputStream、java.net.SocketOutputStreamのインタフェース及び、
 * ソケットからの拡張された、個別の読み込み、書き込み機能を提供します。<br>
 * 当クラスは入力及び、出力のストリーム処理インタフェースを包括的に提供します。<br>
 * <br>
 * 尚、当クラスが提供するインタフェースはあくまでもソケットストリームの処理における
 * 汎用的な処理の機能を提供するのみで、プロトコルに特化したインタフェースは提供しません。<br>
 * 例えば、HTTPプロトコルの特殊なバイト読み込み処理であるChunk形式等のインタフェースは
 * ここでは提供しません。これは、当クラスでは読み込まれたバイトをどのように扱われるかが
 * 不明であるため、それらのデータを必ずバッファリングする必要がありますが、データサイズが
 * メモリで保持できる範囲であることが保証されないためです。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/30	Kitagawa		新規作成
 *-->
 */
public class SocketStream {

	/** ソケット情報行セパレータ **/
	public static final String DEFAULT_SEPARATOR = "\r\n";

	/** ソケットオブジェクト */
	private Socket socket = null;

	/** 入力ストリームオブジェクト */
	private InputStream inputStream = null;

	/** 出力ストリームオブジェクト */
	private OutputStream outputStream = null;

	/** バイトダンプトレース出力フラグ */
	protected static boolean dumpTrace = false;

	/** バイトダンプトレース出力ストリーム */
	protected static PrintStream dumpTraceStream = System.out;

	/**
	 * コンストラクタ<br>
	 */
	private SocketStream() {
		super();
	}

	/**
	 * コンストラクタ<br>
	 * @param socket ソケットオブジェクト
	 * @param match 置換対象文字列
	 * @param replace 置換文字列
	 * @throws IOException クラス初期化時のストリームオブジェクト取得に失敗した場合にスローされます
	 */
	public SocketStream(Socket socket, String match, String replace) throws IOException {
		super();
		this.socket = socket;
		this.inputStream = socket.getInputStream();
		this.outputStream = new ReplacableOutputStream(socket.getOutputStream(), match, replace);

	}

	/**
	 * コンストラクタ<br>
	 * @param socket ソケットオブジェクト
	 * @throws IOException クラス初期化時のストリームオブジェクト取得に失敗した場合にスローされます
	 */
	public SocketStream(Socket socket) throws IOException {
		super();
		this.socket = socket;
		this.inputStream = socket.getInputStream();
		this.outputStream = socket.getOutputStream();
	}

	// Field Accessor Method

	/**
	 * ソケットオブジェクトを取得します。<br>
	 * @return ソケットオブジェクト
	 */
	public Socket getSocket() {
		return socket;
	}

	/**
	 * 入力ストリームオブジェクトを取得します。<br>
	 * @return 入力ストリームオブジェクト
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public InputStream getInputStream() throws IOException {
		return inputStream;
	}

	/**
	 * 出力ストリームオブジェクトを取得します。<br>
	 * @return 出力ストリームオブジェクト
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public OutputStream getOutputStream() throws IOException {
		return outputStream;
	}

	/**
	 * バイトダンプトレース出力フラグを取得します。<br>
	 * @return バイトダンプトレース出力フラグ
	 */
	public static boolean isDumpTrace() {
		return dumpTrace;
	}

	/**
	 * バイトダンプトレース出力フラグを設定します。<br>
	 * @param dumpTrace バイトダンプトレース出力フラグ
	 */
	public static void setDumpTrace(boolean dumpTrace) {
		SocketStream.dumpTrace = dumpTrace;
	}

	/**
	 * バイトダンプトレース出力ストリームを取得します。<br>
	 * @return バイトダンプトレース出力ストリーム
	 */
	public static PrintStream getDumpTraceStream() {
		return dumpTraceStream;
	}

	/**
	 * バイトダンプトレース出力ストリームを設定します。<br>
	 * @param dumpTraceStream バイトダンプトレース出力ストリーム
	 */
	public static void setDumpTraceStream(PrintStream dumpTraceStream) {
		SocketStream.dumpTraceStream = dumpTraceStream;
	}

	// InputStream And OutputStream Common Transfer Method

	/**
	 * 入力及び、出力ストリームを閉じて、そのストリームに関連するすべてのシステムリソースを解放します。<br>
	 * 但し、ソケットオブジェクトのクローズ処理は当メソッドでは行いません。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#close()
	 */
	public void close() throws IOException {
		if (inputStream != null) {
			inputStream.close();
		}
		if (outputStream != null) {
			outputStream.close();
		}
	}

	// InputStream Transfer Method

	/**
	 * 入力ストリームから次のバイトデータを読み込みます。<br>
	 * @return バイトデータ
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#read()
	 */
	public int read() throws IOException {
		return inputStream.read();
	}

	/**
	 * 入力ストリームからlengthバイトまでのデータをバイト配列に読み込みます。<br>
	 * @param bytes 読込先バイト配列
	 * @param offset データが書き込まれる配列のオフセット
	 * @param length 読み込むバイト数
	 * @return バッファに読み込まれたバイトの合計数(ストリームの終わりに達してデータがない場合は-1を返却)
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#read(byte[], int, int)
	 */
	public int read(byte[] bytes, int offset, int length) throws IOException {
		return inputStream.read(bytes, offset, length);
	}

	/**
	 * 入力ストリームからバイト数を読み込み、それをバッファ配列に格納します。<br>
	 * @param bytes 読込先バイト配列
	 * @return バッファに読み込まれたバイトの合計数(ストリームの終わりに達してデータがない場合は-1を返却)
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#read(byte[])
	 */
	public int read(byte[] bytes) throws IOException {
		return inputStream.read(bytes);
	}

	/**
	 * 入力ストリームからのデータをlengthバイトだけスキップしてその範囲のデータを破棄します。<br>
	 * @param length 破棄するバイト数
	 * @return 実際にスキップされたバイト数
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#skip(long)
	 */
	public long skip(long length) throws IOException {
		return inputStream.skip(length);
	}

	/**
	 * 入力ストリームがmarkおよびresetメソッドをサポートしているかどうかを判定します。<br>
	 * @return サポートしている場合にtrueを返却
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#markSupported()
	 */
	public boolean markSupported() throws IOException {
		return inputStream.markSupported();
	}

	/**
	 * 入力ストリームの現在位置にマークを設定します。<br>
	 * @param readlimit マーク位置が無効になる前に読み込み可能なバイトの最大リミット
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#mark(int)
	 */
	public synchronized void mark(int readlimit) throws IOException {
		inputStream.mark(readlimit);
	}

	/**
	 * 入力ストリームの位置を、入力ストリームで最後にmarkメソッドが呼び出されたときのマーク位置に再設定します。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#reset()
	 */
	public synchronized void reset() throws IOException {
		inputStream.reset();
	}

	/**
	 * 入力ストリームのメソッドの次の呼び出し側からブロックされることなく、
	 * 入力ストリームから読み込むことができる(またはスキップできる)バイト数を返します。<br>
	 * @return ブロックしないで入力ストリームから読み込むことができるバイト数
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#available()
	 */
	public int available() throws IOException {
		return inputStream.available();
	}

	// InputStream Extends Method

	/**
	 * 入力ストリームから単一行を読み込みます。<br>
	 * 当メソッドはレコードがブロックされているか否かは判定せず、デリミタが出現するまで
	 * 入力を試みようとします(プロトコル仕様で必ずデリミタが存在する場合等に使用することを想定)。<br>
	 * @param separator 行デリミタ文字列
	 * @return 読み込まれた一行
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public String readLine(String separator) throws IOException {
		ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();
		ByteFIFOStack stack = new ByteFIFOStack(separator.length());
		int b = -1;
		while ((b = inputStream.read()) != -1) {
			byte[] data = stack.push(b);
			byteOutputStream.write(data);
			if (separator.equals(new String(stack.getBytes()))) {
				break;
			}
		}
		String value = new String(byteOutputStream.toByteArray());
		return value;
	}

	/**
	 * 入力ストリームから単一行を読み込みます。<br>
	 * 当メソッドはレコードがブロックされているか否かは判定せず、デリミタが出現するまで
	 * 入力を試みようとします(プロトコル仕様で必ずデリミタが存在する場合等に使用することを想定)。<br>
	 * <b>現在のメソッド仕様では、行のデリミタとなるコードは\nまたは\r\n固定となります。</b><br>
	 * @return 読み込まれた一行
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public String readLine() throws IOException {
		return readLine(DEFAULT_SEPARATOR);
	}

	/**
	 * 入力ストリームから単一行を読み飛ばします。<br>
	 * 当メソッドはレコードがブロックされているか否かは判定せず、デリミタが出現するまで
	 * 入力を試みようとします(プロトコル仕様で必ずデリミタが存在する場合等に使用することを想定)。<br>
	 * @param separator 行デリミタ文字列
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public void skipLine(String separator) throws IOException {
		readLine(separator);
	}

	/**
	 * 入力ストリームから単一行を読み飛ばします。<br>
	 * 当メソッドはレコードがブロックされているか否かは判定せず、デリミタが出現するまで
	 * 入力を試みようとします(プロトコル仕様で必ずデリミタが存在する場合等に使用することを想定)。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public void skipLine() throws IOException {
		readLine(DEFAULT_SEPARATOR);
	}

	/**
	 * 入力ストリームからlengthバイト分のバイトデータを読み込み配列として提供します。<br>
	 * 当メソッドはソケットからの動的なストリームデータを読み込むことを前提としているため、
	 * 内部でバッファリングしながらの読み込みを行いません。なぜなら、ソケットストリームでは
	 * 正確にavaivableでの終了判定が行えない為、ブロックされてしまうことがある為です。
	 * @param length 読み込むバイト数
	 * @return 読み込んだバイト配列
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public byte[] reads(int length) throws IOException {
		if (length <= 0) {
			return new byte[0];
		}

		ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();

		int i = 0;
		int b = -1;

		while ((b = inputStream.read()) != -1) {
			byteOutputStream.write(b);
			i++;
			if (i >= length) {
				break;
			}
		}

		byte[] readed = byteOutputStream.toByteArray();
		if (readed.length == length) {
			return readed;
		} else {
			byte[] result = new byte[length];
			System.arraycopy(readed, 0, result, 0, length);
			return result;
		}
	}

	/**
	 * 入力ストリームから読み込めるバイトデータを全て読み込みます。<br>
	 * 終了判定はストリームのavaivableが提供するブロックされずに読み込めるサイズを条件とします。<br>
	 * 即ち、ストリームがブロックされている状態の箇所で読み込みが完了します。それ以降に再度
	 * ストリームがデータを提供した場合でも、当メソッドは自身で復帰することはありません。<br>
	 * @return 読み込んだバイト配列
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public byte[] reads() throws IOException {
		ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();

		int b = -1;
		while (inputStream.available() > 0 && (b = inputStream.read()) != -1) {
			byteOutputStream.write(b);
		}

		return byteOutputStream.toByteArray();
	}

	// OutputStream Transfer Method

	/**
	 * 出力ストリームに指定されたバイトを書き込みます。<br>
	 * @param b バイトデータ
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.OutputStream#write(int)
	 */
	public void write(int b) throws IOException {
		outputStream.write(b);
		if (dumpTrace && dumpTraceStream != null) {
			dumpTraceStream.println("write 1byte = " + b);
		}
	}

	/**
	 * オフセットoffsetから始まる指定のバイト配列から出力ストリームにlengthバイトを書き込みます。<br>
	 * @param bytes 書き込むバイトデータ配列
	 * @param offset データの開始オフセット
	 * @param length 書き込むバイト数
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.OutputStream#write(byte[], int, int)
	 */
	public void write(byte[] bytes, int offset, int length) throws IOException {
		outputStream.write(bytes, offset, length);
		if (dumpTrace && dumpTraceStream != null) {
			dumpTraceStream.println("write " + bytes.length + "bytes");
			ByteUtil.printByteTrace(bytes, dumpTraceStream);
		}
	}

	/**
	 * 指定のバイト配列を出力ストリームに書き込みます。<br>
	 * @param bytes 書き込むバイトデータ配列
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.OutputStream#write(byte[])
	 */
	public void write(byte[] bytes) throws IOException {
		outputStream.write(bytes);
		if (dumpTrace && dumpTraceStream != null) {
			dumpTraceStream.println("write " + bytes.length + "bytes");
			ByteUtil.printByteTrace(bytes, dumpTraceStream);
		}
	}

	/**
	 * 出力ストリームをフラッシュし、バッファに入っている出力バイトをすべて強制的に書き込みます。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.OutputStream#flush()
	 */
	public void flush() throws IOException {
		outputStream.flush();
	}

	// OutputStream Extends Method

	/**
	 * 指定された文字列を出力ストリームに書き込みます。<br>
	 * <br>
	 * セパレータ文字列を出力文字列末尾に対して付与することと同様の動作を行います。<br>
	 * 他のセパレータ文字列関連メソッドインタフェースとの一連性の為に設けられています。<br>
	 * @param line 書き込む文字列
	 * @param withSeparator 文字列末尾に行セパレータを付与して書き込む場合はtrueを指定
	 * @param separator 行セパレータ文字列
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public void writeLine(String line, boolean withSeparator, String separator) throws IOException {
		outputStream.write(line.getBytes());
		if (dumpTrace && dumpTraceStream != null) {
			dumpTraceStream.println("write " + line.getBytes().length + "bytes");
			ByteUtil.printByteTrace(line.getBytes(), dumpTraceStream);
		}
		if (withSeparator) {
			outputStream.write(separator.getBytes());
		}
		if (dumpTrace && dumpTraceStream != null) {
			dumpTraceStream.println("write " + separator.getBytes().length + "bytes");
			ByteUtil.printByteTrace(separator.getBytes(), dumpTraceStream);
		}
	}

	/**
	 * 指定された文字列を出力ストリームに書き込みます。<br>
	 * @param line 書き込む文字列
	 * @param withSeparator 文字列末尾に行セパレータを付与して書き込む場合はtrueを指定
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public void writeLine(String line, boolean withSeparator) throws IOException {
		writeLine(line, withSeparator, DEFAULT_SEPARATOR);
	}

	/**
	 * 指定された文字列を出力ストリームに書き込みます。<br>
	 * 尚、文字列出力末尾には改行コードが付与されて書き込まれます。<br>
	 * <br>
	 * セパレータ文字列を出力文字列末尾に対して付与することと同様の動作を行います。<br>
	 * 他のセパレータ文字列関連メソッドインタフェースとの一連性の為に設けられています。<br>
	 * @param line 書き込む文字列
	 * @param separator 行セパレータ文字列
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public void writeLine(String line, String separator) throws IOException {
		writeLine(line, true, separator);
	}

	/**
	 * 指定された文字列を出力ストリームに書き込みます。<br>
	 * 尚、文字列出力末尾には改行コードが付与されて書き込まれます。<br>
	 * @param line 書き込む文字列
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public void writeLine(String line) throws IOException {
		writeLine(line, true, DEFAULT_SEPARATOR);
	}

	/**
	 * 出力ストリームに行セパレータを書き込みます。<br>
	 * 通常の文字列出力時には単独での行セパレータ出力は不要となりますが、
	 * データデリミタ識別用に意図的にセパレータのみの出力を行う場合に使用します。<br>
	 * <br>
	 * セパレータ文字列を出力文字列として通常出力することと同様の動作を行います。<br>
	 * 他のセパレータ文字列関連メソッドインタフェースとの一連性の為に設けられています。<br>
	 * @param separator 行セパレータ文字列
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public void writeSeparate(String separator) throws IOException {
		outputStream.write(separator.getBytes());
	}

	/**
	 * 出力ストリームに行セパレータを書き込みます。<br>
	 * 通常の文字列出力時には単独での行セパレータ出力は不要となりますが、
	 * データデリミタ識別用に意図的にセパレータのみの出力を行う場合に使用します。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public void writeSeparate() throws IOException {
		outputStream.write(DEFAULT_SEPARATOR.getBytes());
	}
}
