package org.phosphoresce.commons.util;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

/**
 * 配列に対する操作を行うメソッドを提供するクラス<br>
 *
 * @author Kitagawa<br>
 *
 *<!--
 * 更新日		更新者			更新内容
 * 2005/07/02	Kitagawa		新規作成
 * 2011/05/12	Kitagawa		contains(Object[], Object[])追加
 *-->
 */
public final class ArrayUtil {

	/**
	 * コンストラクタ<br>
	 */
	private ArrayUtil() {
		//
	}

	/**
	 * 指定されたオブジェクト配列をエレメントの追加削除が変更可能な
	 * リストオブジェクトとして取得します。<br>
	 * 但し、Arrays.asListとは異なり、エレメントの変更が配列には反映されません。<br>
	 * また、オブジェクト配列がnullの場合は空のリストオブジェクトとして取得します。<br>
	 * @param objects オブジェクト配列
	 * @return 配列から生成されたリストオブジェクト
	 */
	public static List asResizableList(Object[] objects) {
		return objects == null ? new LinkedList() : new LinkedList(Arrays.asList(objects));
	}

	/**
	 * 指定されたオブジェクト配列に対してオブジェクトを追加します。<br>
	 * 提供されるオブジェクト配列はパラメータのオブジェクトインスタンスとは異なる物となります。<br>
	 * @param objects 追加先オブジェクト配列
	 * @param object 追加対象オブジェクト
	 * @return オブジェクトを追加して新規に作成されたオブジェクト
	 */
	public static Object[] add(Object[] objects, Object object) {
		List list = objects == null ? new LinkedList() : new LinkedList(Arrays.asList(objects));
		list.add(object);
		return list.toArray(new Object[list.size()]);
	}

	/**
	 * 指定されたオブジェクト配列に対してオブジェクトの除去をします。<br>
	 * 提供されるオブジェクト配列はパラメータのオブジェクトインスタンスとは異なる物となります。<br>
	 * @param objects 除去元オブジェクト配列
	 * @param object 除去対象オブジェクト
	 * @return オブジェクトを除去して新規に作成されたオブジェクト
	 */
	public static Object[] remove(Object[] objects, Object object) {
		List list = objects == null ? new LinkedList() : new LinkedList(Arrays.asList(objects));
		list.remove(object);
		return list.toArray(new Object[list.size()]);
	}

	/**
	 * 指定されたオブジェクト配列ににオブジェクトが含まれるか判定します。<br>
	 * @param objects 検索先オブジェクト配列
	 * @param object 検索対象オブジェクト
	 * @return オブジェクトが配列に含まれる場合、trueを返却
	 */
	public static boolean contains(Object[] objects, Object object) {
		List list = objects == null ? new LinkedList() : new LinkedList(Arrays.asList(objects));
		return list.contains(object);
	}

	/**
	 * 指定されたオブジェクト配列ににオブジェクトがオブジェクト配列要素のいずれかが含まれるか判定します。<br>
	 * @param objects 検索先オブジェクト配列
	 * @param object 検索対象オブジェクト要素を含むオブジェクト配列
	 * @return オブジェクト配列要素のいずれかが配列に含まれる場合、trueを返却
	 */
	public static boolean contains(Object[] objects, Object[] targets) {
		for (int i = 0; i <= targets.length - 1; i++) {
			if (contains(objects, targets[i])) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 指定されたStringオブジェクト配列に対してStringオブジェクトを追加します。<br>
	 * 提供されるStringオブジェクト配列はパラメータのオブジェクトインスタンスとは異なる物となります。<br>
	 * @param strings 追加先オブジェクト配列
	 * @param string 追加対象オブジェクト
	 * @return オブジェクトを追加して新規に作成されたオブジェクト
	 */
	public static String[] add(String[] strings, String string) {
		List list = strings == null ? new LinkedList() : new LinkedList(Arrays.asList(strings));
		list.add(string);
		return (String[]) list.toArray(new String[list.size()]);
	}

	/**
	 * 指定されたStringオブジェクト配列に対してStringオブジェクトの除去をします。<br>
	 * 提供されるStringオブジェクト配列はパラメータのオブジェクトインスタンスとは異なる物となります。<br>
	 * @param strings 除去元オブジェクト配列
	 * @param string 除去対象オブジェクト
	 * @return オブジェクトを除去して新規に作成されたオブジェクト
	 */
	public static String[] remove(String[] strings, String string) {
		List list = strings == null ? new LinkedList() : new LinkedList(Arrays.asList(strings));
		list.remove(string);
		return (String[]) list.toArray(new String[list.size()]);
	}
}
