package org.phosphoresce.socket.proxy.http.util;

import java.io.ByteArrayInputStream;
import java.util.Iterator;
import java.util.Map;

import org.phosphoresce.commons.socket.http.container.HttpSocketRequest;
import org.phosphoresce.commons.socket.http.container.HttpSocketResponse;

/**
 * HTTPリバースプロキシコンテンツユーティリティクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/07	Kitagawa		新規作成
 *-->
 */
public final class HttpReverseProxyContentUtil {

	/**
	 * コンストラクタ<br>
	 */
	private HttpReverseProxyContentUtil() {
		super();
	}

	/**
	 * 基底となるリクエスト情報から特定のヘッダ情報を指定されたリクエストにコピーします。<br>
	 * @param source 基底リクエスト情報
	 * @param destination コピー先リクエスト情報
	 * @param headerNames コピー対象ヘッダーキー
	 */
	public static void copyHeaderAttribute(HttpSocketRequest source, HttpSocketRequest destination, String[] headerNames) {
		if (source == null || destination == null || headerNames == null) {
			return;
		}
		for (int i = 0; i <= headerNames.length - 1; i++) {
			String name = headerNames[i];
			if (source.containsHeader(name)) {
				destination.putHeader(name, source.getHeader(name).getValue());
			}
		}
	}

	/**
	 * 基底となるレスポンス情報から特定のヘッダ情報を指定されたレスポンスにコピーします。<br>
	 * @param source 基底レスポンス情報
	 * @param destination コピー先レスポンス情報
	 * @param headerNames コピー対象ヘッダーキー
	 */
	public static void copyHeaderAttribute(HttpSocketResponse source, HttpSocketResponse destination, String[] headerNames) {
		if (source == null || destination == null || headerNames == null) {
			return;
		}
		for (int i = 0; i <= headerNames.length - 1; i++) {
			String name = headerNames[i];
			if (source.containsHeader(name)) {
				destination.putHeader(name, source.getHeader(name).getValue());
			}
		}
	}

	/**
	 * 基底となるリクエスト情報のUser-Agentヘッダ情報を指定されたリクエスト情報にコピーします。<br>
	 * @param source 基底レスポンス情報
	 * @param destination コピー先レスポンス情報
	 */
	public static void copyHeaderUserAgentAttribute(HttpSocketRequest source, HttpSocketRequest destination) {
		copyHeaderAttribute(source, destination, new String[] { "User-Agent" });
	}

	/**
	 * 基底となるリクエスト情報の各種クライアントAcceptヘッダ情報を指定されたリクエスト情報にコピーします。<br>
	 * @param source 基底レスポンス情報
	 * @param destination コピー先レスポンス情報
	 */
	public static void copyHeaderAcceptAttribute(HttpSocketRequest source, HttpSocketRequest destination) {
		copyHeaderAttribute(source, destination, new String[] { "Accept", "Accept-Language", "Accept-Encoding", "Accept-Charset" });
	}

	/**
	 * 指定されたホストとパス宛にパラメータをPOSTするリクエストを生成します。<br>
	 * @param host ホストアドレス
	 * @param path POST先パス
	 * @param parameters パラメータ
	 * @return リクエストオブジェクト
	 */
	public static HttpSocketRequest createPostRequest(String host, String path, Map parameters) {
		HttpSocketRequest request = new HttpSocketRequest();

		// リクエスト基底情報設定
		request.setMethod("POST");
		request.setProtocol("HTTP");
		request.setVersion("1.1");
		request.setPath(path);

		// ヘッダ情報設定
		request.addHeader("Host", host);
		request.addHeader("Content-Type: application/x-www-form-urlencoded");

		// パラメータPOSTコンテンツ生成
		StringBuffer content = new StringBuffer();
		if (parameters != null) {
			for (Iterator iterator = parameters.keySet().iterator(); iterator.hasNext();) {
				String key = (String) iterator.next();
				String value = (String) parameters.get(key);
				content.append(key);
				content.append("=");
				content.append(value);
				if (iterator.hasNext()) {
					content.append("&");
				}
			}
		}
		request.setContentLength(content.toString().getBytes().length);
		request.setContentStream(new ByteArrayInputStream(content.toString().getBytes()));

		return request;
	}

	/**
	 * 指定されたホストとパス宛にパラメータをGETするリクエストを生成します。<br>
	 * @param host ホストアドレス
	 * @param path GET先パス
	 * @param parameters パラメータ
	 * @return リクエストオブジェクト
	 */
	public static HttpSocketRequest createGetRequest(String host, String path, Map parameters) {
		HttpSocketRequest request = new HttpSocketRequest();

		// リクエスト基底情報設定
		request.setMethod("GET");
		request.setProtocol("HTTP");
		request.setVersion("1.1");

		// ヘッダ情報設定
		request.addHeader("Host", host);

		// パラメータPOSTコンテンツ生成
		StringBuffer content = new StringBuffer();
		if (parameters != null) {
			for (Iterator iterator = parameters.keySet().iterator(); iterator.hasNext();) {
				String key = (String) iterator.next();
				String value = (String) parameters.get(key);
				content.append(key);
				content.append("=");
				content.append(value);
				if (iterator.hasNext()) {
					content.append("&");
				}
			}
		}
		request.setContentLength(0);
		if (content.length() > 0) {
			request.setPath(path + "?" + content.toString());
		} else {
			request.setPath(path);
		}

		return request;
	}

	/**
	 * 指定されたホストとパス宛にPOSTするリクエストを生成します。<br>
	 * @param host ホストアドレス
	 * @param path POST先パス
	 * @return リクエストオブジェクト
	 */
	public static HttpSocketRequest createPostRequest(String host, String path) {
		return createPostRequest(host, path, null);
	}

	/**
	 * 指定されたホストとパス宛にGETするリクエストを生成します。<br>
	 * @param host ホストアドレス
	 * @param path GET先パス
	 * @return リクエストオブジェクト
	 */
	public static HttpSocketRequest createGetRequest(String host, String path) {
		return createGetRequest(host, path, null);
	}

	/**
	 * 指定されたホストとパス宛にパラメータをサブミットするリクエストを生成します。<br>
	 * @param host ホストアドレス
	 * @param path リクエスト先パス
	 * @param parameters パラメータ
	 * @param post POSTメソッドでサブミットする場合にtrueを指定、GETメソッドの場合はfalseを指定
	 * @return リクエストオブジェクト
	 */
	public static HttpSocketRequest createSubmitRequest(String host, String path, Map parameters, boolean post) {
		return post ? createPostRequest(host, path, parameters) : createGetRequest(host, path, parameters);
	}
}
