package org.phosphoresce.socket.proxy.http.session;

import java.net.Socket;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.phosphoresce.commons.socket.http.container.HttpSocketRequest;
import org.phosphoresce.commons.socket.http.container.HttpSocketResponse;

/**
 * HTTPリバースプロキシコントロールセッションクラス<br>
 * <br>
 * 当クラスはリバースプロキシ処理時に使用する各種オブジェクトを保持し、
 * コントローラークラス、ハンドラークラス間で共有されます。<br>
 * <br>
 * このセッションオブジェクトは1つのリバースプロキシコントロール実行中に
 * 限り有効なインスタンスで、コントロール処理が完了したタイミングで、
 * 破棄されることを想定したクラスとなっています。<br>
 * サーバー起動中に永続的に情報を保持するための用途で使用するセッションは
 * HttpReverseProxyServerSessionを利用してください。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/18	Kitagawa		新規作成
 *-->
 */
public final class HttpReverseProxyControllerSession {

	/** ロギングオブジェクト */
	private Log log = LogFactory.getLog(getClass());

	/** クライアントソケット */
	private Socket clientSocket;

	/** フォワードソケット */
	private Socket forwardSocket;

	/** リクエスト情報 */
	private HttpSocketRequest request;

	/** レスポンス情報 */
	private HttpSocketResponse response;

	/** 各種データ情報保持マップ */
	private Map data;

	/**
	 * コンストラクタ<br>
	 */
	public HttpReverseProxyControllerSession() {
		super();
		this.data = new HashMap();
	}

	/**
	 * クライアントソケットを取得します。<br>
	 * @return クライアントソケット
	 */
	public Socket getClientSocket() {
		return clientSocket;
	}

	/**
	 * クライアントソケットを設定します。<br>
	 * @param clientSocket クライアントソケット
	 */
	public void setClientSocket(Socket clientSocket) {
		this.clientSocket = clientSocket;
	}

	/**
	 * フォワードソケットを取得します。<br>
	 * @return フォワードソケット
	 */
	public Socket getForwardSocket() {
		return forwardSocket;
	}

	/**
	 * フォワードソケットを設定します。<br>
	 * @param forwardSocket フォワードソケット
	 */
	public void setForwardSocket(Socket forwardSocket) {
		this.forwardSocket = forwardSocket;
	}

	/**
	 * リクエスト情報を取得します。<br>
	 * @return リクエスト情報
	 */
	public HttpSocketRequest getRequest() {
		return request;
	}

	/**
	 * リクエスト情報を設定します。<br>
	 * @param request リクエスト情報
	 */
	public void setRequest(HttpSocketRequest request) {
		this.request = request;
	}

	/**
	 * レスポンス情報を取得します。<br>
	 * @return レスポンス情報
	 */
	public HttpSocketResponse getResponse() {
		return response;
	}

	/**
	 * レスポンス情報を設定します。<br>
	 * @param response レスポンス情報
	 */
	public void setResponse(HttpSocketResponse response) {
		this.response = response;
	}

	/**
	 * 指定されたキーでデータを設定します。<br>
	 * @param name データ保持キー
	 * @param object データオブジェクト
	 */
	public void setData(String name, Object object) {
		data.put(name, object);
	}

	/**
	 * 指定されたキーで保持されているデータを取得します。<br>
	 * @param name 保持キー
	 * @return データオブジェクト
	 */
	public Object getData(String name) {
		return data.get(name);
	}

	/**
	 * 指定されたキーで保持されているデータを削除します。<br>
	 * @param name 保持キー
	 */
	public void removeData(String name) {
		data.remove(name);
	}

	/**
	 * 指定されたキーでデータが保持されているか判定します。<br>
	 * @param name 保持キー
	 */
	public void containsData(String name) {
		data.containsKey(name);
	}

	/**
	 * オブジェクトの文字列表現を返します。<br>
	 * @return オブジェクトの文字列表現
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("{");
		buffer.append("clientSocket=");
		buffer.append(clientSocket);
		buffer.append(",");
		buffer.append("forwardSocket=");
		buffer.append(forwardSocket);
		buffer.append(",");
		buffer.append("request=");
		buffer.append(request);
		buffer.append(",");
		buffer.append("response=");
		buffer.append(response);
		buffer.append(",");
		buffer.append("data=");
		buffer.append(data);
		buffer.append("}");
		return buffer.toString();
	}
}
