package org.phosphoresce.socket.proxy.http.server;

import java.net.ServerSocket;
import java.net.Socket;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.phosphoresce.commons.socket.util.SocketFactory;
import org.phosphoresce.commons.util.CommandResult;
import org.phosphoresce.commons.util.KeytoolUtil;
import org.phosphoresce.socket.proxy.http.HttpReverseProxyGlobal;
import org.phosphoresce.socket.proxy.http.config.HttpReverseProxyConfig;
import org.phosphoresce.socket.proxy.http.daemon.HttpReverseProxyDaemonRegistry;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;

/**
 * HTTPリバースプロキシサーバークラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/08	Kitagawa		新規作成
 *-->
 */
public final class HttpReverseProxyServer {

	/** ロギングオブジェクト */
	private Log log = LogFactory.getLog(getClass());

	/** サーバーソケットオブジェクト */
	private ServerSocket serverSocket;

	/**
	 * コンストラクタ<br>
	 */
	public HttpReverseProxyServer() {
		super();
	}

	/**
	 * HTTPリバースプロキシサーバーを実行します。<br>
	 * 当メソッドは自身のサーバーソケットを起動後、クライアントからの
	 * ソケットリクエストの受信を常に待機し続けます。<br>
	 */
	public void execute() {
		try {
			log.info("Server initializing.");

			// サーバーセッションオブジェクト生成
			HttpReverseProxyServerSession serverSession = new HttpReverseProxyServerSession();

			// サーバーソケットオブジェクト初期化
			initializeServerSocket();

			// ソケット受信処理ループ
			serverSession.setServerSocket(serverSocket);

			log.info("Server initialize succeeded, connection of client is waited for.");

			// サーバーデーモンサービス初期化
			initializeServerDaemonService(serverSession);

			// クライアントソケット受信待ち
			while (true) {
				Socket clientSocket = serverSocket.accept();
				log.trace("Client socket recive " + clientSocket.getInetAddress());

				// クライアントリクエスト処理
				HttpReverseProxyControllerSession controllerSession = new HttpReverseProxyControllerSession();
				controllerSession.setClientSocket(clientSocket);
				HttpReverseProxyController controller = new HttpReverseProxyController(serverSession, controllerSession);
				controller.start();

				//Thread.sleep(HttpReverseProxyGlobal.SERVER_THREAD_DELAY);
			}
		} catch (Throwable e) {
			log.error("Server Fatal error", e);
		}
	}

	/**
	 * サーバーソケットオブジェクトを新規に生成し、初期化します。<br>
	 * @throws Throwable 正常にサーバーソケットオブジェクトを生成できなかった場合にスローされます
	 */
	private void initializeServerSocket() throws Throwable {
		HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();
		if (!config.isServerSSLEnabled()) {
			serverSocket = SocketFactory.createServerSocket(config.getServerPort());
		} else {
			// キーストアファイルが存在しない場合は作成
			if (!KeytoolUtil.isExistKeyStore(config.getServerSSLKeystore())) {
				log.debug("New SSL Keystore create");
				CommandResult result = KeytoolUtil.genkey( //
						config.getServerSSLKeystoreAlias() //
						, config.getServerSSLKeystorePasswd() //
						, config.getServerSSLKeystore() //
						, config.getServerSSLKeystorePasswd() //
						, config.getServerSSLKeystoreCn() //
						, config.getServerSSLKeystoreOu() //
						, config.getServerSSLKeystoreO());
				if (result.hasException()) {
					log.fatal("SSL Keystore create exception", result.getException());
				}
				if (result.hasError()) {
					log.error("SSL Keystore create error [code=" + result.getCode() + "]" + "\n" + result.getError());
				}
				if (result.hasOutput()) {
					log.trace("SSL Keystore created" + "\n" + result.getOutput());
				}
			}
			serverSocket = SocketFactory.createSSLServerSocket(config.getServerPort(), config.getServerSSLKeystore(), config.getServerSSLKeystorePasswd());
		}

		serverSocket.setReceiveBufferSize(config.getSocketBufferSize());
		serverSocket.setReuseAddress(true);
	}

	/**
	 * サーバーデーモンサービスを初期化します。<br>
	 * @param serverSession サーバーセッションオブジェクト
	 */
	private void initializeServerDaemonService(HttpReverseProxyServerSession serverSession) {
		HttpReverseProxyDaemonRegistry.start(serverSession);
	}
}
