package org.phosphoresce.socket.proxy.http.handler;

import java.net.Socket;

import org.phosphoresce.commons.socket.http.container.HttpSocketRequest;
import org.phosphoresce.commons.socket.http.util.HttpSocketClientUtil;
import org.phosphoresce.commons.util.FileUtil;
import org.phosphoresce.socket.proxy.http.config.HttpReverseProxyConfig;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;
import org.phosphoresce.socket.proxy.http.util.HttpReverseProxyVelocityResourceResponseFactory;

/**
 * HTTPリバースプロキシサーバーコンテンツアクセスハンドラークラス<br>
 * <br>
 * 当ハンドラクラスは以下の仕様で動作させることを想定しています。<br>
 * <b>前提</b>
 * ・コントローラーセッションにクライアントソケットオブジェクトが設定されていること<br>
 * ・コントローラーセッションにクライアントリクエスト情報が設定されていること<br>
 * <br>
 * <b>処理</b>
 * ・コンテキストパスがサーバーコンテンツエイリアス以外の場合は処理をスキップ<br>
 * ・サーバーコンテンツをクライアントにレスポンス<br>
 * <br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/17	Kitagawa		新規作成
 *-->
 */
public class ServerContentRequestHandler extends HttpReverseProxyHandlerAdapter {

	/**
	 * コンストラクタ<br>
	 * @param serverSession サーバーセッションオブジェクト
	 * @param controllerSession コントローラーセッションオブジェクト
	 */
	public ServerContentRequestHandler(HttpReverseProxyServerSession serverSession, HttpReverseProxyControllerSession controllerSession) {
		super(serverSession, controllerSession);
	}

	/**
	 * イベントハンドル処理を実行します。<br>
	 * @return イベントハンドル処理後、プロキシコントローラーの後続処理を実施する場合にtrueを返却
	 * @throws Throwable 正常にイベントハンドル処理が行えなかった場合にスローされます
	 * @see org.phosphoresce.socket.proxy.http.handler.HttpReverseProxyHandler#execute()
	 */
	public boolean execute() throws Throwable {
		// 共通フィールド取得
		HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();
		HttpReverseProxyServerSession serverSession = getServerSession();
		HttpReverseProxyControllerSession controllerSession = getControllerSession();
		Socket clientSocket = getClientSocket();
		HttpSocketRequest request = getRequest();

		// コンテキストパスがサーバーコンテンツエイリアス以外の場合は処理をスキップ
		if (!config.getServerContentAlias().equals(request.getContextPath())) {
			return true;
		}

		// サーバーコンテンツをクライアントにレスポンス
		HttpReverseProxyVelocityResourceResponseFactory factory = new HttpReverseProxyVelocityResourceResponseFactory(serverSession, controllerSession);
		factory.setResource(FileUtil.connectPath(new String[] { config.getServerContentRootPath(), config.getServerContentAlias(), request.getResourcePath() }));
		HttpSocketClientUtil.writeResponse(clientSocket.getOutputStream(), factory.create(), config.getSocketBufferSize());

		return false;
	}
}
