package org.phosphoresce.socket.proxy.http.handler;

import java.net.Socket;

import org.phosphoresce.commons.socket.http.HttpResponseStatus;
import org.phosphoresce.commons.socket.http.container.HttpSocketRequest;
import org.phosphoresce.commons.socket.http.util.HttpSocketClientUtil;
import org.phosphoresce.socket.proxy.http.config.HttpReverseProxyConfig;
import org.phosphoresce.socket.proxy.http.config.RequestRedirectConfig;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;
import org.phosphoresce.socket.proxy.http.util.HttpReverseProxyVelocityResourceResponseFactory;

/**
 * HTTPリバースプロキシリクエストリダイレクトハンドラークラス<br>
 * <br>
 * 当ハンドラクラスは以下の仕様で動作させることを想定しています。<br>
 * <b>前提</b>
 * ・コントローラーセッションにクライアントソケットオブジェクトが設定されていること<br>
 * ・コントローラーセッションにクライアントリクエスト情報が設定されていること<br>
 * <br>
 * <b>処理</b>
 * ・リクエストされたパスがリダイレクト定義に設定されたパス以外の場合は処理をスキップ<br>
 * ・リダイレクト定義内容を元にクライアントに対してリダイレクトレスポンスを送信<br>
 * <br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/17	Kitagawa		新規作成
 *-->
 */
public class RequestRedirectingHandler extends HttpReverseProxyHandlerAdapter {

	/**
	 * コンストラクタ<br>
	 * @param serverSession サーバーセッションオブジェクト
	 * @param controllerSession コントローラーセッションオブジェクト
	 */
	public RequestRedirectingHandler(HttpReverseProxyServerSession serverSession, HttpReverseProxyControllerSession controllerSession) {
		super(serverSession, controllerSession);
	}

	/**
	 * イベントハンドル処理を実行します。<br>
	 * @return イベントハンドル処理後、プロキシコントローラーの後続処理を実施する場合にtrueを返却
	 * @throws Throwable 正常にイベントハンドル処理が行えなかった場合にスローされます
	 * @see org.phosphoresce.socket.proxy.http.handler.HttpReverseProxyHandler#execute()
	 */
	public boolean execute() throws Throwable {
		// 共通フィールド取得
		HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();
		HttpReverseProxyServerSession serverSession = getServerSession();
		HttpReverseProxyControllerSession controllerSession = getControllerSession();
		Socket clientSocket = getClientSocket();
		HttpSocketRequest request = getRequest();

		// リダイレクト定義取得
		RequestRedirectConfig redirectConfig = config.getRedirectingConfig(request.getPath());

		// リクエストされたパスがリダイレクト定義に設定されたパス以外の場合は処理をスキップ
		if (redirectConfig == null) {
			log.trace("REDIRECT Skipped " + request.getPath());
			return true;
		}

		// リダイレクト定義内容を元にクライアントに対してリダイレクトレスポンスを送信
		log.debug("REDIRECT " + request.getPath() + " -> " + redirectConfig.getRedirectUrl());
		HttpReverseProxyVelocityResourceResponseFactory factory = new HttpReverseProxyVelocityResourceResponseFactory(serverSession, controllerSession);
		factory.setStatus(HttpResponseStatus.STATUS_302_FOUND);
		factory.addHeader("Location", redirectConfig.getRedirectUrl());
		factory.addHeader("Connection", "close");
		factory.setResource(config.getServerContentResourcePath(config.getServerRedirectContent()));
		factory.putParameter("redirectUrl", redirectConfig.getRedirectUrl());
		HttpSocketClientUtil.writeResponse(clientSocket.getOutputStream(), factory.create(), config.getSocketBufferSize());

		return false;
	}
}
