package org.phosphoresce.socket.proxy.http.handler;

import java.net.ServerSocket;
import java.net.Socket;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.phosphoresce.commons.socket.http.container.HttpSocketRequest;
import org.phosphoresce.commons.socket.http.container.HttpSocketResponse;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;

/**
 * HTTPリバースプロキシサーバー処理実行時のイベントハンドラアダプタクラス<br>
 * <br>
 * 当クラスは各種ハンドラ処理時に情報を共有するためのセッションオブジェクトに対する
 * 委譲インタフェースや、ハンドル処理時の共通インタフェースを提供します。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/18	Kitagawa		新規作成
 *-->
 */
public abstract class HttpReverseProxyHandlerAdapter implements HttpReverseProxyHandler {

	/** ロギングオブジェクト */
	protected final Log log = LogFactory.getLog(getClass());

	/** サーバーセッションオブジェクト */
	private HttpReverseProxyServerSession serverSession;

	/** コントローラーセッションオブジェクト */
	private HttpReverseProxyControllerSession controllerSession;

	/**
	 * コンストラクタ<br>
	 */
	private HttpReverseProxyHandlerAdapter() {
		super();
	}

	/**
	 * コンストラクタ<br>
	 * @param serverSession サーバーセッションオブジェクト
	 * @param controllerSession コントローラーセッションオブジェクト
	 */
	public HttpReverseProxyHandlerAdapter(HttpReverseProxyServerSession serverSession, HttpReverseProxyControllerSession controllerSession) {
		super();
		this.serverSession = serverSession;
		this.controllerSession = controllerSession;
	}

	// Accessor Method

	/**
	 * コントローラーセッションオブジェクトを取得します。<br>
	 * @return コントローラーセッションオブジェクト
	 */
	public final HttpReverseProxyControllerSession getControllerSession() {
		return controllerSession;
	}

	/**
	 * サーバーセッションオブジェクトを取得します。<br>
	 * @return サーバーセッションオブジェクト
	 */
	public HttpReverseProxyServerSession getServerSession() {
		return serverSession;
	}

	// Session Transfer Method

	/**
	 * サーバーソケットを取得します。<br>
	 * @return サーバーソケット
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession#getServerSocket()
	 */
	public final ServerSocket getServerSocket() {
		return serverSession.getServerSocket();
	}

	/**
	 * サーバーソケットを設定します。<br>
	 * @param serverSocket サーバーソケット
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession#setServerSocket(java.net.ServerSocket)
	 */
	public final void setServerSocket(ServerSocket serverSocket) {
		serverSession.setServerSocket(serverSocket);
	}

	/**
	 * クライアントソケットを取得します。<br>
	 * @return クライアントソケット
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession#getClientSocket()
	 */
	public final Socket getClientSocket() {
		return controllerSession.getClientSocket();
	}

	/**
	 * クライアントソケットを設定します。<br>
	 * @param clientSocket クライアントソケット
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession#setClientSocket(java.net.Socket)
	 */
	public final void setClientSocket(Socket clientSocket) {
		controllerSession.setClientSocket(clientSocket);
	}

	/**
	 * フォワードソケットを取得します。<br>
	 * @return フォワードソケット
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession#getForwardSocket()
	 */
	public final Socket getForwardSocket() {
		return controllerSession.getForwardSocket();
	}

	/**
	 * フォワードソケットを設定します。<br>
	 * @param forwardSocket フォワードソケット
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession#setForwardSocket(java.net.Socket)
	 */
	public final void setForwardSocket(Socket forwardSocket) {
		controllerSession.setForwardSocket(forwardSocket);
	}

	/**
	 * リクエスト情報を取得します。<br>
	 * @return リクエスト情報
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession#getRequest()
	 */
	public final HttpSocketRequest getRequest() {
		return controllerSession.getRequest();
	}

	/**
	 * リクエスト情報を設定します。<br>
	 * @param request リクエスト情報
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession#setRequest(org.phosphoresce.commons.socket.http.container.HttpSocketRequest)
	 */
	public final void setRequest(HttpSocketRequest request) {
		controllerSession.setRequest(request);
	}

	/**
	 * レスポンス情報を取得します。<br>
	 * @return レスポンス情報
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession#getResponse()
	 */
	public final HttpSocketResponse getResponse() {
		return controllerSession.getResponse();
	}

	/**
	 * レスポンス情報を設定します。<br>
	 * @param response レスポンス情報
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession#setResponse(org.phosphoresce.commons.socket.http.container.HttpSocketResponse)
	 */
	public final void setResponse(HttpSocketResponse response) {
		controllerSession.setResponse(response);
	}

	/**
	 * 指定されたキーで保持されているコントローラーセッションデータを取得します。<br>
	 * @param name 保持キー
	 * @return セッションデータオブジェクト
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession#getData(java.lang.String)
	 */
	public final Object getControllerSessionData(String name) {
		return controllerSession.getData(name);
	}

	/**
	 * 指定されたキーでコントローラーセッションデータを設定します。<br>
	 * @param name セッションデータ保持キー
	 * @param object セッションデータオブジェクト
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession#setData(java.lang.String, java.lang.Object)
	 */
	public final void setControllerSessionData(String name, Object object) {
		controllerSession.setData(name, object);
	}

	/**
	 * 指定されたキーで保持されているコントローラーセッションデータを削除します。<br>
	 * @param name 保持キー
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession#removeData(java.lang.String)
	 */
	public final void removeControllerSessionData(String name) {
		controllerSession.removeData(name);
	}

	/**
	 * 指定されたキーでコントローラーセッションデータが保持されているか判定します。<br>
	 * @param name 保持キー
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession#containsData(java.lang.String)
	 */
	public final void containsControllerSessionData(String name) {
		controllerSession.containsData(name);
	}

	/**
	 * 指定されたキーで保持されているコントローラーセッションデータを取得します。<br>
	 * @param name 保持キー
	 * @return セッションデータオブジェクト
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession#getData(java.lang.String)
	 */
	public final Object getServerSessionData(String name) {
		return serverSession.getData(name);
	}

	/**
	 * 指定されたキーでコントローラーセッションデータを設定します。<br>
	 * @param name セッションデータ保持キー
	 * @param object セッションデータオブジェクト
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession#setData(java.lang.String, java.lang.Object)
	 */
	public final void setServerSessionData(String name, Object object) {
		serverSession.setData(name, object);
	}

	/**
	 * 指定されたキーで保持されているコントローラーセッションデータを削除します。<br>
	 * @param name 保持キー
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession#removeData(java.lang.String)
	 */
	public final void removeServerSessionData(String name) {
		serverSession.removeData(name);
	}

	/**
	 * 指定されたキーでコントローラーセッションデータが保持されているか判定します。<br>
	 * @param name 保持キー
	 * @see org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession#containsData(java.lang.String)
	 */
	public final void containsServerSessionData(String name) {
		serverSession.containsData(name);
	}
}