package org.phosphoresce.socket.proxy.http.handler;

import java.net.Socket;

import org.phosphoresce.commons.socket.http.container.HttpSocketRequest;
import org.phosphoresce.commons.socket.http.container.HttpSocketResponse;
import org.phosphoresce.commons.socket.http.util.HttpSocketClientUtil;
import org.phosphoresce.commons.socket.http.util.HttpSocketLoggingUtil;
import org.phosphoresce.socket.proxy.http.config.HttpReverseProxyConfig;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyControllerSession;
import org.phosphoresce.socket.proxy.http.session.HttpReverseProxyServerSession;

/**
 * HTTPリバースプロキシクライアントリクエスト転送ハンドラークラス<br>
 * <br>
 * 当ハンドラクラスは以下の仕様で動作させることを想定しています。<br>
 * <b>前提</b>
 * ・コントローラーセッションにクライアントソケットオブジェクトが設定されていること<br>
 * ・コントローラーセッションにクライアントリクエスト情報が設定されていること<br>
 * ・コントローラーセッションに転送先ソケットオブジェクトが設定されていること<br>
 * <br>
 * <b>処理</b>
 * ・転送先ソケットに対してリクエスト情報を送信<br>
 * ・転送先ソケットからのレスポンス情報を受信せいてコントローラーセッションに設定<br>
 * <br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/10	Kitagawa		新規作成
 *-->
 */
public class ForwardClientRequestHandler extends HttpReverseProxyHandlerAdapter {

	/**
	 * コンストラクタ<br>
	 * @param serverSession サーバーセッションオブジェクト
	 * @param controllerSession コントローラーセッションオブジェクト
	 */
	public ForwardClientRequestHandler(HttpReverseProxyServerSession serverSession, HttpReverseProxyControllerSession controllerSession) {
		super(serverSession, controllerSession);
	}

	/**
	 * イベントハンドル処理を実行します。<br>
	 * @return イベントハンドル処理後、プロキシコントローラーの後続処理を実施する場合にtrueを返却
	 * @throws Throwable 正常にイベントハンドル処理が行えなかった場合にスローされます
	 * @see org.phosphoresce.socket.proxy.http.handler.HttpReverseProxyHandler#execute()
	 */
	public boolean execute() throws Throwable {
		// 共通フィールド取得
		HttpReverseProxyConfig config = HttpReverseProxyConfig.instance();
		HttpReverseProxyServerSession serverSession = getServerSession();
		HttpReverseProxyControllerSession controllerSession = getControllerSession();
		Socket clientSocket = getClientSocket();
		Socket forwardSocket = getForwardSocket();
		String clientAddress = getClientSocket().getInetAddress().getHostAddress();
		HttpSocketRequest request = getRequest();

		// 転送先ソケットに対してリクエスト情報を送信
		HttpSocketClientUtil.writeRequest(forwardSocket.getOutputStream(), request, config.getSocketBufferSize());

		// 転送先ソケットからのレスポンス情報を受信せいてコントローラーセッションに設定
		HttpSocketResponse response = HttpSocketClientUtil.readResponse(forwardSocket.getInputStream(), config.getSocketBufferSize());
		if (HttpSocketLoggingUtil.isTraceTarget(request)) {
			log.trace("\n" + HttpSocketLoggingUtil.createSocketResponseTrace(response));
		}
		setResponse(response);

		return true;
	}
}
