package org.phosphoresce.socket.proxy.http.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;

import org.phosphoresce.commons.util.FileUtil;
import org.phosphoresce.commons.util.StringUtil;
import org.phosphoresce.socket.proxy.http.HttpReverseProxySetting;
import org.phosphoresce.socket.proxy.http.daemon.HttpReverseProxyDaemon;
import org.phosphoresce.socket.proxy.http.daemon.HttpReverseProxyDaemonRegistry;
import org.phosphoresce.socket.proxy.http.exception.ConfigureException;

/**
 * HTTPリバースプロキシ環境設定クラス<br>
 * <br>
 * 当クラスはリバースプロキシの各種実行環境設定情報を管理します。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/08	Kitagawa		新規作成
 *-->
 */
public class HttpReverseProxyConfig implements Serializable {

	/** セルフインスタンス */
	private static HttpReverseProxyConfig instance;

	/** プロキシサーバー名 */
	private String serverName;

	/** プロキシサーバーアドレス */
	private String serverAddress;

	/** プロキシサーバーポート */
	private int serverPort;

	/** Log4J環境定義リソース */
	private String serverLog4JConfig;

	/** プロキシサーバーコンテンツ基底パス */
	private String serverContentRootPath;

	/** サーバーコンテンツVelocity環境定義リソース */
	private String serverContentVelocityConfig;

	/** プロキシサーバーコンテンツVelocityテンプレート拡張子 */
	private String serverContentVelocityExtension;

	/** プロキシサーバーコンテンツ文字コード */
	private String serverContentCharset;

	/** プロキシサーバーコンテンツエイリアス */
	private String serverContentAlias;

	/** サーバーコンテンツ(リダイレクトページ) */
	private String serverRedirectContent;

	/** サーバーコンテンツ(マッピング不正ページ) */
	private String serverInvalidMappingContent;

	/** サーバーコンテンツ(サーバー致命的エラーページ) */
	private String serverFatalErrorContent;

	/** サーバーコンテンツ(サーバータイムアウトエラーページ) */
	private String serverTimeoutErrorContent;

	/** 転送定義環境設定ファイル */
	private String serverForwardConfig;

	/** リダイレクト定義環境設定ファイル */
	private String serverRedirectConfig;

	/** ハンドラ定義環境設定ファイル */
	private String serverHandlerConfig;

	/** デーモンサービス環境設定ファイル */
	private String serverDaemonConfig;

	/** プロキシSSL有効フラグ */
	private boolean serverSSLEnabled;

	/** プロキシSSLサーバーキーストアパス */
	private String serverSSLKeystore;

	/** プロキシSSLサーバーキーストアパスワード */
	private String serverSSLKeystorePasswd;

	/** プロキシSSLサーバーキーストアエイリアス */
	private String serverSSLKeystoreAlias;

	/** プロキシSSLサーバーキーストアCN(姓名) */
	private String serverSSLKeystoreCn;

	/** プロキシSSLサーバーキーストアOU(組織単位名) */
	private String serverSSLKeystoreOu;

	/** プロキシSSLサーバーキーストアO(組織名) */
	private String serverSSLKeystoreO;

	/** ソケットストリームバッファサイズ */
	private int socketBufferSize;

	/** ソケットタイムアウト時間 */
	private int socketTimeout;

	/** プロキシ転送環境設定リスト */
	private List forwardingConfigs = new LinkedList();

	/** リダイレクト環境設定リスト */
	private List redirectingConfigs = new LinkedList();

	/** リクエスト受信側ハンドラリスト */
	private List reciveRequestHandlers = new LinkedList();

	/** レスポンス転送側ハンドラリスト */
	private List forwardResponseHandlers = new LinkedList();

	/**
	 * コンストラクタ<br>
	 */
	private HttpReverseProxyConfig() {
		super();
	}

	// Singleton Method

	/**
	 * セルフインスタンスを取得します。<br>
	 * @return セルフインスタンス
	 * @throws IOException Propertiesからのロード時に入出力例外が発生した場合にスローされます
	 */
	public static HttpReverseProxyConfig instance() throws Throwable {
		if (instance == null) {
			instance = new HttpReverseProxyConfig();
			instance.load();
		}
		return instance;
	}

	/**
	 * プロパティリソースから環境定義を読み込みます。<br>
	 * @throws Throwable リソースロード時に入出力例外が発生した場合にスローされます
	 */
	public void load() throws Throwable {
		HttpReverseProxySetting setting = HttpReverseProxySetting.instance();

		// ファイルが存在しない場合はエラー
		String configName = setting.getConfigPath() + "/" + setting.getConfigName();
		if (!new File(configName).exists()) {
			throw new ConfigureException(configName, "Server config resource not found");
		}

		// Propertiesリソース読み込み
		FileInputStream stream = new FileInputStream(configName);
		Properties properties = new Properties();
		properties.load(stream);
		stream.close();

		// フィールド値設定
		this.serverName = properties.getProperty("serverName");
		this.serverAddress = properties.getProperty("serverAddress");
		this.serverPort = Integer.parseInt(properties.getProperty("serverPort"));
		this.serverLog4JConfig = properties.getProperty("serverLog4JConfig");
		this.serverContentRootPath = properties.getProperty("serverContentRootPath");
		this.serverContentVelocityConfig = properties.getProperty("serverContentVelocityConfig");
		this.serverContentVelocityExtension = properties.getProperty("serverContentVelocityExtension");
		this.serverContentCharset = properties.getProperty("serverContentCharset");
		this.serverContentAlias = properties.getProperty("serverContentAlias");
		this.serverRedirectContent = properties.getProperty("serverRedirectContent");
		this.serverInvalidMappingContent = properties.getProperty("serverInvalidMappingContent");
		this.serverFatalErrorContent = properties.getProperty("serverFatalErrorContent");
		this.serverTimeoutErrorContent = properties.getProperty("serverTimeoutErrorContent");
		this.serverForwardConfig = properties.getProperty("serverForwardConfig");
		this.serverRedirectConfig = properties.getProperty("serverRedirectConfig");
		this.serverHandlerConfig = properties.getProperty("serverHandlerConfig");
		this.serverDaemonConfig = properties.getProperty("serverDaemonConfig");
		this.serverSSLEnabled = Boolean.valueOf(properties.getProperty("serverSSLEnabled")).booleanValue();
		this.serverSSLKeystore = properties.getProperty("serverSSLKeystore");
		this.serverSSLKeystorePasswd = properties.getProperty("serverSSLKeystorePasswd");
		this.serverSSLKeystoreAlias = properties.getProperty("serverSSLKeystoreAlias");
		this.serverSSLKeystoreCn = properties.getProperty("serverSSLKeystoreCn");
		this.serverSSLKeystoreOu = properties.getProperty("serverSSLKeystoreOu");
		this.serverSSLKeystoreO = properties.getProperty("serverSSLKeystoreO");
		this.socketBufferSize = Integer.parseInt(properties.getProperty("socketBufferSize"));
		this.socketTimeout = Integer.parseInt(properties.getProperty("socketTimeout"));

		// 転送定義読み込み
		loadForwardConfig();

		// リダイレクト定義読み込み
		loadRedirectConfig();

		// ハンドラ定義読み込み
		loadHandlerConfig();

		// デーモン定義読み込み
		loadDaemonConfig();
	}

	/**
	 * 転送定義環境設定情報を読み込みます。<br>
	 * @throws Throwable リソースロード時に入出力例外が発生した場合にスローされます
	 */
	private void loadForwardConfig() throws Throwable {
		HttpReverseProxySetting setting = HttpReverseProxySetting.instance();

		// ファイルが存在しない場合は処理をスキップ
		String configName = setting.getConfigPath() + "/" + this.serverForwardConfig;
		if (StringUtil.isEmpty(this.serverForwardConfig) || !new File(configName).exists()) {
			//throw new ConfigureException(configName, "Forward Config resource not found");
			return;
		}

		// Propertiesリソース読み込み
		FileInputStream stream = new FileInputStream(configName);
		Properties properties = new Properties();
		properties.load(stream);
		stream.close();

		// 定義読み込みループ処理
		int forwardDefinitionCount = Integer.parseInt(properties.getProperty("forwardDefinitionCount"));
		for (int i = 0; i <= forwardDefinitionCount; i++) {
			String matchUrl = properties.getProperty("forward." + i + ".matchUrl");
			String forwardAddress = properties.getProperty("forward." + i + ".forwardAddress");
			String forwardPort = properties.getProperty("forward." + i + ".forwardPort");
			if ( //
			StringUtil.isEmpty(matchUrl) || //
					StringUtil.isEmpty(forwardAddress) || //
					StringUtil.isEmpty(forwardPort) //
			) {
				continue;
			}
			RequestForwardConfig forwardConfig = new RequestForwardConfig();
			forwardConfig.setMatchUrl(matchUrl);
			forwardConfig.setForwardAddress(forwardAddress);
			forwardConfig.setForwardPort(Integer.parseInt(forwardPort));
			this.forwardingConfigs.add(forwardConfig);
		}
	}

	/**
	 * リダイレクト定義環境設定情報を読み込みます。<br>
	 * @throws Throwable リソースロード時に入出力例外が発生した場合にスローされます
	 */
	private void loadRedirectConfig() throws Throwable {
		HttpReverseProxySetting setting = HttpReverseProxySetting.instance();

		// ファイルが存在しない場合は処理をスキップ
		String configName = setting.getConfigPath() + "/" + this.serverRedirectConfig;
		if (StringUtil.isEmpty(this.serverRedirectConfig) || !new File(configName).exists()) {
			return;
		}

		// Propertiesリソース読み込み
		FileInputStream stream = new FileInputStream(configName);
		Properties properties = new Properties();
		properties.load(stream);
		stream.close();

		// 定義読み込みループ処理
		int redirectDefinitionCount = Integer.parseInt(properties.getProperty("redirectDefinitionCount"));
		for (int i = 0; i <= redirectDefinitionCount; i++) {
			String matchUrl = properties.getProperty("redirect." + i + ".matchUrl");
			String redirectUrl = properties.getProperty("redirect." + i + ".redirectUrl");
			if ( //
			StringUtil.isEmpty(matchUrl) || //
					StringUtil.isEmpty(redirectUrl) //
			) {
				continue;
			}
			RequestRedirectConfig redirectConfig = new RequestRedirectConfig();
			redirectConfig.setMatchUrl(matchUrl);
			redirectConfig.setRedirectUrl(redirectUrl);
			this.redirectingConfigs.add(redirectConfig);
		}
	}

	/**
	 * ハンドラ定義環境設定情報を読み込みます。<br>
	 * @throws Throwable リソースロード時に入出力例外が発生した場合にスローされます
	 */
	private void loadHandlerConfig() throws Throwable {
		HttpReverseProxySetting setting = HttpReverseProxySetting.instance();

		// ファイルが存在しない場合は処理をスキップ
		String configName = setting.getConfigPath() + "/" + this.serverHandlerConfig;
		if (StringUtil.isEmpty(this.serverHandlerConfig) || !new File(configName).exists()) {
			return;
		}

		// Propertiesリソース読み込み
		FileInputStream stream = new FileInputStream(configName);
		Properties properties = new Properties();
		properties.load(stream);
		stream.close();

		// 受信側ハンドラ定義読み込みループ処理
		int reciveHandlerDefinitionCount = Integer.parseInt(properties.getProperty("reciveHandlerDefinitionCount"));
		for (int i = 0; i <= reciveHandlerDefinitionCount; i++) {
			String className = properties.getProperty("reciveHandler." + i + ".class");
			if (StringUtil.isEmpty(className)) {
				continue;
			}
			Class clazz = Class.forName(className);
			this.reciveRequestHandlers.add(clazz);
		}

		// 転送側ハンドラ定義読み込みループ処理
		int forwardHandlerDefinitionCount = Integer.parseInt(properties.getProperty("forwardHandlerDefinitionCount"));
		for (int i = 0; i <= forwardHandlerDefinitionCount; i++) {
			String className = properties.getProperty("forwardHandler." + i + ".class");
			if (StringUtil.isEmpty(className)) {
				continue;
			}
			Class clazz = Class.forName(className);
			this.forwardResponseHandlers.add(clazz);
		}
	}

	/**
	 * デーモン定義環境設定情報を読み込みます。<br>
	 * @throws Throwable リソースロード時に入出力例外が発生した場合にスローされます
	 */
	private void loadDaemonConfig() throws Throwable {
		HttpReverseProxySetting setting = HttpReverseProxySetting.instance();

		// ファイルが存在しない場合は処理をスキップ
		String configName = setting.getConfigPath() + "/" + this.serverDaemonConfig;
		if (StringUtil.isEmpty(this.serverDaemonConfig) || !new File(configName).exists()) {
			return;
		}

		// Propertiesリソース読み込み
		FileInputStream stream = new FileInputStream(configName);
		Properties properties = new Properties();
		properties.load(stream);
		stream.close();

		// デーモン定義読み込みループ処理
		int daemonDefinitionCount = Integer.parseInt(properties.getProperty("daemonDefinitionCount"));
		for (int i = 0; i <= daemonDefinitionCount; i++) {
			String daemonClassName = properties.getProperty("daemon." + i + ".daemonClass");
			String name = properties.getProperty("daemon." + i + ".name");
			String loopDelay = properties.getProperty("daemon." + i + ".loopDelay");
			String loopCount = properties.getProperty("daemon." + i + ".loopCount");
			String retryCountOnError = properties.getProperty("daemon." + i + ".retryCountOnError");
			if (StringUtil.isEmpty(daemonClassName)) {
				continue;
			}
			if (StringUtil.isEmpty(name)) {
				continue;
			}
			if (StringUtil.isEmpty(loopDelay)) {
				continue;
			}
			if (StringUtil.isEmpty(loopCount)) {
				continue;
			}
			if (StringUtil.isEmpty(retryCountOnError)) {
				continue;
			}
			HttpReverseProxyDaemonRegistry registry = HttpReverseProxyDaemonRegistry.instance();

			Class daemonClass = Class.forName(daemonClassName);
			Constructor daemonConstructor = daemonClass.getConstructor(new Class[] { String.class, long.class, int.class, int.class });
			HttpReverseProxyDaemon daemon = (HttpReverseProxyDaemon) daemonConstructor.newInstance(new Object[] { //
					name, //
							Long.valueOf(loopDelay), //
							Integer.valueOf(loopCount), //
							Integer.valueOf(retryCountOnError) });
			registry.add(daemon);
		}
	}

	// Accessor Method

	/**
	 * プロキシサーバー名を取得します。<br>
	 * @return プロキシサーバー名
	 */
	public String getServerName() {
		return StringUtil.nullsafe(serverName);
	}

	/**
	 * プロキシサーバー名を設定します。<br>
	 * @param serverName プロキシサーバー名
	 */
	public void setServerName(String serverName) {
		this.serverName = serverName;
	}

	/**
	 * プロキシサーバーアドレスを取得します。<br>
	 * @return プロキシサーバーアドレス
	 */
	public String getServerAddress() {
		return StringUtil.nullsafe(serverAddress);
	}

	/**
	 * プロキシサーバーアドレスを設定します。<br>
	 * @param serverAddress プロキシサーバーアドレス
	 */
	public void setServerAddress(String serverAddress) {
		this.serverAddress = serverAddress;
	}

	/**
	 * プロキシサーバーポートを取得します。<br>
	 * @return プロキシサーバーポート
	 */
	public int getServerPort() {
		return serverPort;
	}

	/**
	 * プロキシサーバーポートを設定します。<br>
	 * @param serverPort プロキシサーバーポート
	 */
	public void setServerPort(int serverPort) {
		this.serverPort = serverPort;
	}

	/**
	 * Log4J環境定義リソースを取得します。<br>
	 * @return Log4J環境定義リソース
	 */
	public String getServerLog4JConfig() {
		return StringUtil.nullsafe(serverLog4JConfig);
	}

	/**
	 * Log4J環境定義リソースを設定します。<br>
	 * @param serverLog4JConfig Log4J環境定義リソース
	 */
	public void setServerLog4JConfig(String serverLog4JConfig) {
		this.serverLog4JConfig = serverLog4JConfig;
	}

	/**
	 * プロキシサーバーコンテンツ基底パスを取得します。<br>
	 * @return プロキシサーバーコンテンツ基底パス
	 */
	public String getServerContentRootPath() {
		return StringUtil.nullsafe(serverContentRootPath);
	}

	/**
	 * プロキシサーバーコンテンツ基底パスを設定します。<br>
	 * @param serverContentRootPath プロキシサーバーコンテンツ基底パス
	 */
	public void setServerContentRootPath(String serverContentRootPath) {
		this.serverContentRootPath = serverContentRootPath;
	}

	/**
	 * サーバーコンテンツVelocity環境定義リソースを取得します。<br>
	 * @return サーバーコンテンツVelocity環境定義リソース
	 */
	public String getServerContentVelocityConfig() {
		return StringUtil.nullsafe(serverContentVelocityConfig);
	}

	/**
	 * サーバーコンテンツVelocity環境定義リソースを設定します。<br>
	 * @param serverContentVelocityConfig サーバーコンテンツVelocity環境定義リソース
	 */
	public void setServerContentVelocityConfig(String serverContentVelocityConfig) {
		this.serverContentVelocityConfig = serverContentVelocityConfig;
	}

	/**
	 * プロキシサーバーコンテンツVelocityテンプレート拡張子を取得します。<br>
	 * @return プロキシサーバーコンテンツVelocityテンプレート拡張子
	 */
	public String getServerContentVelocityExtension() {
		return StringUtil.nullsafe(serverContentVelocityExtension);
	}

	/**
	 * プロキシサーバーコンテンツVelocityテンプレート拡張子を設定します。<br>
	 * @param serverContentVelocityExtension プロキシサーバーコンテンツVelocityテンプレート拡張子
	 */
	public void setServerContentVelocityExtension(String serverContentVelocityExtension) {
		this.serverContentVelocityExtension = serverContentVelocityExtension;
	}

	/**
	 * プロキシサーバーコンテンツ文字コードを取得します。<br>
	 * @return プロキシサーバーコンテンツ文字コード
	 */
	public String getServerContentCharset() {
		return serverContentCharset;
	}

	/**
	 * プロキシサーバーコンテンツ文字コードを設定します。<br>
	 * @param serverContentCharset プロキシサーバーコンテンツ文字コード
	 */
	public void setServerContentCharset(String serverContentCharset) {
		this.serverContentCharset = serverContentCharset;
	}

	/**
	 * プロキシサーバーコンテンツエイリアスを取得します。<br>
	 * @return プロキシサーバーコンテンツエイリアス
	 */
	public String getServerContentAlias() {
		return StringUtil.isEmpty(serverContentAlias) ? "/" : serverContentAlias.startsWith("/") ? serverContentAlias : "/" + serverContentAlias;
	}

	/**
	 * プロキシサーバーコンテンツエイリアスを設定します。<br>
	 * @param serverContentAlias プロキシサーバーコンテンツエイリアス
	 */
	public void setServerContentAlias(String serverContentAlias) {
		this.serverContentAlias = serverContentAlias;
	}

	/**
	 * サーバーコンテンツ(リダイレクトページ)を取得します。<br>
	 * @return サーバーコンテンツ(リダイレクトページ)
	 */
	public String getServerRedirectContent() {
		return StringUtil.nullsafe(serverRedirectContent);
	}

	/**
	 * サーバーコンテンツ(リダイレクトページ)を設定します。<br>
	 * @param serverRedirectContent サーバーコンテンツ(リダイレクトページ)
	 */
	public void setServerRedirectContent(String serverRedirectContent) {
		this.serverRedirectContent = serverRedirectContent;
	}

	/**
	 * サーバーコンテンツ(マッピング不正ページ)を取得します。<br>
	 * @return サーバーコンテンツ(マッピング不正ページ)
	 */
	public String getServerInvalidMappingContent() {
		return StringUtil.nullsafe(serverInvalidMappingContent);
	}

	/**
	 * サーバーコンテンツ(マッピング不正ページ)を設定します。<br>
	 * @param serverInvalidMappingContent サーバーコンテンツ(マッピング不正ページ)
	 */
	public void setServerInvalidMappingContent(String serverInvalidMappingContent) {
		this.serverInvalidMappingContent = serverInvalidMappingContent;
	}

	/**
	 * サーバーコンテンツ(サーバー致命的エラーページ)を取得します。<br>
	 * @return サーバーコンテンツ(サーバー致命的エラーページ)
	 */
	public String getServerFatalErrorContent() {
		return StringUtil.nullsafe(serverFatalErrorContent);
	}

	/**
	 * サーバーコンテンツ(サーバー致命的エラーページ)を設定します。<br>
	 * @param serverFatalErrorContent サーバーコンテンツ(サーバー致命的エラーページ)
	 */
	public void setServerFatalErrorContent(String serverFatalErrorContent) {
		this.serverFatalErrorContent = serverFatalErrorContent;
	}

	/**
	 * サーバーコンテンツ(サーバータイムアウトエラーページ)を取得します。<br>
	 * @return サーバーコンテンツ(サーバータイムアウトエラーページ)
	 */
	public String getServerTimeoutErrorContent() {
		return StringUtil.nullsafe(serverTimeoutErrorContent);
	}

	/**
	 * サーバーコンテンツ(サーバータイムアウトエラーページ)を設定します。<br>
	 * @param serverTimeoutErrorContent サーバーコンテンツ(サーバータイムアウトエラーページ)
	 */
	public void setServerTimeoutErrorContent(String serverTimeoutErrorContent) {
		this.serverTimeoutErrorContent = serverTimeoutErrorContent;
	}

	/**
	 * リダイレクト定義環境設定ファイルを取得します。<br>
	 * @return リダイレクト定義環境設定ファイル
	 */
	public String getServerRedirectConfig() {
		return StringUtil.nullsafe(serverRedirectConfig);
	}

	/**
	 * リダイレクト定義環境設定ファイルを設定します。<br>
	 * @param serverRedirectConfig リダイレクト定義環境設定ファイル
	 */
	public void setServerRedirectConfig(String serverRedirectConfig) {
		this.serverRedirectConfig = serverRedirectConfig;
	}

	/**
	 * 転送定義環境設定ファイルを取得します。<br>
	 * @return 転送定義環境設定ファイル
	 */
	public String getServerForwardConfig() {
		return StringUtil.nullsafe(serverForwardConfig);
	}

	/**
	 * 転送定義環境設定ファイルを設定します。<br>
	 * @param serverForwardConfig 転送定義環境設定ファイル
	 */
	public void setServerForwardConfig(String serverForwardConfig) {
		this.serverForwardConfig = serverForwardConfig;
	}

	/**
	 * ハンドラ定義環境設定ファイルを取得します。<br>
	 * @return ハンドラ定義環境設定ファイル
	 */
	public String getServerHandlerConfig() {
		return StringUtil.nullsafe(serverHandlerConfig);
	}

	/**
	 * ハンドラ定義環境設定ファイルを設定します。<br>
	 * @param serverHandlerConfig ハンドラ定義環境設定ファイル
	 */
	public void setServerHandlerConfig(String serverHandlerConfig) {
		this.serverHandlerConfig = serverHandlerConfig;
	}

	/**
	 * デーモンサービス環境設定ファイルを取得します。<br>
	 * @return デーモンサービス環境設定ファイル
	 */
	public String getServerDaemonConfig() {
		return serverDaemonConfig;
	}

	/**
	 * デーモンサービス環境設定ファイルを設定します。<br>
	 * @param serverDaemonConfig デーモンサービス環境設定ファイル
	 */
	public void setServerDaemonConfig(String serverDaemonConfig) {
		this.serverDaemonConfig = serverDaemonConfig;
	}

	/**
	 * プロキシSSL有効フラグを取得します。<br>
	 * @return プロキシSSL有効フラグ
	 */
	public boolean isServerSSLEnabled() {
		return serverSSLEnabled;
	}

	/**
	 * プロキシSSL有効フラグを設定します。<br>
	 * @param serverSSLEnabled プロキシSSL有効フラグ
	 */
	public void setServerSSLEnabled(boolean serverSSLEnabled) {
		this.serverSSLEnabled = serverSSLEnabled;
	}

	/**
	 * プロキシSSLサーバーキーストアパスを取得します。<br>
	 * @return プロキシSSLサーバーキーストアパス
	 */
	public String getServerSSLKeystore() {
		return StringUtil.nullsafe(serverSSLKeystore);
	}

	/**
	 * プロキシSSLサーバーキーストアパスを設定します。<br>
	 * @param serverSSLKeystore プロキシSSLサーバーキーストアパス
	 */
	public void setServerSSLKeystore(String serverSSLKeystore) {
		this.serverSSLKeystore = serverSSLKeystore;
	}

	/**
	 * プロキシSSLサーバーキーストアパスワードを取得します。<br>
	 * @return プロキシSSLサーバーキーストアパスワード
	 */
	public String getServerSSLKeystorePasswd() {
		return StringUtil.nullsafe(serverSSLKeystorePasswd);
	}

	/**
	 * プロキシSSLサーバーキーストアパスワードを設定します。<br>
	 * @param serverSSLKeystorePasswd プロキシSSLサーバーキーストアパスワード
	 */
	public void setServerSSLKeystorePasswd(String serverSSLKeystorePasswd) {
		this.serverSSLKeystorePasswd = serverSSLKeystorePasswd;
	}

	/**
	 * プロキシSSLサーバーキーストアエイリアスを取得します。<br>
	 * @return プロキシSSLサーバーキーストアエイリアス
	 */
	public String getServerSSLKeystoreAlias() {
		return StringUtil.nullsafe(serverSSLKeystoreAlias);
	}

	/**
	 * プロキシSSLサーバーキーストアエイリアスを設定します。<br>
	 * @param serverSSLKeystoreAlias プロキシSSLサーバーキーストアエイリアス
	 */
	public void setServerSSLKeystoreAlias(String serverSSLKeystoreAlias) {
		this.serverSSLKeystoreAlias = serverSSLKeystoreAlias;
	}

	/**
	 * プロキシSSLサーバーキーストアCN(姓名)を取得します。<br>
	 * @return プロキシSSLサーバーキーストアCN(姓名)
	 */
	public String getServerSSLKeystoreCn() {
		return StringUtil.nullsafe(serverSSLKeystoreCn);
	}

	/**
	 * プロキシSSLサーバーキーストアCN(姓名)を設定します。<br>
	 * @param serverSSLKeystoreCn プロキシSSLサーバーキーストアCN(姓名)
	 */
	public void setServerSSLKeystoreCn(String serverSSLKeystoreCn) {
		this.serverSSLKeystoreCn = serverSSLKeystoreCn;
	}

	/**
	 * プロキシSSLサーバーキーストアOU(組織単位名)を取得します。<br>
	 * @return プロキシSSLサーバーキーストアOU(組織単位名)
	 */
	public String getServerSSLKeystoreOu() {
		return StringUtil.nullsafe(serverSSLKeystoreOu);
	}

	/**
	 * プロキシSSLサーバーキーストアOU(組織単位名)を設定します。<br>
	 * @param serverSSLKeystoreOu プロキシSSLサーバーキーストアOU(組織単位名)
	 */
	public void setServerSSLKeystoreOu(String serverSSLKeystoreOu) {
		this.serverSSLKeystoreOu = serverSSLKeystoreOu;
	}

	/**
	 * プロキシSSLサーバーキーストアO(組織名)を取得します。<br>
	 * @return プロキシSSLサーバーキーストアO(組織名)
	 */
	public String getServerSSLKeystoreO() {
		return StringUtil.nullsafe(serverSSLKeystoreO);
	}

	/**
	 * プロキシSSLサーバーキーストアO(組織名)を設定します。<br>
	 * @param serverSSLKeystoreO プロキシSSLサーバーキーストアO(組織名)
	 */
	public void setServerSSLKeystoreO(String serverSSLKeystoreO) {
		this.serverSSLKeystoreO = serverSSLKeystoreO;
	}

	/**
	 * ソケットストリームバッファサイズを取得します。<br>
	 * @return ソケットストリームバッファサイズ
	 */
	public int getSocketBufferSize() {
		return socketBufferSize;
	}

	/**
	 * ソケットストリームバッファサイズを設定します。<br>
	 * @param socketBufferSize ソケットストリームバッファサイズ
	 */
	public void setSocketBufferSize(int socketBufferSize) {
		this.socketBufferSize = socketBufferSize;
	}

	/**
	 * ソケットタイムアウト時間を取得します。<br>
	 * @return ソケットタイムアウト時間
	 */
	public int getSocketTimeout() {
		return socketTimeout;
	}

	/**
	 * ソケットタイムアウト時間を設定します。<br>
	 * @param socketTimeout ソケットタイムアウト時間
	 */
	public void setSocketTimeout(int socketTimeout) {
		this.socketTimeout = socketTimeout;
	}

	/**
	 * 指定されたプロキシ転送定義を追加します。<br>
	 * @param config プロキシ転送定義
	 */
	public void addForwardingConfig(RequestForwardConfig config) {
		forwardingConfigs.add(config);
	}

	/**
	 * 指定されたURLに合致するプロキシ転送定義を取得します。<br>
	 * 合致する定義が存在しない場合はnullを返却します。<br>
	 * @param url 転送対象判別URL
	 * @return プロキシ転送定義
	 */
	public RequestForwardConfig getForwardingConfig(String url) {
		for (Iterator iterator = forwardingConfigs.iterator(); iterator.hasNext();) {
			RequestForwardConfig config = (RequestForwardConfig) iterator.next();
			if (url.matches(config.getMatchUrl())) {
				return config;
			}
		}
		return null;
	}

	/**
	 * 指定されたプロキシ転送定義を削除します。<br>
	 * @param config プロキシ転送定義
	 */
	public void removeForwardingConfig(RequestForwardConfig config) {
		forwardingConfigs.remove(config);
	}

	/**
	 * 指定されたURLに合致するプロキシ転送定義を削除します。<br>
	 * @param url 転送対象判別URL
	 */
	public void removeForwardingConfig(String url) {
		RequestForwardConfig config = getForwardingConfig(url);
		if (config != null) {
			removeForwardingConfig(config);
		}
	}

	/**
	 * 転送定義数を取得します。<br>
	 * @return 転送定義数
	 */
	public int getForwardingConfigCount() {
		return forwardingConfigs.size();
	}

	/**
	 * 指定されたリダイレクト定義を追加します。<br>
	 * @param config リダイレクト定義
	 */
	public void addRedirectingConfig(RequestRedirectConfig config) {
		redirectingConfigs.add(config);
	}

	/**
	 * 指定されたURLに合致するリダイレクト定義を取得します。<br>
	 * 合致する定義が存在しない場合はnullを返却します。<br>
	 * @param url 転送対象判別URL
	 * @return リダイレクト定義
	 */
	public RequestRedirectConfig getRedirectingConfig(String url) {
		for (Iterator iterator = redirectingConfigs.iterator(); iterator.hasNext();) {
			RequestRedirectConfig config = (RequestRedirectConfig) iterator.next();
			if (url.matches(config.getMatchUrl())) {
				return config;
			}
		}
		return null;
	}

	/**
	 * 指定されたリダイレクト定義を削除します。<br>
	 * @param config リダイレクト定義
	 */
	public void removeRedirectingConfig(RequestRedirectConfig config) {
		redirectingConfigs.remove(config);
	}

	/**
	 * 指定されたURLに合致するリダイレクト定義を削除します。<br>
	 * @param url 転送対象判別URL
	 */
	public void removeRedirectingConfig(String url) {
		RequestRedirectConfig config = getRedirectingConfig(url);
		if (config != null) {
			removeRedirectingConfig(config);
		}
	}

	/**
	 * リダイレクト定義数を取得します。<br>
	 * @return リダイレクト定義数
	 */
	public int getRedirectingConfigCount() {
		return redirectingConfigs.size();
	}

	/**
	 * リクエスト受信側ハンドラを追加します。<br>
	 * @param clazz リクエスト受信ハンドラクラス
	 */
	public void addReciveRequestHandler(Class clazz) {
		if (clazz != null) {
			reciveRequestHandlers.add(clazz);
		}
	}

	/**
	 * リクエスト受信側ハンドラを削除します。<br>
	 * @param clazz リクエスト受信ハンドラクラス
	 */
	public void removeReciveRequestHandler(Class clazz) {
		if (clazz != null) {
			reciveRequestHandlers.remove(clazz);
		}
	}

	/**
	 * リクエスト受信側ハンドラをクリアします。<br>
	 */
	public void clearReciveRequestHandler() {
		reciveRequestHandlers.clear();
	}

	/**
	 * 登録されているリクエスト受信側ハンドラの反復子を取得します。<br>
	 * @return 登録されているリクエスト受信側ハンドラの反復子
	 */
	public Iterator getReciveRequestHandlerIterator() {
		return reciveRequestHandlers.iterator();
	}

	/**
	 * 登録されているリクエスト受信側ハンドラの数を取得します。<br>
	 * @return 登録されているリクエスト受信側ハンドラの数
	 */
	public int getReciveRequestHandlerCount() {
		return reciveRequestHandlers.size();
	}

	/**
	 * レスポンス転送側ハンドラを追加します。<br>
	 * @param clazz レスポンス転送ハンドラクラス
	 */
	public void addForwardResponseHandler(Class clazz) {
		if (clazz != null) {
			forwardResponseHandlers.add(clazz);
		}
	}

	/**
	 * レスポンス転送側ハンドラを追加します。<br>
	 * @param clazz レスポンス転送ハンドラクラス
	 */
	public void removeForwardResponseHandler(Class clazz) {
		if (clazz != null) {
			forwardResponseHandlers.remove(clazz);
		}
	}

	/**
	 * レスポンス転送側ハンドラをクリアします。<br>
	 */
	public void clearForwardResponseHandler() {
		forwardResponseHandlers.clear();
	}

	/**
	 * 登録されているレスポンス転送側ハンドラの反復子を取得します。<br>
	 * @return 登録されているレスポンス転送側ハンドラの反復子
	 */
	public Iterator getForwardResponseHandlerIterator() {
		return forwardResponseHandlers.iterator();
	}

	/**
	 * 登録されているレスポンス転送側ハンドラの数を取得します。<br>
	 * @return 登録されているレスポンス転送側ハンドラの数
	 */
	public int getForwardResponseHandlerCount() {
		return forwardResponseHandlers.size();
	}

	// Supplement Method

	/**
	 * 指定されたサーバーコンテンツファイルリソースパスを取得します。<br>
	 * @param filename リソースファイル名
	 * @return サーバーコンテンツファイルリソースパス
	 * @throws IOException 指定されたパスからFileオブジェクトを正しく生成できなかった場合に発生
	 */
	public String getServerContentResourcePath(String filename) throws IOException {
		return FileUtil.connectPath(new String[] { getServerContentRootPath(), getServerContentAlias(), filename });
	}

	/**
	 * オブジェクトの文字列表現を返します。<br>
	 * @return オブジェクトの文字列表現
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("{");
		buffer.append("serverName=");
		buffer.append(serverName);
		buffer.append(",");
		buffer.append("serverAddress=");
		buffer.append(serverAddress);
		buffer.append(",");
		buffer.append("serverPort=");
		buffer.append(serverPort);
		buffer.append(",");
		buffer.append("serverLog4JConfig=");
		buffer.append(serverLog4JConfig);
		buffer.append(",");
		buffer.append("serverContentRootPath=");
		buffer.append(serverContentRootPath);
		buffer.append(",");
		buffer.append("serverContentVelocityConfig=");
		buffer.append(serverContentVelocityConfig);
		buffer.append(",");
		buffer.append("serverContentVelocityExtension=");
		buffer.append(serverContentVelocityExtension);
		buffer.append(",");
		buffer.append("serverContentCharset=");
		buffer.append(serverContentCharset);
		buffer.append(",");
		buffer.append("serverContentAlias=");
		buffer.append(serverContentAlias);
		buffer.append(",");
		buffer.append("serverRedirectContent=");
		buffer.append(serverRedirectContent);
		buffer.append(",");
		buffer.append("serverInvalidMappingContent=");
		buffer.append(serverInvalidMappingContent);
		buffer.append(",");
		buffer.append("serverFatalErrorContent=");
		buffer.append(serverFatalErrorContent);
		buffer.append(",");
		buffer.append("serverTimeoutErrorContent=");
		buffer.append(serverTimeoutErrorContent);
		buffer.append(",");
		buffer.append("serverForwardConfig=");
		buffer.append(serverForwardConfig);
		buffer.append(",");
		buffer.append("serverRedirectConfig=");
		buffer.append(serverRedirectConfig);
		buffer.append(",");
		buffer.append("serverHandlerConfig=");
		buffer.append(serverHandlerConfig);
		buffer.append(",");
		buffer.append("serverDaemonConfig=");
		buffer.append(serverDaemonConfig);
		buffer.append(",");
		buffer.append("serverSSLEnabled=");
		buffer.append(serverSSLEnabled);
		buffer.append(",");
		buffer.append("serverSSLKeystore=");
		buffer.append(serverSSLKeystore);
		buffer.append(",");
		buffer.append("serverSSLKeystorePasswd=");
		buffer.append(serverSSLKeystorePasswd);
		buffer.append(",");
		buffer.append("serverSSLKeystoreAlias=");
		buffer.append(serverSSLKeystoreAlias);
		buffer.append(",");
		buffer.append("serverSSLKeystoreCn=");
		buffer.append(serverSSLKeystoreCn);
		buffer.append(",");
		buffer.append("serverSSLKeystoreOu=");
		buffer.append(serverSSLKeystoreOu);
		buffer.append(",");
		buffer.append("serverSSLKeystoreO=");
		buffer.append(serverSSLKeystoreO);
		buffer.append(",");
		buffer.append("socketBufferSize=");
		buffer.append(socketBufferSize);
		buffer.append(",");
		buffer.append("socketTimeout=");
		buffer.append(socketTimeout);
		buffer.append(",");
		buffer.append("forwardingConfigs=");
		buffer.append(forwardingConfigs);
		buffer.append(",");
		buffer.append("redirectingConfigs=");
		buffer.append(redirectingConfigs);
		buffer.append("}");
		return buffer.toString();
	}
}
