package org.phosphoresce.webcore.ext.struts.action;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.phosphoresce.webcore.core.TemporaryResource;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.phosphoresce.webcore.core.servlet.ServletOutputStream;
import org.phosphoresce.webcore.ext.struts.exception.StrutsProcessInternalException;
import org.slf4j.LoggerFactory;

/**
 * ストリームコンテナクラス<br>
 * <br>
 * サーブレット処理中に利用するレスポンスストリームを含む、入出力リソースストリームのキャッシュコンテナクラスです。<br>
 * リクエスト中に操作するリソースストリームは全て当クラスが一元管理することでクローズ漏れを回避するフレームワーク設計とします。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2011/01/13	Kitagawa		新規作成
 * 2013/05/17	Kitagawa		全体的に再構築
 *-->
 */
public final class StreamContainer {

	/** ロガーオブジェクト */
	private CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(getClass()));

	/** リクエストオブジェクト */
	private HttpServletRequest request;

	/** レスポンスオブジェクト */
	private HttpServletResponse response;

	/** レスポンスストリーム */
	private OutputStream responseStream;

	/** 出力ストリームキャッシュ */
	private Map<String, List<OutputStream>> resourceOutputStreams;

	/** 出力ストリームキャッシュ */
	private Map<String, List<InputStream>> resourceInputStreams;

	/**
	 * コンストラクタ<br>
	 * @param request リクエストオブジェクト
	 * @param response レスポンスオブジェクト
	 */
	StreamContainer(HttpServletRequest request, HttpServletResponse response) {
		super();
		if (request == null) {
			throw new IllegalArgumentException("request is required");
		}
		if (response == null) {
			throw new IllegalArgumentException("response is required");
		}
		this.request = request;
		this.response = response;
		resourceOutputStreams = new HashMap<String, List<OutputStream>>();
		resourceInputStreams = new HashMap<String, List<InputStream>>();
	}

	/**
	 * コンテナで管理されているストリームオブジェクトを全てクローズします。<br>
	 */
	void close() {
		try {
			// レスポンスストリームクローズ
			if (responseStream != null) {
				responseStream.flush();
				responseStream.close();
			}
			// 出力ストリーム一括クローズ
			synchronized (resourceOutputStreams) {
				for (String path : resourceOutputStreams.keySet()) {
					List<OutputStream> resourceOutputStreamList = resourceOutputStreams.get(path);
					for (OutputStream resourceOutputStream : resourceOutputStreamList) {
						resourceOutputStream.flush();
						resourceOutputStream.close();
					}
				}
				resourceOutputStreams.clear();
			}
			// 入力ストリーム一括クローズ
			synchronized (resourceInputStreams) {
				for (String path : resourceInputStreams.keySet()) {
					List<InputStream> resourceInputStreamList = resourceInputStreams.get(path);
					for (InputStream resourceInputStream : resourceInputStreamList) {
						resourceInputStream.close();
					}
				}
				resourceInputStreams.clear();
			}
		} catch (Throwable e) {
			throw new StrutsProcessInternalException("FSTR10003", e);
		}
	}

	/**
	 * 指定されたMIMEタイプのレスポンスストリームオブジェクトを取得します。<br>
	 * @param charset キャラクタセット
	 * @param mimeType MIMEタイプ
	 * @param contentLength ダウンロードファイルサイズ(不明な場合はnull)
	 * @return レスポンスストリームオブジェクト
	 */
	OutputStream getResponseStream(String charset, String mimeType, Integer contentLength) {
		if (responseStream != null) {
			throw new StrutsProcessInternalException("FSTR10005");
		}
		responseStream = new BufferedOutputStream(ServletOutputStream.create(request, response, charset, mimeType, contentLength));
		return responseStream;
	}

	/**
	 * 指定されたMIMEタイプのレスポンスストリームオブジェクトを取得します。<br>
	 * @param mimeType MIMEタイプ
	 * @param contentLength ダウンロードファイルサイズ(不明な場合はnull)
	 * @return レスポンスストリームオブジェクト
	 */
	public OutputStream getResponseStream(String mimeType, Integer contentLength) {
		return getResponseStream(null, mimeType, contentLength);
	}

	/**
	 * 指定されたMIMEタイプのレスポンスストリームオブジェクトを取得します。<br>
	 * @param mimeType MIMEタイプ
	 * @return レスポンスストリームオブジェクト
	 */
	public OutputStream getResponseStream(String mimeType) {
		return getResponseStream(null, mimeType, null);
	}

	/**
	 * ダウンロード用レスポンスストリームオブジェクトを取得します。<br>
	 * @param charset キャラクタセット
	 * @param mimeType MIMEタイプ文字列
	 * @param contentLength ダウンロードファイルサイズ(不明な場合はnull)
	 * @param filename ダウンロードファイル名
	 * @return レスポンスストリームオブジェクト
	 */
	public OutputStream getDownloadStream(String charset, String mimeType, Integer contentLength, String filename) {
		if (responseStream != null) {
			throw new StrutsProcessInternalException("FSTR10005");
		}
		responseStream = ServletOutputStream.createDownloadStream(request, response, charset, mimeType, contentLength, filename);
		return responseStream;
	}

	/**
	 * ダウンロード用レスポンスストリームオブジェクトを取得します。<br>
	 * @param contentLength ダウンロードファイルサイズ(不明な場合はnull)
	 * @param filename ダウンロードファイル名
	 * @return レスポンスストリームオブジェクト
	 */
	public OutputStream getDownloadStream(Integer contentLength, String filename) {
		return getDownloadStream(null, null, contentLength, filename);
	}

	/**
	 * ダウンロード用レスポンスストリームオブジェクトを取得します。<br>
	 * @param filename ダウンロードファイル名
	 * @return レスポンスストリームオブジェクト
	 */
	public OutputStream getDownloadStream(String filename) {
		return getDownloadStream(null, null, null, filename);
	}

	/**
	 * JSONレスポンスストリームオブジェクトを取得します。<br>
	 * @param contentLength コンテンツサイズ(不明な場合はnull)
	 * @return レスポンスストリームオブジェクト
	 */
	public OutputStream getJSONStream(Integer contentLength) {
		if (responseStream != null) {
			throw new StrutsProcessInternalException("FSTR10005");
		}
		responseStream = ServletOutputStream.createJSONStream(request, response, null, null);
		return responseStream;
	}

	/**
	 * JSONレスポンスストリームオブジェクトを取得します。<br>
	 * @return レスポンスストリームオブジェクト
	 */
	public OutputStream getJSONStream() {
		return getJSONStream(null);
	}

	/**
	 * 指定されたパスの出力ストリームを取得します。<br>
	 * @param path リソースパス
	 * @return 出力ストリームオブジェクト
	 */
	public OutputStream getResourceOutputStream(String path) {
		try {
			synchronized (resourceOutputStreams) {
				if (!resourceOutputStreams.containsKey(path)) {
					resourceOutputStreams.put(path, new LinkedList<OutputStream>());
				}
				List<OutputStream> resourceOutputStreamList = resourceOutputStreams.get(path);
				OutputStream resourceOutputStream = new BufferedOutputStream(new FileOutputStream(path));
				resourceOutputStreamList.add(resourceOutputStream);
				return resourceOutputStream;
			}
		} catch (Throwable e) {
			throw new StrutsProcessInternalException("FSTR10001", new Object[] { path }, e);
		}
	}

	/**
	 * 指定されたパスの入力ストリームを取得します。<br>
	 * @param path リソースパス
	 * @return 入力ストリームオブジェクト
	 */
	public InputStream getResourceInputStream(String path) {
		try {
			synchronized (resourceInputStreams) {
				if (!resourceInputStreams.containsKey(path)) {
					resourceInputStreams.put(path, new LinkedList<InputStream>());
				}
				List<InputStream> resourceInputStreamList = resourceInputStreams.get(path);
				InputStream resourceInputStream = new BufferedInputStream(new FileInputStream(path));
				resourceInputStreamList.add(resourceInputStream);
				return resourceInputStream;
			}
		} catch (Throwable e) {
			throw new StrutsProcessInternalException("FSTR10002", new Object[] { path }, e);
		}
	}

	/**
	 * 指定されたテンポラリリソースパスの出力ストリームオブジェクトを取得します。<br>
	 * @param relativePath テンポラリリソースパス(環境定義されているテンポラリパスからの相対パス)
	 * @return 出力ストリームオブジェクト
	 */
	public OutputStream getTemporaryOutputStream(String relativePath) {
		String path = TemporaryResource.getFile(relativePath).getAbsolutePath();
		return getResourceOutputStream(path);
	}

	/**
	 * 指定されたテンポラリリソースパスの入力ストリームオブジェクトを取得します。<br>
	 * @param relativePath テンポラリリソースパス(環境定義されているテンポラリパスからの相対パス)
	 * @return 入力ストリームオブジェクト
	 */
	public InputStream getTemporaryInputStream(String relativePath) {
		String path = TemporaryResource.getFile(relativePath).getAbsolutePath();
		return getResourceInputStream(path);
	}
}
