package org.phosphoresce.webcore.ext.struts.action;

import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.util.Date;

import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.phosphoresce.webcore.ext.struts.exception.StrutsProcessInternalException;
import org.phosphoresce.webcore.ext.struts.exception.StrutsSessionTimeoutException;
import org.phosphoresce.webcore.ext.struts.util.StrutsAnnotationUtil;
import org.slf4j.LoggerFactory;

/**
 * ロジック上位抽象クラス<br>
 * <br>
 * 業務ロジックを実行するクラスで、トランザクションを必要とする処理は全てこのクラスでのみ実行可能とし、各サブロジッククラスはこれを継承して実装します。<br>
 * 当クラスはトランザクション処理を可能とするクラスである反面、アクションレイヤスコープであるディスパッチ処理は当クラスでは行いません。<br>
 * 当クラスで制限されるアクションレイヤスコープで実行する処理についてはサーブレットコンテナが提供するインタフェースのうち、レスポンス処理等も同様です。<br>
 * また、ディスパッチイベント単位でトランザクションが単一構成であることを前提とし、リダイレクトするサーブレット間でのトランザクション共有は対応しません。<br>
 * これはトランザクション管理を曖昧にすることなく、サーブレット単位で完結させるための思想であり、リダイレクトするサーブレット間でのトランザクション共有が必要な設計を不正と位置付けます。<br>
 * 但し、ロジッククラス処理のチェーンは思想上も許可しているため、業務処理設計でトランザクションを共有する必要がある場合は、ロジッククラスレイヤ間で業務処理を関連付ける設計とすることで対応します。<br>
 * <br>
 * 当クラスはサーブレットに依存する処理を担当するクラスの位置づけでフォームには依存しません。<br>
 * これはサーブレット処理単位でトランザクション業務処理の共通化が行えることを意味します。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/19	Kitagawa		新規作成
 * 2012/07/05	Kitagawa		全体的に再構築
 *-->
 */
public abstract class AbstractLogic<L extends Serializable> {

	/** ロガーオブジェクト */
	protected CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(getClass()));

	/** サーブレットコンテナオブジェクト */
	private ServletContainer<L> servlet;

	/**
	 * コンストラクタ<br>
	 * @param servlet サーブレットコンテナオブジェクト
	 */
	protected AbstractLogic(ServletContainer<L> servlet) {
		super();

		if (servlet == null) {
			throw new IllegalArgumentException("servlet is required");
		}

		this.servlet = servlet;
	}

	/**
	 * 現在時刻を取得します。<br>
	 * @return 現在時刻
	 */
	protected final Date getProcessDate() {
		//return new Date();
		return servlet.getProcessDate();
	}

	/**
	 * サーブレットコンテナオブジェクトを取得します。<br>
	 * @return サーブレットコンテナオブジェクト
	 */
	protected final ServletContainer<L> getServlet() {
		return new ServletContainerLogicWrapper<L>(servlet);
	}

	/**
	 * ウィンドウセッションオブジェクトを取得します。<br>
	 * @return ウィンドウセッションオブジェクト
	 */
	protected final WindowSessionContainer<L> getSession() {
		WindowSessionContainer<L> session = servlet.getSession().getWindowSession(servlet.getWindowId());
		if (session == null) {
			throw new StrutsSessionTimeoutException();
		}
		return servlet.getSession().getWindowSession(servlet.getWindowId());
	}

	/**
	 * ロジッククラスを生成します。<br>
	 * @param clazz ロジッククラス
	 * @return ロジッククラスインスタンス
	 */
	protected final <G extends AbstractLogic<L>> G createLogic(Class<G> clazz) {
		try {
			if (StrutsAnnotationUtil.getStrutsLogicAnnotation(clazz) == null) {
				throw new StrutsProcessInternalException("FSTR07001", new Object[] { (clazz == null ? "null" : clazz.getSimpleName()) });
			}
			Constructor<G> constructor = clazz.getConstructor(ServletContainer.class);
			G g = constructor.newInstance(servlet);
			servlet.addLogic(g);
			return g;
		} catch (Throwable e) {
			throw new StrutsProcessInternalException("FSTR07002", new Object[] { (clazz == null ? "null" : clazz.getSimpleName()) }, e);
		}
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param code メッセージコード
	 * @param binds バインドオブジェクト
	 * @see org.phosphoresce.webcore.ext.struts.action.MessageContainer
	 */
	public final void addMessage(String code, Object... binds) {
		servlet.getMessage().addMessage(code, binds);
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param code メッセージコード
	 * @see org.phosphoresce.webcore.ext.struts.action.MessageContainer
	 */
	public final void addMessage(String code) {
		servlet.getMessage().addMessage(code);
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param row 明細行数
	 * @param code メッセージコード
	 * @param binds バインドオブジェクト
	 * @see org.phosphoresce.webcore.ext.struts.action.MessageContainer
	 */
	public final void addMessage(Integer row, String code, Object... binds) {
		servlet.getMessage().addMessage(row, code, binds);
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param row 明細行数
	 * @param code メッセージコード
	 * @see org.phosphoresce.webcore.ext.struts.action.MessageContainer
	 */
	public final void addMessage(Integer row, String code) {
		servlet.getMessage().addMessage(row, code);
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param code メッセージコード
	 * @param binds バインドオブジェクト
	 * @see org.phosphoresce.webcore.ext.struts.action.MessageContainer
	 */
	public final void addMessageOnHead(String code, Object... binds) {
		servlet.getMessage().addMessageOnHead(code, binds);
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param code メッセージコード
	 * @see org.phosphoresce.webcore.ext.struts.action.MessageContainer
	 */
	public final void addMessageOnHead(String code) {
		servlet.getMessage().addMessageOnHead(code);
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param row 明細行数
	 * @param code メッセージコード
	 * @param binds バインドオブジェクト
	 * @see org.phosphoresce.webcore.ext.struts.action.MessageContainer
	 */
	public final void addMessageOnHead(Integer row, String code, Object... binds) {
		servlet.getMessage().addMessageOnHead(row, code, binds);
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param row 明細行数
	 * @param code メッセージコード
	 * @see org.phosphoresce.webcore.ext.struts.action.MessageContainer
	 */
	public final void addMessageOnHead(Integer row, String code) {
		servlet.getMessage().addMessageOnHead(row, code);
	}
}
