package org.phosphoresce.webcore.ext.smartlayout.taglib;

import java.lang.reflect.Field;

import javax.servlet.jsp.JspException;
import javax.servlet.jsp.PageContext;
import javax.servlet.jsp.tagext.Tag;

import org.phosphoresce.lib.commons.util.AnnotationUtil;
import org.phosphoresce.lib.commons.util.ReflectionUtil;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.phosphoresce.webcore.ext.smartlayout.exception.SmartLayoutTaglibProcessException;
import org.phosphoresce.webcore.ext.smartlayout.model.SmartTagModel;
import org.phosphoresce.webcore.ext.smartlayout.taglib.annotation.LooseTagConfig;
import org.phosphoresce.webcore.ext.smartlayout.taglib.annotation.SmartTagAttributeField;
import org.slf4j.LoggerFactory;

/**
 * 汎用カスタムタグクラス<br>
 * <br>
 * 当クラスはtldによって定義された属性を元にJSPで指定された内容を対応するモデルクラスへ反映する機能を提供します。<br>
 * 各カスタムタグサブクラスはこれを継承し、必要なフィールドのアクセッサを設けて対応するtld属性定義を行って利用します。<br>
 * 尚、実際のページコンテンツ自体の出力機能は当クラスでは提供せず、対応するモデルクラスへ処理を委譲します。<br>
 * <br>
 * 各カスタムタグサブクラスは当クラスを継承する形で実装される構成としていますが、汎用的な構成とする為に、
 * フレームワークでは{@link org.phosphoresce.webcore.ext.smartlayout.taglib.LooseTagAdapter}を提供します。<br>
 * これは全てのカスタムタグの属性を持つアダプタクラスであり、これを継承してtldで必要な属性のみを定義することで、
 * カスタムタグサブクラスの実装負荷を軽減する構成としています。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2013/03/13	Kitagawa		新規作成
 *-->
 */
abstract class LooseTag implements Tag {

	/** ロガーオブジェクト */
	private CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(getClass()));

	/** ページコンテキストオブジェクト */
	private PageContext context;

	/** 親タグエレメントオブジェクト */
	private Tag parent;

	/** タグモデルインスタンス */
	private SmartTagModel model;

	/**
	 * コンストラクタ<br>
	 */
	public LooseTag() {
		super();
	}

	/**
	 * ページコンテキストオブジェクトを設定します。<br>
	 * @param context ページコンテキストオブジェクト
	 * @see javax.servlet.jsp.tagext.Tag#setPageContext(javax.servlet.jsp.PageContext)
	 */
	@Override
	public final void setPageContext(PageContext context) {
		this.context = context;
	}

	/**
	 * ページコンテキストオブジェクトを取得します。<br>
	 * @return ページコンテキストオブジェクト
	 */
	public final PageContext getPageContext() {
		return context;
	}

	/**
	 * 親タグエレメントオブジェクトを設定します。<br>
	 * @param parent 親タグエレメントオブジェクト
	 * @see javax.servlet.jsp.tagext.Tag#setParent(javax.servlet.jsp.tagext.Tag)
	 */
	@Override
	public final void setParent(Tag parent) {
		this.parent = parent;
	}

	/**
	 * 親タグエレメントオブジェクトを取得します。<br>
	 * @return 親タグエレメントオブジェクト
	 */
	@Override
	public final Tag getParent() {
		return parent;
	}

	/**
	 * タグモデルインスタンスを取得します。<br>
	 * @return タグモデルインスタンス
	 */
	protected SmartTagModel getModel() {
		return model;
	}

	/**
	 * タグハンドラクラスのオブジェクトが不要と判断された時の処理を実行します。<br>
	 * 不要と判断されるまで、タグハンドラクラスのオブジェクトは繰返し再利用されます。<br>
	 * @see javax.servlet.jsp.tagext.Tag#release()
	 */
	@Override
	public void release() {
		context = null;
		parent = null;
	}

	/**
	 * 開始タグが呼び出されたときの処理を実行します。<br>
	 * 但し、当タグクラスの開始タグ処理はタグ自体の出力は行わず、指定されたモデルクラスへの情報反映のみが行われます。<br>
	 * @return 処理終了後の挙動定数
	 * @see javax.servlet.jsp.tagext.Tag#doStartTag()
	 */
	@Override
	public final int doStartTag() throws JspException {
		try {
			// タグフィールドにモデルインスタンスを設定
			LooseTagConfig looseTagConfig = AnnotationUtil.getClassAnnotation(LooseTagConfig.class, this.getClass());
			if (looseTagConfig == null) {
				throw new SmartLayoutTaglibProcessException("FSML00017", new Object[] { getClass().getName() });
			}
			Class<? extends SmartTagModel> modelClass = looseTagConfig.modelClass();
			model = modelClass.newInstance();
		} catch (Throwable e) {
			log.output(e, "FSML00009");
			throw new JspException(e);
		}
		return EVAL_BODY_INCLUDE;
	}

	/**
	 * 終了タグが呼び出されたときの処理を実行します。<br>
	 * 但し、当タグクラスの終了タグ処理は何も実行されません。<br>
	 * @return 処理終了後の挙動定数
	 * @see javax.servlet.jsp.tagext.Tag#doEndTag()
	 */
	@Override
	public final int doEndTag() throws JspException {
		try {
			// タグ環境設定取得
			LooseTagConfig looseTagConfig = AnnotationUtil.getClassAnnotation(LooseTagConfig.class, this.getClass());
			if (looseTagConfig == null) {
				throw new SmartLayoutTaglibProcessException("FSML00017", new Object[] { getClass().getName() });
			}

			// モデルオブジェクトにタグ指定属性を反映
			Class<? extends SmartTagModel> modelClass = AnnotationUtil.getClassAnnotation(LooseTagConfig.class, getClass()).modelClass();
			for (Field field : ReflectionUtil.getFields(modelClass)) {
				field.setAccessible(true);
				SmartTagAttributeField tagAttributeField = AnnotationUtil.getFieldAnnotation(SmartTagAttributeField.class, modelClass, field.getName());
				if (tagAttributeField != null) {
					Field tagField = ReflectionUtil.getField(getClass(), field.getName());
					if (tagField != null) {
						Object object = tagField.get(this);
						if (object == null && tagAttributeField.defaultValue() != null) {
							object = tagAttributeField.defaultValue();
						}
						field.set(model, object);
					}
				}
			}

			// 出力処理指示属性が付与されている場合に出力
			if (looseTagConfig.output()) {
				model.output(getPageContext());
			} else {
				// 親モデルに自モデルを追加
				LooseTag parentLooseTag = findParentLooseTag();
				if (parentLooseTag != null) {
					parentLooseTag.model.addChild(model);
				}

				// 親モデルが存在しない場合再上位スマートレイアウトクラスインスタンスに追加
				if (parentLooseTag == null) {
					SmartLayout smartLayout = findSmartLayout();
					if (smartLayout != null) {
						smartLayout.getModel().addChild(model);
					}
				}
			}
		} catch (Throwable e) {
			log.output(e, "FSML00010");
			throw new JspException(e);
		} finally {
			model = null;
		}
		return EVAL_PAGE;
	}

	/**
	 * 親に持つべきスマートレイアウトタグインスタンスを取得します。<br>
	 * @return スマートレイアウトタグインスタンス
	 */
	private SmartLayout findSmartLayout() {
		Tag target = this;
		while (target != null) {
			if (target instanceof SmartLayout) {
				return (SmartLayout) target;
			}
			target = target.getParent();
		}
		throw new SmartLayoutTaglibProcessException("FSML00011");
	}

	/**
	 * 親に存在する一番近い汎用タグクラスインスタンスを取得します。<br>
	 * @return 汎用タグクラスインスタンス
	 */
	private LooseTag findParentLooseTag() {
		Tag target = getParent();
		while (target != null) {
			if (target instanceof LooseTag) {
				return (LooseTag) target;
			}
			target = target.getParent();
		}
		return null;
	}
}
