package org.phosphoresce.webcore.ext.jasper.export;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.OutputStream;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;

import net.sf.jasperreports.engine.JRExporter;
import net.sf.jasperreports.engine.JRExporterParameter;

import org.phosphoresce.webcore.ext.jasper.JasperConstants;
import org.phosphoresce.webcore.ext.jasper.exception.JasperCriticalException;

/**
 * JasperReports帳票出力上位抽象クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2012/07/06	Kitagawa		新規作成
 *-->
 */
public abstract class JasperWriter implements JasperConstants {

	/** プリンタ名 */
	private String printerName;

	/**
	 * コンストラクタ<br>
	 */
	protected JasperWriter() {
		super();
	}

	/**
	 * プリンタ名を取得します。<br>
	 * @return プリンタ名
	 */
	public final String getPrinterName() {
		return printerName;
	}

	/**
	 * プリンタ名を設定します。<br>
	 * @param printerName プリンタ名
	 */
	public final void setPrinterName(String printerName) {
		this.printerName = printerName;
	}

	/**
	 * 自身が保持しているパラメータオブジェクトを指定されたJRExporterに対して設定します。<br>
	 * @param exporter JRExporterオブジェクト
	 */
	protected abstract void fillParameter(JRExporter exporter);

	/**
	 * エクスポート処理を行います。<br>
	 * サブクラスにおいてエクスポートを行う際のラップ処理を追加する為に設けられました。<br>
	 * @param exporter エクスポーターオブジェクト
	 * @throws 処理時に予期せぬ例外が発生した場合にスローされます
	 */
	protected void exportProcessWrapper(JRExporter exporter) throws Throwable {
		exporter.exportReport();
	}

	/**
	 * 自身が保持する情報からレポートを生成し、byte配列として出力します。<br>
	 * @param exporter JRExporterオブジェクト
	 * @return レポートオブジェクト(byte配列)
	 */
	public final byte[] write(JRExporter exporter) {
		try {
			// パラメータ設定
			fillParameter(exporter);

			// 出力処理
			ByteArrayOutputStream stream = new ByteArrayOutputStream();
			exporter.setParameter(JRExporterParameter.OUTPUT_STREAM, stream);
			exportProcessWrapper(exporter);
			byte[] bytes = stream.toByteArray();
			stream.close();

			return bytes;
		} catch (JasperCriticalException e) {
			throw e;
		} catch (Throwable e) {
			throw new JasperCriticalException("FJPR00008", e);
		}
	}

	/**
	 * 自身が保持する情報からレポートを生成し、指定されたFileオブジェクトに出力します。<br>
	 * @param exporter JRExporterオブジェクト
	 * @param file 出力先Fileオブジェクト
	 */
	public final void write(JRExporter exporter, File file) {
		try {
			// パラメータ設定
			fillParameter(exporter);

			// 出力処理
			exporter.setParameter(JRExporterParameter.OUTPUT_FILE, file);
			exportProcessWrapper(exporter);
		} catch (JasperCriticalException e) {
			throw e;
		} catch (Throwable e) {
			throw new JasperCriticalException("FJPR00008", e);
		}
	}

	/**
	 * 自身が保持する情報からレポートを生成し、指定されたファイルに出力します。<br>
	 * @param exporter JRExporterオブジェクト
	 * @param filename 出力先ファイル名
	 */
	public final void write(JRExporter exporter, String filename) {
		try {
			// パラメータ設定
			fillParameter(exporter);

			// 出力処理
			exporter.setParameter(JRExporterParameter.OUTPUT_FILE_NAME, filename);
			exportProcessWrapper(exporter);
		} catch (JasperCriticalException e) {
			throw e;
		} catch (Throwable e) {
			throw new JasperCriticalException("FJPR00008", e);
		}
	}

	/**
	 * 自身が保持する情報からレポートを生成し、指定されたOutputStreamオブジェクトに出力します。<br>
	 * OutputStreamオブジェクトに対するクローズ処理は行いません。<br>
	 * @param exporter JRExporterオブジェクト
	 * @param stream OutputStreamオブジェクト
	 */
	public final void write(JRExporter exporter, OutputStream stream) {
		try {
			// パラメータ設定
			fillParameter(exporter);

			// 出力処理
			exporter.setParameter(JRExporterParameter.OUTPUT_STREAM, stream);
			exportProcessWrapper(exporter);
		} catch (JasperCriticalException e) {
			throw e;
		} catch (Throwable e) {
			throw new JasperCriticalException("FJPR00008", e);
		}
	}

	/**
	 * 自身が保持する情報からレポートを生成し、指定されたWriterオブジェクトに出力します。<br>
	 * Writerオブジェクトに対するクローズ処理は行いません。<br>
	 * @param exporter JRExporterオブジェクト
	 * @param writer Writerオブジェクト
	 */
	public final void write(JRExporter exporter, Writer writer) {
		try {
			// パラメータ設定
			fillParameter(exporter);

			// 出力処理
			exporter.setParameter(JRExporterParameter.OUTPUT_WRITER, writer);
			exportProcessWrapper(exporter);
		} catch (JasperCriticalException e) {
			throw e;
		} catch (Throwable e) {
			throw new JasperCriticalException("FJPR00008", e);
		}
	}

	/**
	 * 自身が保持する情報からレポートを生成し、byte配列として出力します。<br>
	 * @param type 出力タイプ
	 * @return レポートオブジェクト(byte配列)
	 */
	public final byte[] write(JasperExportType type, Map<JRExporterParameter, Object> exporterPrameter) {
		JasperExporterFactory factory = new JasperExporterFactory(type, exporterPrameter);
		factory.setPrinterName(printerName);
		JRExporter exporter = factory.create();
		return write(exporter);
	}

	/**
	 * 自身が保持する情報からレポートを生成し、指定されたFileオブジェクトに出力します。<br>
	 * @param type 出力タイプ
	 * @param file 出力先Fileオブジェクト
	 */
	public final void write(JasperExportType type, Map<JRExporterParameter, Object> exporterPrameter, File file) {
		JasperExporterFactory factory = new JasperExporterFactory(type, exporterPrameter);
		factory.setPrinterName(printerName);
		JRExporter exporter = factory.create();
		write(exporter, file);
	}

	/**
	 * 自身が保持する情報からレポートを生成し、指定されたファイルに出力します。<br>
	 * @param type 出力タイプ
	 * @param filename 出力先ファイル名
	 */
	public final void write(JasperExportType type, Map<JRExporterParameter, Object> exporterPrameter, String filename) {
		JasperExporterFactory factory = new JasperExporterFactory(type, exporterPrameter);
		factory.setPrinterName(printerName);
		JRExporter exporter = factory.create();
		write(exporter, filename);
	}

	/**
	 * 自身が保持する情報からレポートを生成し、指定されたOutputStreamオブジェクトに出力します。<br>
	 * OutputStreamオブジェクトに対するクローズ処理は行いません。<br>
	 * @param type 出力タイプ
	 * @param stream OutputStreamオブジェクト
	 */
	public final void write(JasperExportType type, Map<JRExporterParameter, Object> exporterPrameter, OutputStream stream) {
		JasperExporterFactory factory = new JasperExporterFactory(type, exporterPrameter);
		factory.setPrinterName(printerName);
		JRExporter exporter = factory.create();
		write(exporter, stream);
	}

	/**
	 * 自身が保持する情報からレポートを生成し、指定されたWriterオブジェクトに出力します。<br>
	 * Writerオブジェクトに対するクローズ処理は行いません。<br>
	 * @param type 出力タイプ
	 * @param writer Writerオブジェクト
	 */
	public final void write(JasperExportType type, Map<JRExporterParameter, Object> exporterPrameter, Writer writer) {
		JasperExporterFactory factory = new JasperExporterFactory(type, exporterPrameter);
		factory.setPrinterName(printerName);
		JRExporter exporter = factory.create();
		write(exporter, writer);
	}

	/**
	 * 自身が保持する情報からレポートを生成し、byte配列として出力します。<br>
	 * @param type 出力タイプ
	 * @return レポートオブジェクト(byte配列)
	 */
	public final byte[] write(JasperExportType type) {
		return write(type, new HashMap<JRExporterParameter, Object>());
	}

	/**
	 * 自身が保持する情報からレポートを生成し、指定されたFileオブジェクトに出力します。<br>
	 * @param type 出力タイプ
	 * @param file 出力先Fileオブジェクト
	 */
	public final void write(JasperExportType type, File file) {
		write(type, new HashMap<JRExporterParameter, Object>(), file);
	}

	/**
	 * 自身が保持する情報からレポートを生成し、指定されたファイルに出力します。<br>
	 * @param type 出力タイプ
	 * @param filename 出力先ファイル名
	 */
	public final void write(JasperExportType type, String filename) {
		write(type, new HashMap<JRExporterParameter, Object>(), filename);
	}

	/**
	 * 自身が保持する情報からレポートを生成し、指定されたOutputStreamオブジェクトに出力します。<br>
	 * OutputStreamオブジェクトに対するクローズ処理は行いません。<br>
	 * @param type 出力タイプ
	 * @param stream OutputStreamオブジェクト
	 */
	public final void write(JasperExportType type, OutputStream stream) {
		write(type, new HashMap<JRExporterParameter, Object>(), stream);
	}

	/**
	 * 自身が保持する情報からレポートを生成し、指定されたWriterオブジェクトに出力します。<br>
	 * Writerオブジェクトに対するクローズ処理は行いません。<br>
	 * @param type 出力タイプ
	 * @param writer Writerオブジェクト
	 */
	public final void write(JasperExportType type, Writer writer) {
		write(type, new HashMap<JRExporterParameter, Object>(), writer);
	}
}
