package org.phosphoresce.webcore.ext.hibernate4.transaction;

import java.lang.reflect.Constructor;
import java.util.Date;
import java.util.List;

import org.hibernate.LockMode;
import org.hibernate.LockOptions;
import org.phosphoresce.webcore.ext.hibernate4.exception.HibernateTransactionException;
import org.phosphoresce.webcore.ext.hibernate4.model.enums.DeleteType;

/**
 * サービス上位抽象クラス<br>
 * <br>
 * サービスオブジェクトの作成単位を任意に作成可能とする為、エンティティ等でのジェネリクス制限は設けられていません。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2013/02/01	Kitagawa		新規作成
 *-->
 */
public abstract class AbstractHibernateService {

	/** 処理コンテナオブジェクト */
	private HibernateTransactionContext process;

	/**
	 * コンストラクタ<br>
	 * @param process 処理コンテナオブジェクト
	 */
	protected AbstractHibernateService(HibernateTransactionContext process) {
		super();
		this.process = process;
	}

	/**
	 * データベースアクセスオブジェクトクラスインスタンスを生成します。<br>
	 * 生成されたインスタンスでのデータベースアクセスはロック処理が行われません。<br>
	 * 更新中のレコードに対して取得処理を行う場合、更新前のレコードキャッシュから取得が行われます。<br>
	 * @param clazz データベースアクセスオブジェクトクラス
	 * @return クラスインスタンス
	 */
	protected final <E extends AbstractHibernateEntity, D extends AbstractHibernateDAO<E>> D createDAO(Class<D> clazz) {
		try {
			Constructor<D> constructor = clazz.getConstructor(new Class[] { HibernateTransactionManager.class });
			D instance = constructor.newInstance(new Object[] { process.getTransaction() });
			return instance;
		} catch (Throwable e) {
			throw new HibernateTransactionException("FHBN00012", e);
		}
	}

	/**
	 * エンティティに対して楽観的排他を実施します。<br>
	 * @param entity エンティティオブジェクト
	 */
	protected final void lockOptimistic(AbstractHibernateEntity entity) {
		process.getTransaction().getSession().buildLockRequest(LockOptions.NONE).setLockMode(LockMode.OPTIMISTIC).lock(entity);
	}

	/**
	 * エンティティに対して悲観的排他を実施します。<br>
	 * @param entity エンティティオブジェクト
	 */
	protected final void lockPessimistic(AbstractHibernateEntity entity) {
		process.getTransaction().getSession().buildLockRequest(LockOptions.NONE).setLockMode(LockMode.PESSIMISTIC_WRITE).lock(entity);
	}

	/**
	 * エンティティに対して悲観的排他を実施します。<br>
	 * @param entity エンティティオブジェクト
	 */
	protected final void lockPessimisticNoWait(AbstractHibernateEntity entity) {
		process.getTransaction().getSession().buildLockRequest(LockOptions.NONE).setLockMode(LockMode.UPGRADE_NOWAIT).lock(entity);
	}

	/**
	 * エンティティを保存します。<br>
	 * @param entity エンティティオブジェクト
	 */
	protected final void save(AbstractHibernateEntity entity) {
		if (entity != null) {
			if (entity.getEnrollDate() == null) {
				entity.setEnrollUser(process.getProcessUser());
				entity.setEnrollDate(new Date(process.getProcessTime()));
			}
			entity.setModifyUser(process.getProcessUser());
			entity.setModifyDate(new Date(process.getProcessTime()));
			entity.validate();
			process.getTransaction().getSession().saveOrUpdate(entity);
		}
	}

	/**
	 * エンティティを保存します。<br>
	 * @param entities エンティティオブジェクト
	 */
	protected final <E extends AbstractHibernateEntity> void save(List<E> entities) {
		if (entities != null) {
			for (AbstractHibernateEntity entity : entities) {
				save(entity);
			}
		}
	}

	/**
	 * エンティティを論理削除します。<br>
	 * @param entity エンティティオブジェクト
	 */
	protected final void delete(AbstractHibernateEntity entity) {
		if (entity != null) {
			if (entity.getEnrollDate() == null) {
				entity.setEnrollUser(process.getProcessUser());
				entity.setEnrollDate(new Date(process.getProcessTime()));
			}
			entity.setModifyUser(process.getProcessUser());
			entity.setModifyDate(new Date(process.getProcessTime()));
			entity.setDeleteUser(process.getProcessUser());
			entity.setDeleteDate(new Date(process.getProcessTime()));
			entity.setDeleteType(DeleteType.Deleted);
			entity.validate();
			process.getTransaction().getSession().saveOrUpdate(entity);
		}
	}

	/**
	 * エンティティを論理削除します。<br>
	 * @param entities エンティティオブジェクト
	 */
	protected final <E extends AbstractHibernateEntity> void delete(List<E> entities) {
		if (entities != null) {
			for (AbstractHibernateEntity entity : entities) {
				delete(entity);
			}
		}
	}

	/**
	 * エンティティを物理削除します。<br>
	 * @param entity エンティティオブジェクト
	 */
	protected final void erase(AbstractHibernateEntity entity) {
		if (entity != null) {
			process.getTransaction().getSession().delete(entity);
		}
	}

	/**
	 * エンティティを論理削除します。<br>
	 * @param entities エンティティオブジェクト
	 */
	protected final <E extends AbstractHibernateEntity> void erase(List<E> entities) {
		if (entities != null) {
			for (AbstractHibernateEntity entity : entities) {
				delete(entity);
			}
		}
	}
}
