package org.phosphoresce.webcore.ext.hibernate4;

import org.dom4j.Document;
import org.hibernate.cfg.Configuration;
import org.phosphoresce.lib.commons.util.DOM4JUtil;
import org.phosphoresce.lib.commons.util.PropertiesUtil;
import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.lib.jod.JODSOfficeProcessManager;
import org.phosphoresce.webcore.core.Environment;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.exception.WebcoreConfigurationException;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.phosphoresce.webcore.ext.hibernate4.exception.HibernateConfigureException;
import org.phosphoresce.webcore.ext.hibernate4.transaction.TransactionFactory;
import org.slf4j.LoggerFactory;

/**
 * Hibernateアドオン環境初期化クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2011/12/24	Kitagawa		新規作成
 *-->
 */
public class HibernateConfiguration implements HibernateConstants {

	/** ロガーオブジェクト */
	private static CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(HibernateConfiguration.class));

	/**
	 * コンストラクタ<br>
	 */
	private HibernateConfiguration() {
		super();
	}

	/**
	 * Hibernateアドオンの初期化処理を行います。<br>
	 * 当初期化処理はアプリケーション共通初期化処理が完了されていることが前提となります。<br>
	 */
	public synchronized static void configure() {
		try {
			log.output("FHBN00001");

			// ディフォルト定義分初期化処理
			configure(null);
			log.output("FHBN00004");

			// 名前空間定義分初期化処理
			String[] namespaces = Config.getStrings(HibernateConfigName.HIBERNATE_NAMESPACES);
			for (String namespace : namespaces) {
				configure(namespace);
				log.output("FHBN00005", namespace);
			}

			log.output("FHBN00002");
		} catch (Throwable e) {
			throw new WebcoreConfigurationException("FHBN00006", e);
		}
	}

	/**
	 * Hibernateアドオンの初期化処理を行います。<br>
	 * 当初期化処理はアプリケーション共通初期化処理が完了されていることが前提となります。<br>
	 * @param namespace 名前空間文字列
	 */
	private synchronized static void configure(String namespace) {
		try {
			String suffix = "";
			if (!StringUtil.isEmpty(namespace)) {
				suffix += "." + namespace;
			}

			/*
			 * 基底環境設定定義ロード
			 */
			String setting = Config.getString(HibernateConfigName.HIBERNATE_CONFIG.suffixedName(suffix));
			Configuration configuration = new Configuration();
			if (setting.toLowerCase().endsWith(CONFIG_XML_EXTENSION)) {
				Document document = DOM4JUtil.loadByDOM4J(setting);
				configuration.configure(DOM4JUtil.convertW3C(Environment.bindEnv(document)));
			} else if (setting.toLowerCase().endsWith(CONFIG_PROPERTIES_EXTENSION)) {
				configuration.setProperties(Environment.bindEnv(PropertiesUtil.load(setting)));
			}

			/*
			 * アノテーションモデルクラス追加
			 */
			for (String modelClass : Config.getString(HibernateConfigName.HIBERNATE_ANNOTATION_CLASSES.suffixedName(suffix)).split(",")) {
				if (!StringUtil.isEmpty(modelClass) && !modelClass.trim().startsWith("#")) {
					configuration.addAnnotatedClass(Class.forName(modelClass.trim()));
					log.output("FHBN00023", modelClass.trim());
				}
			}

			/*
			 * トランザクションファクトリクラス登録
			 */
			if (!StringUtil.isEmpty(namespace)) {
				TransactionFactory.initialize(namespace, configuration);
			} else {
				TransactionFactory.initialize(configuration);
			}
		} catch (Throwable e) {
			throw new HibernateConfigureException("FHBN00007", e);
		}
	}

	/**
	 * Hibernateアドオンの破棄処理を実行します。<br>
	 * @throws Throwable 終了処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void destroy() throws Throwable {
		JODSOfficeProcessManager.destroy();
		log.output("FHBN00003");
	}
}
