package org.phosphoresce.webcore.core.logger;

import org.phosphoresce.webcore.core.config.StringResource;
import org.phosphoresce.webcore.core.config.StringResourceLevel;
import org.slf4j.Logger;

/**
 * メッセージコードロガークラス<br>
 * <br>
 * コード定義されているメッセージをロギングします。<br>
 * また、以下の命名規約に沿ったメッセージコードにより自動で出力レベルが選択されます。<br>
 * 以下に合致しないレベルの場合は常にトレースレベルでログ出力が行われます。<br>
 * <ul>
 * <li>*****-I : 情報レベル<li>
 * <li>*****-E : エラーレベル<li>
 * <li>*****-W : 警告レベル<li>
 * <li>*****-N : 連絡レベル(ログレベルはトレースとなります)<li>
 * <li>*****-D : デバッグレベル<li>
 * <li>*****-T : トレースレベル<li>
 * </ul>
 * <br>
 * <b>
 * 当クラスを利用したログ出力を行う場合、ラップされたロガーオブジェクトによる呼び出し階層を補正するため、
 * logbackに対するエンコーダークラスにCallerOffsetPatternLayoutEncoderを利用する必要があります。<br>
 * </b>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2013/02/08	Kitagawa		新規作成
 *-->
 */
public class CodeConvertLogger {

	/** ロガーオブジェクト */
	private Logger logger;

	/**
	 * コンストラクタ<br>
	 * @param logger
	 */
	public CodeConvertLogger(Logger logger) {
		super();
		this.logger = logger;
	}

	/**
	 * メッセージログ出力処理を行います。<br>
	 * ログ出力時の呼び出し階層を一定とするために内部でメソッドの切り分けが行われています。<br>
	 * @param e 例外オブジェクト
	 * @param code メッセージコード
	 * @param binds バインドオブジェクト配列
	 */
	private void outputCore(Throwable e, String code, Object... binds) {
		if (logger == null) {
			return;
		}
		StringResourceLevel level = StringResourceLevel.getDefinitionLevel(code);
		if (StringResourceLevel.INFO.equals(level)) {
			if (logger.isInfoEnabled()) {
				if (e != null) {
					logger.info(getMessage(code, binds), e);
				} else {
					logger.info(getMessage(code, binds));
				}
			}
		} else if (StringResourceLevel.ERROR.equals(level)) {
			if (logger.isErrorEnabled()) {
				if (e != null) {
					logger.error(getMessage(code, binds), e);
				} else {
					logger.error(getMessage(code, binds));
				}
			}
		} else if (StringResourceLevel.WARN.equals(level)) {
			if (logger.isWarnEnabled()) {
				if (e != null) {
					logger.warn(getMessage(code, binds), e);
				} else {
					logger.warn(getMessage(code, binds));
				}
			}
		} else if (StringResourceLevel.NOTICE.equals(level)) {
			if (logger.isTraceEnabled()) {
				if (e != null) {
					logger.trace(getMessage(code, binds), e);
				} else {
					logger.trace(getMessage(code, binds));
				}
			}
		} else if (StringResourceLevel.DEBUG.equals(level)) {
			if (logger.isDebugEnabled()) {
				if (e != null) {
					logger.debug(getMessage(code, binds), e);
				} else {
					logger.debug(getMessage(code, binds));
				}
			}
		} else if (StringResourceLevel.TRACE.equals(level)) {
			if (logger.isTraceEnabled()) {
				if (e != null) {
					logger.trace(getMessage(code, binds), e);
				} else {
					logger.trace(getMessage(code, binds));
				}
			}
		} else {
			if (logger.isTraceEnabled()) {
				if (e != null) {
					logger.trace(getMessage(code, binds), e);
				} else {
					logger.trace(getMessage(code, binds));
				}
			}
		}
	}

	/**
	 * 指定されたコードとバインドオブジェクトでメッセージを取得します。<br>
	 * @param code メッセージコード
	 * @param binds バインドオブジェクト配列
	 * @return メッセージ文字列
	 */
	private String getMessage(String code, Object... binds) {
		if (code == null) {
			return "--------- | < Empty message code >";
		} else {
			return code + " | " + StringResource.getString(code, binds);
		}
	}

	/**
	 * 指定されたコードとバインドオブジェクトでメッセージログを出力します。<br>
	 * @param e 例外オブジェクト
	 * @param code メッセージコード
	 * @param binds バインドオブジェクト配列
	 */
	public void output(Throwable e, String code, Object... binds) {
		outputCore(e, code, binds);
	}

	/**
	 * 指定されたコードとバインドオブジェクトでメッセージログを出力します。<br>
	 * @param code メッセージコード
	 * @param binds バインドオブジェクト配列
	 */
	public void output(String code, Object... binds) {
		outputCore(null, code, binds);
	}

	/**
	 * 指定されたコードでメッセージログを出力します。<br>
	 * @param e 例外オブジェクト
	 * @param code メッセージコード
	 */
	public void output(Throwable e, String code) {
		outputCore(e, code, (Object[]) null);
	}

	/**
	 * 指定されたコードでメッセージログを出力します。<br>
	 * @param code メッセージコード
	 */
	public void output(String code) {
		outputCore(null, code, (Object[]) null);
	}
}
