package org.phosphoresce.webcore.core;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.NoSuchElementException;

import org.phosphoresce.lib.commons.util.StringUtil;

/**
 * アプリケーション環境設定定義名定数列挙クラス<br>
 * <br>
 * 各フレームワークモジュールの環境定義アクセス用のプロパティキーを定義するクラスです。<br>
 * サブモジュール環境定義名列挙クラスは必ずこれを継承して作成し、当コンストラクタを呼び出します。<br>
 * 環境定義キーの変更時等でのリファクタリングを可能とする為に、文字列型としては定義されません。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/03	Kitagawa		新規作成
 * 2013/07/04	Kitagawa		定数インタフェースから列挙形式のクラスに変更(Configクラスアクセス時に必ず列挙クラス定義を行う制限を設けさせる為)
 *-->
 */
public class ConfigName implements Serializable {

	/** アプリケーションタイトル定義 */
	public static final ConfigName COMMON_APPLICATION_TITLE = new ConfigName("common.application.title");

	/** アプリケーションタイトルイメージURLパス定義 */
	public static final ConfigName COMMON_APPLICATION_TITLE_IMAGE_URL = new ConfigName("common.application.title.image.url");

	/** アプリケーションフッター定義 */
	public static final ConfigName COMMON_APPLICATION_FOOTER = new ConfigName("common.application.footer");

	/** ディフォルトキャラクタセット定義 */
	public static final ConfigName COMMON_CHARSET_DEFAULT = new ConfigName("common.charset.default");

	/** コンテンツ絶対パス定義 */
	public static final ConfigName COMMON_ABSOLUTE_CONTENT_PATH = new ConfigName("common.absolute.content.path");

	/** WEB-INF絶対パス定義 */
	public static final ConfigName COMMON_ABSOLUTE_WEBINF_PATH = new ConfigName("common.absolute.webinf.path");

	/** Javascript用サーバーメッセージ利用サーブレットパス定義 */
	public static final ConfigName COMMON_SERVLET_PATH_JS_MESSAGE = new ConfigName("common.webxml.jsmsg.url");

	/** サーバー静的リソースリクエストサーブレットパス定義 */
	public static final ConfigName COMMON_SERVLET_PATH_STATIC_RESOURCE = new ConfigName("common.webxml.staticres.url");

	/** 拡張初期化クラス定義 */
	public static final ConfigName COMMON_INITIALIZE_CLASSES = new ConfigName("common.initialize.classes");

	/** 追加読み込みプロパティ定義 */
	public static final ConfigName COMMON_RESOURCE_CONFIG_EXTENDS = new ConfigName("common.resource.config.extends");

	/** 文字列定義リソースパス定義 */
	public static final ConfigName COMMON_RESOURCE_STRING = new ConfigName("common.resource.string");

	/** MIMEタイプ対応リソース定義 */
	public static final ConfigName COMMON_RESROUCE_MIME_TYPE = new ConfigName("common.resource.mimetype");

	/** Logback環境設定リソースパス定義 */
	public static final ConfigName COMMON_RESOURCE_LOGBACK = new ConfigName("common.resource.logback.config");

	/** 汎用サーバダウンロードリソースパス定義 */
	public static final ConfigName COMMON_GENERAL_DOWNLOAD_RESOURCE = new ConfigName("common.resource.download.general");

	/** 汎用サーバアップロードリソースパス定義 */
	public static final ConfigName COMMON_GENERAL_UPLOAD_RESOURCE = new ConfigName("common.resource.upload.general");

	/** 汎用サーバテンポラリリソースパス定義 */
	public static final ConfigName COMMON_GENERAL_TEMPORARY_RESOURCE = new ConfigName("common.resource.temporary.general");

	/** 静的サーバリソースパス定義 */
	public static final ConfigName COMMON_STATIC_RESOURCE = new ConfigName("common.resource.static");

	/** メッセージ出力時メッセージコード出力フラグ */
	public static final ConfigName COMMON_MESSAGE_SHOWCODE = new ConfigName("common.message.showCode");

	/** アプリケーションガベージコレクションデーモン定義 */
	public static final ConfigName COMMON_DAEMON_GC = new ConfigName("common.daemon.gc");

	/** アプリケーションガベージコレクションデーモン実行間隔(ミリ秒定義 */
	public static final ConfigName COMMON_DAEMON_GC_INTERVAL = new ConfigName("common.daemon.gc.interval");

	/** Cookie有効時間(秒 */
	public static final ConfigName COMMON_SERVLET_COOKIE_TIMEOUT = new ConfigName("common.servlet.cookie.timeout");

	/** サーブレットページ配置基底パス定義 */
	public static final ConfigName COMMON_JSP_PATH = new ConfigName("common.jsp.path");

	/** システムエラーページパス定義 */
	public static final ConfigName COMMON_JSP_SYSTEMERROR = new ConfigName("common.jsp.systemerror");

	/** メッセージページパス定義 */
	public static final ConfigName COMMON_JSP_MESSAGE = new ConfigName("common.jsp.message");

	/** プロパティキー */
	private String key;

	/** インスタンスキャッシュ */
	private static Map<String, ConfigName> instances;

	/**
	 * コンストラクタ<br>
	 * @param key プロパティキー
	 */
	protected ConfigName(String key) {
		if (key == null) {
			throw new NullPointerException();
		}
		this.key = key;
		if (instances == null) {
			instances = new HashMap<String, ConfigName>();
		}
		synchronized (instances) {
			instances.put(key, this);
		}
	}

	/**
	 * プロパティキーを取得します。<br>
	 * @return プロパティキー
	 */
	public final String key() {
		return key;
	}

	/**
	 * 指定されたプロパティキーの定義インスタンスを取得します。<br>
	 * @param key プロパティキー
	 * @return 定義インスタンス
	 */
	public static final ConfigName get(String key) {
		synchronized (instances) {
			if (instances == null || !instances.containsKey(key)) {
				throw new NoSuchElementException();
			}
			return instances.get(key);
		}
	}

	/**
	 * サフィックスを装飾した定義インスタンスを提供します。<br>
	 * @param suffix サフィックス文字列
	 * @return 定義インスタンス
	 */
	public final ConfigName suffixedName(String suffix) {
		if (StringUtil.isEmpty(suffix)) {
			return this;
		} else {
			String targetKey = key + suffix;
			synchronized (instances) {
				if (instances.containsKey(targetKey)) {
					return instances.get(targetKey);
				}
			}
			return new ConfigName(key + suffix);
		}
	}

	/**
	 * クラス情報を文字列で定義します。<br>
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return key;
	}
}
