package org.phosphoresce.lib.poi.print;

import java.io.Serializable;

import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.ss.usermodel.PrintSetup;
import org.apache.poi.ss.usermodel.Sheet;
import org.phosphoresce.lib.poi.PoiGlobal;
import org.phosphoresce.lib.poi.adapter.PoiSheet;

/**
 * シート印刷環境設定クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日       更新者          更新内容
 * 2006/01/25	Kitagawa		新規作成
 * 2012/07/09	Kitagawa		バージョン2にアップデート
 *-->
 */
public final class PoiPrintConfig implements PoiGlobal, Serializable {

	/** 印刷方向(true : 横置 / false : 縦置(ディフォルト)) */
	private boolean landscape = false;

	/** 拡大縮小印刷方法(true : 指定縦横数改ページ印刷方式 / false : 拡大縮小倍率指定方式(ディフォルト) */
	private boolean autobreaks = false;

	/** 印刷倍率(ディフォルト値:100) */
	private short scale = 100;

	/** ページ中央印刷(水平方向)方法(true : 水平方向中央寄せ印刷 / false : 水平方向左寄せ印刷(ディフォルト)) */
	private boolean horizontallyCenter = false;

	/** ページ中央印刷(垂直方向)方法(true : 垂直方向中央寄せ印刷 / false : 垂直方向上寄せ印刷(ディフォルト)) */
	private boolean verticallyCenter = false;

	/** ページ印刷マージン上部(ディフォルト値 : 2.0) */
	private double marginOfTop = 2.0;

	/** ページ印刷マージン下部(ディフォルト値 : 2.0) */
	private double marginOfBottom = 2.0;

	/** ページ印刷マージン左部(ディフォルト値 : 2.0) */
	private double marginOfLeft = 2.0;

	/** ページ印刷マージン右部(ディフォルト値 : 2.0) */
	private double marginOfRight = 2.0;

	/** ヘッダー左部文字列(ディフォルト値 : なし) */
	private PoiHeaderFooter headerLeftString = null;

	/** ヘッダー中央文字列(ディフォルト値 : なし) */
	private PoiHeaderFooter headerCenterString = null;

	/** ヘッダー右部文字列(ディフォルト値 : なし) */
	private PoiHeaderFooter headerRightString = null;

	/** フッター左部文字列(ディフォルト値 : なし) */
	private PoiHeaderFooter footerLeftString = null;

	/** フッター中央文字列(ディフォルト値 : なし) */
	private PoiHeaderFooter footerCenterString = null;

	/** フッター右部文字列(ディフォルト値 : なし) */
	private PoiHeaderFooter footerRightString = null;

	/** ヘッダーマージン(ディフォルト値 : 1.0) */
	private double marginOfHeader = 1.0;

	/** フッターマージン(ディフォルト値 : 1.0) */
	private double marginOfFooter = 1.0;

	/** 指定縦横数改ページ印刷方式での縦方向のページ数(ディフォルト値 : 1) */
	private short fitPageVerticalCount = 1;

	/** 指定縦横数改ページ印刷方式での横方向のページ数(ディフォルト値 : 1) */
	private short fitPageHorizontalCount = 1;

	/** 印刷部数(ディフォルト値 : 1) */
	private short copies = 1;

	/** 簡易印刷設定(true : 簡易印刷 / false : 通常印刷(ディフォルト)) */
	private boolean draft = false;

	/** 印刷品質(dpi)(ディフォルト値 : 600) */
	private short dpi = 600;

	/** ページ印刷方向を上から下とする(true : 上から下へ / false : 左から右へ(ディフォルト)) */
	private boolean printTopToDown = false;

	/** 白黒印刷設定(true : 白黒印刷 / false : 通常印刷(ディフォルト)) */
	private boolean monoPrint = false;

	/** コメント印刷設定(true : コメントを印刷 / false : コメント未印刷(ディフォルト)) */
	private boolean printComment = false;

	/** 先頭印刷ページ(ディフォルト値 : 1(先頭ページ)) */
	private short startPage = 1;

	/** 用紙サイズ(ディフォルト値 : org.apache.poi.ss.usermodel.PrintSetup.A4_PAPERSIZE) */
	private short paperSize = PrintSetup.A4_PAPERSIZE;

	/**
	 * コンストラクタ<br>
	 */
	public PoiPrintConfig() {
		super();
	}

	/**
	 * シートオブジェクトから印刷設定オブジェクトを生成します。<br>
	 * @param sheet シートオブジェクト
	 * @return 印刷設定オブジェクト
	 */
	public static PoiPrintConfig create(Sheet sheet) {
		final double MARGIN_COEFFICIENT = 2.54;

		PrintSetup setup = sheet.getPrintSetup();
		PoiPrintConfig config = new PoiPrintConfig();
		config.landscape = setup.getLandscape();
		config.autobreaks = sheet.getAutobreaks();
		config.scale = setup.getScale();
		config.horizontallyCenter = sheet.getHorizontallyCenter();
		config.verticallyCenter = sheet.getVerticallyCenter();
		config.marginOfTop = sheet.getMargin(HSSFSheet.TopMargin) * MARGIN_COEFFICIENT;
		config.marginOfBottom = sheet.getMargin(HSSFSheet.BottomMargin) * MARGIN_COEFFICIENT;
		config.marginOfLeft = sheet.getMargin(HSSFSheet.LeftMargin) * MARGIN_COEFFICIENT;
		config.marginOfRight = sheet.getMargin(HSSFSheet.RightMargin) * MARGIN_COEFFICIENT;
		config.headerLeftString = new PoiHeaderFooter(sheet.getHeader().getLeft());
		config.headerCenterString = new PoiHeaderFooter(sheet.getHeader().getCenter());
		config.headerRightString = new PoiHeaderFooter(sheet.getHeader().getRight());
		config.footerLeftString = new PoiHeaderFooter(sheet.getFooter().getLeft());
		config.footerCenterString = new PoiHeaderFooter(sheet.getFooter().getCenter());
		config.footerRightString = new PoiHeaderFooter(sheet.getFooter().getRight());
		config.marginOfHeader = setup.getHeaderMargin() * MARGIN_COEFFICIENT;
		config.marginOfFooter = setup.getFooterMargin() * MARGIN_COEFFICIENT;
		config.fitPageVerticalCount = setup.getFitHeight();
		config.fitPageHorizontalCount = setup.getFitWidth();
		config.copies = setup.getCopies();
		config.draft = setup.getDraft();
		config.dpi = setup.getHResolution();
		//config.dpi = setup.getVResolution();
		config.printTopToDown = setup.getLeftToRight();
		config.monoPrint = setup.getNoColor();
		config.printComment = setup.getNotes();
		config.startPage = setup.getPageStart();
		config.paperSize = setup.getPaperSize();

		return config;
	}

	/**
	 * シートに対して印刷設定を適用します。<br>
	 * @param sheet シートオブジェクト
	 */
	public void apply(PoiSheet sheet) {
		final double MARGIN_COEFFICIENT = 2.54;

		sheet.getPoiSheet().setAutobreaks(autobreaks);
		sheet.getPoiSheet().setHorizontallyCenter(horizontallyCenter);
		sheet.getPoiSheet().setVerticallyCenter(verticallyCenter);
		sheet.getPoiSheet().setMargin(HSSFSheet.TopMargin, marginOfTop / MARGIN_COEFFICIENT);
		sheet.getPoiSheet().setMargin(HSSFSheet.BottomMargin, marginOfBottom / MARGIN_COEFFICIENT);
		sheet.getPoiSheet().setMargin(HSSFSheet.LeftMargin, marginOfLeft / MARGIN_COEFFICIENT);
		sheet.getPoiSheet().setMargin(HSSFSheet.RightMargin, marginOfRight / MARGIN_COEFFICIENT);

		sheet.getPoiSheet().getHeader().setLeft(headerLeftString.toString());
		sheet.getPoiSheet().getHeader().setCenter(headerCenterString.toString());
		sheet.getPoiSheet().getHeader().setRight(headerRightString.toString());
		sheet.getPoiSheet().getFooter().setLeft(footerLeftString.toString());
		sheet.getPoiSheet().getFooter().setCenter(footerCenterString.toString());
		sheet.getPoiSheet().getFooter().setRight(footerRightString.toString());

		sheet.getPoiSheet().getPrintSetup().setHeaderMargin(marginOfHeader / MARGIN_COEFFICIENT);
		sheet.getPoiSheet().getPrintSetup().setFooterMargin(marginOfFooter / MARGIN_COEFFICIENT);
		sheet.getPoiSheet().getPrintSetup().setFitHeight(fitPageVerticalCount);
		sheet.getPoiSheet().getPrintSetup().setFitWidth(fitPageHorizontalCount);
		sheet.getPoiSheet().getPrintSetup().setLandscape(landscape);
		sheet.getPoiSheet().getPrintSetup().setScale(scale);
		sheet.getPoiSheet().getPrintSetup().setCopies(copies);
		sheet.getPoiSheet().getPrintSetup().setDraft(draft);
		sheet.getPoiSheet().getPrintSetup().setHResolution(dpi);
		sheet.getPoiSheet().getPrintSetup().setVResolution(dpi);
		sheet.getPoiSheet().getPrintSetup().setLeftToRight(printTopToDown);
		sheet.getPoiSheet().getPrintSetup().setNoColor(monoPrint);
		sheet.getPoiSheet().getPrintSetup().setNotes(printComment);
		sheet.getPoiSheet().getPrintSetup().setPageStart(startPage);
		sheet.getPoiSheet().getPrintSetup().setPaperSize(paperSize);
	}

	/**
	 * 印刷方向(true : 横置 / false : 縦置(ディフォルト))を取得します。<br>
	 * @return 印刷方向(true : 横置 / false : 縦置(ディフォルト))
	 */
	public boolean isLandscape() {
		return landscape;
	}

	/**
	 * 印刷方向(true : 横置 / false : 縦置(ディフォルト))を設定します。<br>
	 * @param landscape 印刷方向(true : 横置 / false : 縦置(ディフォルト))
	 */
	public void setLandscape(boolean landscape) {
		this.landscape = landscape;
	}

	/**
	 * 拡大縮小印刷方法(true : 指定縦横数改ページ印刷方式 / false : 拡大縮小倍率指定方式(ディフォルト)を取得します。<br>
	 * @return 拡大縮小印刷方法(true : 指定縦横数改ページ印刷方式 / false : 拡大縮小倍率指定方式(ディフォルト)
	 */
	public boolean isAutobreaks() {
		return autobreaks;
	}

	/**
	 * 拡大縮小印刷方法(true : 指定縦横数改ページ印刷方式 / false : 拡大縮小倍率指定方式(ディフォルト)を設定します。<br>
	 * @param autobreaks 拡大縮小印刷方法(true : 指定縦横数改ページ印刷方式 / false : 拡大縮小倍率指定方式(ディフォルト)
	 */
	public void setAutobreaks(boolean autobreaks) {
		this.autobreaks = autobreaks;
	}

	/**
	 * 印刷倍率(ディフォルト値:100)を取得します。<br>
	 * @return 印刷倍率(ディフォルト値:100)
	 */
	public short getScale() {
		return scale;
	}

	/**
	 * 印刷倍率(ディフォルト値:100)を設定します。<br>
	 * @param scale 印刷倍率(ディフォルト値:100)
	 */
	public void setScale(short scale) {
		this.scale = scale;
	}

	/**
	 * ページ中央印刷(水平方向)方法(true : 水平方向中央寄せ印刷 / false : 水平方向左寄せ印刷(ディフォルト))を取得します。<br>
	 * @return ページ中央印刷(水平方向)方法(true : 水平方向中央寄せ印刷 / false : 水平方向左寄せ印刷(ディフォルト))
	 */
	public boolean isHorizontallyCenter() {
		return horizontallyCenter;
	}

	/**
	 * ページ中央印刷(水平方向)方法(true : 水平方向中央寄せ印刷 / false : 水平方向左寄せ印刷(ディフォルト))を設定します。<br>
	 * @param horizontallyCenter ページ中央印刷(水平方向)方法(true : 水平方向中央寄せ印刷 / false : 水平方向左寄せ印刷(ディフォルト))
	 */
	public void setHorizontallyCenter(boolean horizontallyCenter) {
		this.horizontallyCenter = horizontallyCenter;
	}

	/**
	 * ページ中央印刷(垂直方向)方法(true : 垂直方向中央寄せ印刷 / false : 垂直方向上寄せ印刷(ディフォルト))を取得します。<br>
	 * @return ページ中央印刷(垂直方向)方法(true : 垂直方向中央寄せ印刷 / false : 垂直方向上寄せ印刷(ディフォルト))
	 */
	public boolean isVerticallyCenter() {
		return verticallyCenter;
	}

	/**
	 * ページ中央印刷(垂直方向)方法(true : 垂直方向中央寄せ印刷 / false : 垂直方向上寄せ印刷(ディフォルト))を設定します。<br>
	 * @param verticallyCenter ページ中央印刷(垂直方向)方法(true : 垂直方向中央寄せ印刷 / false : 垂直方向上寄せ印刷(ディフォルト))
	 */
	public void setVerticallyCenter(boolean verticallyCenter) {
		this.verticallyCenter = verticallyCenter;
	}

	/**
	 * ページ印刷マージン上部(ディフォルト値 : 2.0)を取得します。<br>
	 * @return ページ印刷マージン上部(ディフォルト値 : 2.0)
	 */
	public double getMarginOfTop() {
		return marginOfTop;
	}

	/**
	 * ページ印刷マージン上部(ディフォルト値 : 2.0)を設定します。<br>
	 * @param marginOfTop ページ印刷マージン上部(ディフォルト値 : 2.0)
	 */
	public void setMarginOfTop(double marginOfTop) {
		this.marginOfTop = marginOfTop;
	}

	/**
	 * ページ印刷マージン下部(ディフォルト値 : 2.0)を取得します。<br>
	 * @return ページ印刷マージン下部(ディフォルト値 : 2.0)
	 */
	public double getMarginOfBottom() {
		return marginOfBottom;
	}

	/**
	 * ページ印刷マージン下部(ディフォルト値 : 2.0)を設定します。<br>
	 * @param marginOfBottom ページ印刷マージン下部(ディフォルト値 : 2.0)
	 */
	public void setMarginOfBottom(double marginOfBottom) {
		this.marginOfBottom = marginOfBottom;
	}

	/**
	 * ページ印刷マージン左部(ディフォルト値 : 2.0)を取得します。<br>
	 * @return ページ印刷マージン左部(ディフォルト値 : 2.0)
	 */
	public double getMarginOfLeft() {
		return marginOfLeft;
	}

	/**
	 * ページ印刷マージン左部(ディフォルト値 : 2.0)を設定します。<br>
	 * @param marginOfLeft ページ印刷マージン左部(ディフォルト値 : 2.0)
	 */
	public void setMarginOfLeft(double marginOfLeft) {
		this.marginOfLeft = marginOfLeft;
	}

	/**
	 * ページ印刷マージン右部(ディフォルト値 : 2.0)を取得します。<br>
	 * @return ページ印刷マージン右部(ディフォルト値 : 2.0)
	 */
	public double getMarginOfRight() {
		return marginOfRight;
	}

	/**
	 * ページ印刷マージン右部(ディフォルト値 : 2.0)を設定します。<br>
	 * @param marginOfRight ページ印刷マージン右部(ディフォルト値 : 2.0)
	 */
	public void setMarginOfRight(double marginOfRight) {
		this.marginOfRight = marginOfRight;
	}

	/**
	 * ヘッダー左部文字列(ディフォルト値 : なし)を取得します。<br>
	 * @return ヘッダー左部文字列(ディフォルト値 : なし)
	 */
	public PoiHeaderFooter getHeaderLeftString() {
		return headerLeftString;
	}

	/**
	 * ヘッダー左部文字列(ディフォルト値 : なし)を設定します。<br>
	 * @param headerLeftString ヘッダー左部文字列(ディフォルト値 : なし)
	 */
	public void setHeaderLeftString(PoiHeaderFooter headerLeftString) {
		this.headerLeftString = headerLeftString;
	}

	/**
	 * ヘッダー中央文字列(ディフォルト値 : なし)を取得します。<br>
	 * @return ヘッダー中央文字列(ディフォルト値 : なし)
	 */
	public PoiHeaderFooter getHeaderCenterString() {
		return headerCenterString;
	}

	/**
	 * ヘッダー中央文字列(ディフォルト値 : なし)を設定します。<br>
	 * @param headerCenterString ヘッダー中央文字列(ディフォルト値 : なし)
	 */
	public void setHeaderCenterString(PoiHeaderFooter headerCenterString) {
		this.headerCenterString = headerCenterString;
	}

	/**
	 * ヘッダー右部文字列(ディフォルト値 : なし)を取得します。<br>
	 * @return ヘッダー右部文字列(ディフォルト値 : なし)
	 */
	public PoiHeaderFooter getHeaderRightString() {
		return headerRightString;
	}

	/**
	 * ヘッダー右部文字列(ディフォルト値 : なし)を設定します。<br>
	 * @param headerRightString ヘッダー右部文字列(ディフォルト値 : なし)
	 */
	public void setHeaderRightString(PoiHeaderFooter headerRightString) {
		this.headerRightString = headerRightString;
	}

	/**
	 * フッター左部文字列(ディフォルト値 : なし)を取得します。<br>
	 * @return フッター左部文字列(ディフォルト値 : なし)
	 */
	public PoiHeaderFooter getFooterLeftString() {
		return footerLeftString;
	}

	/**
	 * フッター左部文字列(ディフォルト値 : なし)を設定します。<br>
	 * @param footerLeftString フッター左部文字列(ディフォルト値 : なし)
	 */
	public void setFooterLeftString(PoiHeaderFooter footerLeftString) {
		this.footerLeftString = footerLeftString;
	}

	/**
	 * フッター中央文字列(ディフォルト値 : なし)を取得します。<br>
	 * @return フッター中央文字列(ディフォルト値 : なし)
	 */
	public PoiHeaderFooter getFooterCenterString() {
		return footerCenterString;
	}

	/**
	 * フッター中央文字列(ディフォルト値 : なし)を設定します。<br>
	 * @param footerCenterString フッター中央文字列(ディフォルト値 : なし)
	 */
	public void setFooterCenterString(PoiHeaderFooter footerCenterString) {
		this.footerCenterString = footerCenterString;
	}

	/**
	 * フッター右部文字列(ディフォルト値 : なし)を取得します。<br>
	 * @return フッター右部文字列(ディフォルト値 : なし)
	 */
	public PoiHeaderFooter getFooterRightString() {
		return footerRightString;
	}

	/**
	 * フッター右部文字列(ディフォルト値 : なし)を設定します。<br>
	 * @param footerRightString フッター右部文字列(ディフォルト値 : なし)
	 */
	public void setFooterRightString(PoiHeaderFooter footerRightString) {
		this.footerRightString = footerRightString;
	}

	/**
	 * ヘッダーマージン(ディフォルト値 : 1.0)を取得します。<br>
	 * @return ヘッダーマージン(ディフォルト値 : 1.0)
	 */
	public double getMarginOfHeader() {
		return marginOfHeader;
	}

	/**
	 * ヘッダーマージン(ディフォルト値 : 1.0)を設定します。<br>
	 * @param marginOfHeader ヘッダーマージン(ディフォルト値 : 1.0)
	 */
	public void setMarginOfHeader(double marginOfHeader) {
		this.marginOfHeader = marginOfHeader;
	}

	/**
	 * フッターマージン(ディフォルト値 : 1.0)を取得します。<br>
	 * @return フッターマージン(ディフォルト値 : 1.0)
	 */
	public double getMarginOfFooter() {
		return marginOfFooter;
	}

	/**
	 * フッターマージン(ディフォルト値 : 1.0)を設定します。<br>
	 * @param marginOfFooter フッターマージン(ディフォルト値 : 1.0)
	 */
	public void setMarginOfFooter(double marginOfFooter) {
		this.marginOfFooter = marginOfFooter;
	}

	/**
	 * 指定縦横数改ページ印刷方式での縦方向のページ数(ディフォルト値 : 1)を取得します。<br>
	 * @return 指定縦横数改ページ印刷方式での縦方向のページ数(ディフォルト値 : 1)
	 */
	public short getFitPageVerticalCount() {
		return fitPageVerticalCount;
	}

	/**
	 * 指定縦横数改ページ印刷方式での縦方向のページ数(ディフォルト値 : 1)を設定します。<br>
	 * @param fitPageVerticalCount 指定縦横数改ページ印刷方式での縦方向のページ数(ディフォルト値 : 1)
	 */
	public void setFitPageVerticalCount(short fitPageVerticalCount) {
		this.fitPageVerticalCount = fitPageVerticalCount;
	}

	/**
	 * 指定縦横数改ページ印刷方式での横方向のページ数(ディフォルト値 : 1)を取得します。<br>
	 * @return 指定縦横数改ページ印刷方式での横方向のページ数(ディフォルト値 : 1)
	 */
	public short getFitPageHorizontalCount() {
		return fitPageHorizontalCount;
	}

	/**
	 * 指定縦横数改ページ印刷方式での横方向のページ数(ディフォルト値 : 1)を設定します。<br>
	 * @param fitPageHorizontalCount 指定縦横数改ページ印刷方式での横方向のページ数(ディフォルト値 : 1)
	 */
	public void setFitPageHorizontalCount(short fitPageHorizontalCount) {
		this.fitPageHorizontalCount = fitPageHorizontalCount;
	}

	/**
	 * 印刷部数(ディフォルト値 : 1)を取得します。<br>
	 * @return 印刷部数(ディフォルト値 : 1)
	 */
	public short getCopies() {
		return copies;
	}

	/**
	 * 印刷部数(ディフォルト値 : 1)を設定します。<br>
	 * @param copies 印刷部数(ディフォルト値 : 1)
	 */
	public void setCopies(short copies) {
		this.copies = copies;
	}

	/**
	 * 簡易印刷設定(true : 簡易印刷 / false : 通常印刷(ディフォルト))を取得します。<br>
	 * @return 簡易印刷設定(true : 簡易印刷 / false : 通常印刷(ディフォルト))
	 */
	public boolean isDraft() {
		return draft;
	}

	/**
	 * 簡易印刷設定(true : 簡易印刷 / false : 通常印刷(ディフォルト))を設定します。<br>
	 * @param draft 簡易印刷設定(true : 簡易印刷 / false : 通常印刷(ディフォルト))
	 */
	public void setDraft(boolean draft) {
		this.draft = draft;
	}

	/**
	 * 印刷品質(dpi)(ディフォルト値 : 600)を取得します。<br>
	 * @return 印刷品質(dpi)(ディフォルト値 : 600)
	 */
	public short getDpi() {
		return dpi;
	}

	/**
	 * 印刷品質(dpi)(ディフォルト値 : 600)を設定します。<br>
	 * @param dpi 印刷品質(dpi)(ディフォルト値 : 600)
	 */
	public void setDpi(short dpi) {
		this.dpi = dpi;
	}

	/**
	 * ページ印刷方向を上から下とする(true : 上から下へ / false : 左から右へ(ディフォルト))を取得します。<br>
	 * @return ページ印刷方向を上から下とする(true : 上から下へ / false : 左から右へ(ディフォルト))
	 */
	public boolean isPrintTopToDown() {
		return printTopToDown;
	}

	/**
	 * ページ印刷方向を上から下とする(true : 上から下へ / false : 左から右へ(ディフォルト))を設定します。<br>
	 * @param printTopToDown ページ印刷方向を上から下とする(true : 上から下へ / false : 左から右へ(ディフォルト))
	 */
	public void setPrintTopToDown(boolean printTopToDown) {
		this.printTopToDown = printTopToDown;
	}

	/**
	 * 白黒印刷設定(true : 白黒印刷 / false : 通常印刷(ディフォルト))を取得します。<br>
	 * @return 白黒印刷設定(true : 白黒印刷 / false : 通常印刷(ディフォルト))
	 */
	public boolean isMonoPrint() {
		return monoPrint;
	}

	/**
	 * 白黒印刷設定(true : 白黒印刷 / false : 通常印刷(ディフォルト))を設定します。<br>
	 * @param monoPrint 白黒印刷設定(true : 白黒印刷 / false : 通常印刷(ディフォルト))
	 */
	public void setMonoPrint(boolean monoPrint) {
		this.monoPrint = monoPrint;
	}

	/**
	 * コメント印刷設定(true : コメントを印刷 / false : コメント未印刷(ディフォルト))を取得します。<br>
	 * @return コメント印刷設定(true : コメントを印刷 / false : コメント未印刷(ディフォルト))
	 */
	public boolean isPrintComment() {
		return printComment;
	}

	/**
	 * コメント印刷設定(true : コメントを印刷 / false : コメント未印刷(ディフォルト))を設定します。<br>
	 * @param printComment コメント印刷設定(true : コメントを印刷 / false : コメント未印刷(ディフォルト))
	 */
	public void setPrintComment(boolean printComment) {
		this.printComment = printComment;
	}

	/**
	 * 先頭印刷ページ(ディフォルト値 : 1(先頭ページ))を取得します。<br>
	 * @return 先頭印刷ページ(ディフォルト値 : 1(先頭ページ))
	 */
	public short getStartPage() {
		return startPage;
	}

	/**
	 * 先頭印刷ページ(ディフォルト値 : 1(先頭ページ))を設定します。<br>
	 * @param startPage 先頭印刷ページ(ディフォルト値 : 1(先頭ページ))
	 */
	public void setStartPage(short startPage) {
		this.startPage = startPage;
	}

	/**
	 * 用紙サイズ(ディフォルト値 : org.apache.poi.ss.usermodel.PrintSetup.A4_PAPERSIZE)を取得します。<br>
	 * @return 用紙サイズ(ディフォルト値 : org.apache.poi.ss.usermodel.PrintSetup.A4_PAPERSIZE)
	 */
	public short getPaperSize() {
		return paperSize;
	}

	/**
	 * 用紙サイズ(ディフォルト値 : org.apache.poi.ss.usermodel.PrintSetup.A4_PAPERSIZE)を設定します。<br>
	 * @param paperSize 用紙サイズ(ディフォルト値 : org.apache.poi.ss.usermodel.PrintSetup.A4_PAPERSIZE)
	 */
	public void setPaperSize(short paperSize) {
		this.paperSize = paperSize;
	}
}
