package org.phosphoresce.lib.poi.adapter;

import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.ss.usermodel.Row;

/**
 * 行抽象クラスクラス<br>
 * <br>
 * POIオブジェクトインタフェースの拡張及び、関連インスタンスへのアクセッサを提供します。<br>
 * また、Excel上でのオペレーションに類似したインタフェースを併せて提供します。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2005/11/21	Kitagawa		新規作成
 * 2006/02/09	Kitagawa		setValueにてnullオブジェクトを判定して空文字列を入れるように修正
 * 2006/03/24	Kitagawa		setValueの文字列設定時に数値に変換できる文字列は数値としてセットするように修正(Excel仕様)
 * 2006/04/19	Kitagawa		getForceStringValueのインタフェースとして計算後の値を取得する為のパラメタを持つメソッドを追加
 * 2012/07/09	Kitagawa		バージョン2にアップデート
 * 2012/11/20	Kitagawa		セル高さを自動調整する為に各種サイズアクセッサを追加
 *-->
 */
public class PoiRow extends PoiBaseRow {

	/**
	 * コンストラクタ<br>
	 * @param workbook ワークブックオブジェクト
	 * @param sheet シートオブジェクト
	 * @param poiRow 行オブジェクト
	 */
	PoiRow(PoiWorkbook workbook, PoiSheet sheet, Row poiRow) {
		super(workbook, sheet, poiRow);
	}

	/**
	 * 現在の行の内容が表示されるように行高さを調整します。<br>
	 * 最大の高さを要するセル幅を基底として調整が行われます。<br>
	 * @param forceResize 現在のセル高さが十分な場合でも調整を行う場合にtrueを指定
	 */
	public void autoSizeRow(boolean forceResize) {
		int maxHeight = 0;
		for (int i = getPoiRow().getFirstCellNum(); i <= getPoiRow().getLastCellNum() - 1; i++) {
			if (!getCell(i).getPoiCell().getCellStyle().getWrapText()) {
				continue;
			}
			int fixedHeight = getCell(i).getFixedHeight();
			if (maxHeight < fixedHeight) {
				maxHeight = fixedHeight;
			}
		}
		if (!forceResize && getPoiRow().getHeight() > maxHeight) {
			return;
		}
		getPoiRow().setHeight((short) maxHeight);
	}

	/**
	 * 現在の行の内容が表示されるように行高さを調整します。<br>
	 * 最大の高さを要するセル幅を基底として調整が行われます。<br>
	 */
	public void autoSizeRow() {
		autoSizeRow(true);
	}

	/**
	 * 行のコピーを行います。<br>
	 * @param sheetIndex シートインデックス(0～)
	 * @param rowIndex コピー先行位置(0～)
	 */
	public void copy(int sheetIndex, int rowIndex) {
		// コピー先オブジェクト取得
		PoiRow dest = workbook.getSheet(sheetIndex).getRow(rowIndex);

		// 基本設定コピー
		dest.getPoiRow().setHeight(poiRow.getHeight());
		dest.getPoiRow().setZeroHeight(poiRow.getZeroHeight());
		if (poiRow.getRowStyle() != null) {
			dest.getPoiRow().setRowStyle(poiRow.getRowStyle());
		}

		// コピー処理
		int maxCellNum = poiRow.getLastCellNum() < dest.poiRow.getLastCellNum() ? dest.poiRow.getLastCellNum() : poiRow.getLastCellNum();
		if (getPoiRow() instanceof HSSFRow) {
			if (maxCellNum > 256) {
				maxCellNum = 256;
			}
		}
		for (int i = 0; i <= maxCellNum - 1; i++) {
			getCell(i).copy(sheetIndex, rowIndex, i);
		}
	}

	/**
	 * 行のコピーを行います。<br>
	 * @param sheetName コピー先シート名
	 * @param rowIndex コピー先行位置(0～)
	 */
	public void copy(String sheetName, int rowIndex) {
		copy(workbook.getSheet(sheetName).getSheetIndex(), rowIndex);
	}

	/**
	 * 行のコピーを行います。<br>
	 * @param rowIndex コピー先行位置(0～)
	 */
	public void copy(int rowIndex) {
		copy(sheet.getSheetIndex(), rowIndex);
	}
}
