/**
 * フィールド妥当性チェッククラス<br>
 * <br>
 * 各種フィールド入力状態のチェック処理機能を実現する為のインタフェースを提供します。<br>
 *
 * @author Kitagawa<br>
 *
 *<!--
 * 更新日		更新者				更新内容
 * 2012/07/02	Kitagawa			新規作成
 *-->
 */

/**
 * コンストラクタ<br>
 * @param {Object} options 動作オプション<br>
 * <ul>
 * </ul>
 */
var FieldValidator = function(options) {

	/** クラス名(必須入力) */
	this.clnRequire;

	/** クラス名(半角文字) */
	this.clnAscii;

	/** クラス名(半角数字) */
	this.clnAsciiNumeric;

	/** クラス名(半角英字) */
	this.clnAsciiAlphabet;

	/** クラス名(半角英数字) */
	this.clnAsciiAlphaNumeric;

	/** クラス名(半角英数字記号) */
	this.clnAsciiAlphaNumericSign;

	/** クラス名(半角カナ) */
	this.clnAsciiKana;

	/** クラス名(半角カナ) */
	this.clnAsciiKanaSign;

	/** クラス名(半角カナ数字記号) */
	this.clnAsciiKanaNumericSign;

	/** クラス名(全角文字) */
	this.clnEm;

	/** クラス名(全角数字) */
	this.clnEmNumeric;

	/** クラス名(全角英字) */
	this.clnEmAlphabet;

	/** クラス名(全角英数字) */
	this.clnEmAlphaNumeric;

	/** クラス名(全角英数字記号) */
	this.clnEmAlphaNumericSign;

	/** クラス名(全角カナ) */
	this.clnEmKana;

	/** クラス名(全角カナ) */
	this.clnEmKanaSign;

	/** クラス名(全角カナ数字記号) */
	this.clnEmKanaNumericSign;

	/**
	 * コンストラクタ処理
	 */
	var defaults = {
		clnRequire : "validateRequire",
		clnAscii : "validateAscii",
		clnAsciiNumeric : "validateAsciiNumeric",
		clnAsciiAlphabet : "validateAsciiAlphabet",
		clnAsciiAlphaNumeric : "validateAsciiAlphaNumeric",
		clnAsciiAlphaNumericSign : "validateAsciiAlphaNumericSign",
		clnAsciiKana : "validateAsciiKana",
		clnAsciiKanaSign : "validateAsciiKanaSign",
		clnAsciiKanaNumericSign : "validateAsciiKanaNumericSign",
		clnEm : "validateEm",
		clnEmNumeric : "validateEmNumeric",
		clnEmAlphabet : "validateEmAlphabet",
		clnEmAlphaNumeric : "validateEmAlphaNumeric",
		clnEmAlphaNumericSign : "validateEmAlphaNumericSign",
		clnEmKana : "validateEmKana",
		clnEmKanaSign : "validateEmKanaSign",
		clnEmKanaNumericSign : "validateEmKanaNumericSign"
	};
	var mergedOptions = $.extend({}, defaults, options);
	{
		this.clnRequire = mergedOptions.clnRequire;
		this.clnAscii = mergedOptions.clnAscii;
		this.clnAsciiNumeric = mergedOptions.clnAsciiNumeric;
		this.clnAsciiAlphabet = mergedOptions.clnAsciiAlphabet;
		this.clnAsciiAlphaNumeric = mergedOptions.clnAsciiAlphaNumeric;
		this.clnAsciiAlphaNumericSign = mergedOptions.clnAsciiAlphaNumericSign;
		this.clnAsciiKana = mergedOptions.clnAsciiKana;
		this.clnAsciiKanaSign = mergedOptions.clnAsciiKanaSign;
		this.clnAsciiKanaNumericSign = mergedOptions.clnAsciiKanaNumericSign;
		this.clnEm = mergedOptions.clnEm;
		this.clnEmNumeric = mergedOptions.clnEmNumeric;
		this.clnEmAlphabet = mergedOptions.clnEmAlphabet;
		this.clnEmAlphaNumeric = mergedOptions.clnEmAlphaNumeric;
		this.clnEmAlphaNumericSign = mergedOptions.clnEmAlphaNumericSign;
		this.clnEmKana = mergedOptions.clnEmKana;
		this.clnEmKanaSign = mergedOptions.clnEmKanaSign;
		this.clnEmKanaNumericSign = mergedOptions.clnEmKanaNumericSign;
	}
};

/** 正規表現範囲(全角文字) */
FieldValidator.REGEXP_RANGE_EM = "^ -~｡-ﾟ";

/** 正規表現範囲(全角数字) */
FieldValidator.REGEXP_RANGE_EM_NUMERIC = "０-９";

/** 正規表現範囲(全角英字(大文字)) */
FieldValidator.REGEXP_RANGE_EM_ALPHABET_UPPER = "Ａ-Ｚ";

/** 正規表現範囲(全角英字(小文字)) */
FieldValidator.REGEXP_RANGE_EM_ALPHABET_LOWER = "ａ-ｚ";

/** 正規表現範囲(全角ひらがな) */
FieldValidator.REGEXP_RANGE_EM_HIRAGANA = "ぁ-ん";

/** 正規表現範囲(全角カタカナ) */
FieldValidator.REGEXP_RANGE_EM_KATAKANA = "ァ-ヶ";

/** 正規表現範囲(全角記号) */
FieldValidator.REGEXP_RANGE_EM_SIGN = "　-◯";

/** 正規表現範囲(半角文字) */
FieldValidator.REGEXP_RANGE_ASCII = " -~｡-ﾟ";

/** 正規表現範囲(半角数字) */
FieldValidator.REGEXP_RANGE_ASCII_NUMERIC = "0-9";

/** 正規表現範囲(半角英字(大文字)) */
FieldValidator.REGEXP_RANGE_ASCII_ALPHABET_UPPER = "A-Z";

/** 正規表現範囲(半角英字(小文字)) */
FieldValidator.REGEXP_RANGE_ASCII_ALPHABET_LOWER = "a-z";

/** 正規表現範囲(半角カナ) */
FieldValidator.REGEXP_RANGE_ASCII_KANA = "ｦ-ﾟ";

/** 正規表現範囲(半角記号) */
FieldValidator.REGEXP_RANGE_ASCII_SIGN = " -/:-@\[-`\{-~｡-､ﾞ-ﾟ";

/** 全角-半角変換マップ(数字) */
FieldValidator.EM2A_MAP_NUMERIC = {
	"０" : "0",
	"１" : "1",
	"２" : "2",
	"３" : "3",
	"４" : "4",
	"５" : "5",
	"６" : "6",
	"７" : "7",
	"８" : "8",
	"９" : "9",
	"" : ""
};

/** 全角-半角変換マップ(英字) */
FieldValidator.EM2A_MAP_ALPHABET = {
	"Ａ" : "A",
	"Ｂ" : "B",
	"Ｃ" : "C",
	"Ｄ" : "D",
	"Ｅ" : "E",
	"Ｆ" : "F",
	"Ｇ" : "G",
	"Ｈ" : "H",
	"Ｉ" : "I",
	"Ｊ" : "J",
	"Ｋ" : "K",
	"Ｌ" : "L",
	"Ｍ" : "M",
	"Ｎ" : "N",
	"Ｏ" : "O",
	"Ｐ" : "P",
	"Ｑ" : "Q",
	"Ｒ" : "R",
	"Ｓ" : "S",
	"Ｔ" : "T",
	"Ｕ" : "U",
	"Ｖ" : "V",
	"Ｗ" : "W",
	"Ｘ" : "X",
	"Ｙ" : "Y",
	"Ｚ" : "Z",
	"ａ" : "a",
	"ｂ" : "b",
	"ｃ" : "c",
	"ｄ" : "d",
	"ｅ" : "e",
	"ｆ" : "f",
	"ｇ" : "g",
	"ｈ" : "h",
	"ｉ" : "i",
	"ｊ" : "j",
	"ｋ" : "k",
	"ｌ" : "l",
	"ｍ" : "m",
	"ｎ" : "n",
	"ｏ" : "o",
	"ｐ" : "p",
	"ｑ" : "q",
	"ｒ" : "r",
	"ｓ" : "s",
	"ｔ" : "t",
	"ｕ" : "u",
	"ｖ" : "v",
	"ｗ" : "w",
	"ｘ" : "x",
	"ｙ" : "y",
	"ｚ" : "z",
	"" : ""
};

/** 全角-半角変換マップ(かな) */
FieldValidator.EM2A_MAP_KANA = {
	"あ" : "ｱ",
	"い" : "ｲ",
	"う" : "ｳ",
	"え" : "ｴ",
	"お" : "ｵ",
	"か" : "ｶ",
	"き" : "ｷ",
	"く" : "ｸ",
	"け" : "ｹ",
	"こ" : "ｺ",
	"さ" : "ｻ",
	"し" : "ｼ",
	"す" : "ｽ",
	"せ" : "ｾ",
	"そ" : "ｿ",
	"た" : "ﾀ",
	"ち" : "ﾁ",
	"つ" : "ﾂ",
	"て" : "ﾃ",
	"と" : "ﾄ",
	"な" : "ﾅ",
	"に" : "ﾆ",
	"ぬ" : "ﾇ",
	"ね" : "ﾈ",
	"の" : "ﾉ",
	"は" : "ﾊ",
	"ひ" : "ﾋ",
	"ふ" : "ﾌ",
	"へ" : "ﾍ",
	"ほ" : "ﾎ",
	"ま" : "ﾏ",
	"み" : "ﾐ",
	"む" : "ﾑ",
	"め" : "ﾒ",
	"も" : "ﾓ",
	"や" : "ﾔ",
	"ゆ" : "ﾕ",
	"よ" : "ﾖ",
	"ら" : "ﾗ",
	"り" : "ﾘ",
	"る" : "ﾙ",
	"れ" : "ﾚ",
	"ろ" : "ﾛ",
	"わ" : "ﾜ",
	"を" : "ｦ",
	"ん" : "ﾝ",
	"ゃ" : "ｬ",
	"ゅ" : "ｭ",
	"ょ" : "ｮ",
	"っ" : "ｯ",
	"ぁ" : "ｧ",
	"ぃ" : "ｨ",
	"ぅ" : "ｩ",
	"ぇ" : "ｪ",
	"ぉ" : "ｫ",
	"が" : "ｶﾞ",
	"ぎ" : "ｷﾞ",
	"ぐ" : "ｸﾞ",
	"げ" : "ｹﾞ",
	"ご" : "ｺﾞ",
	"ざ" : "ｻﾞ",
	"じ" : "ｼﾞ",
	"ず" : "ｽﾞ",
	"ぜ" : "ｾﾞ",
	"ぞ" : "ｿﾞ",
	"だ" : "ﾀﾞ",
	"ぢ" : "ﾁﾞ",
	"づ" : "ﾂﾞ",
	"で" : "ﾃﾞ",
	"ど" : "ﾄﾞ",
	"ば" : "ﾊﾞ",
	"び" : "ﾋﾞ",
	"ぶ" : "ﾌﾞ",
	"べ" : "ﾍﾞ",
	"ぼ" : "ﾎﾞ",
	"ぱ" : "ﾊﾟ",
	"ぴ" : "ﾋﾟ",
	"ぷ" : "ﾌﾟ",
	"ぺ" : "ﾍﾟ",
	"ぽ" : "ﾎﾟ",
	"ア" : "ｱ",
	"イ" : "ｲ",
	"ウ" : "ｳ",
	"エ" : "ｴ",
	"オ" : "ｵ",
	"カ" : "ｶ",
	"キ" : "ｷ",
	"ク" : "ｸ",
	"ケ" : "ｹ",
	"コ" : "ｺ",
	"サ" : "ｻ",
	"シ" : "ｼ",
	"ス" : "ｽ",
	"セ" : "ｾ",
	"ソ" : "ｿ",
	"タ" : "ﾀ",
	"チ" : "ﾁ",
	"ツ" : "ﾂ",
	"テ" : "ﾃ",
	"ト" : "ﾄ",
	"ナ" : "ﾅ",
	"ニ" : "ﾆ",
	"ヌ" : "ﾇ",
	"ネ" : "ﾈ",
	"ノ" : "ﾉ",
	"ハ" : "ﾊ",
	"ヒ" : "ﾋ",
	"フ" : "ﾌ",
	"ヘ" : "ﾍ",
	"ホ" : "ﾎ",
	"マ" : "ﾏ",
	"ミ" : "ﾐ",
	"ム" : "ﾑ",
	"メ" : "ﾒ",
	"モ" : "ﾓ",
	"ヤ" : "ﾔ",
	"ユ" : "ﾕ",
	"ヨ" : "ﾖ",
	"ラ" : "ﾗ",
	"リ" : "ﾘ",
	"ル" : "ﾙ",
	"レ" : "ﾚ",
	"ロ" : "ﾛ",
	"ワ" : "ﾜ",
	"ヲ" : "ｦ",
	"ン" : "ﾝ",
	"ャ" : "ｬ",
	"ュ" : "ｭ",
	"ョ" : "ｮ",
	"ッ" : "ｯ",
	"ァ" : "ｧ",
	"ィ" : "ｨ",
	"ゥ" : "ｩ",
	"ェ" : "ｪ",
	"ォ" : "ｫ",
	"ガ" : "ｶﾞ",
	"ギ" : "ｷﾞ",
	"グ" : "ｸﾞ",
	"ゲ" : "ｹﾞ",
	"ゴ" : "ｺﾞ",
	"ザ" : "ｻﾞ",
	"ジ" : "ｼﾞ",
	"ズ" : "ｽﾞ",
	"ゼ" : "ｾﾞ",
	"ゾ" : "ｿﾞ",
	"ダ" : "ﾀﾞ",
	"ヂ" : "ﾁﾞ",
	"ヅ" : "ﾂﾞ",
	"デ" : "ﾃﾞ",
	"ド" : "ﾄﾞ",
	"バ" : "ﾊﾞ",
	"ビ" : "ﾋﾞ",
	"ブ" : "ﾌﾞ",
	"ベ" : "ﾍﾞ",
	"ボ" : "ﾎﾞ",
	"パ" : "ﾊﾟ",
	"ピ" : "ﾋﾟ",
	"プ" : "ﾌﾟ",
	"ペ" : "ﾍﾟ",
	"ポ" : "ﾎﾟ",
	"ヴ" : "ｳﾞ",
	"ヵ" : "ｶ",
	"ヶ" : "ｹ",
	"" : ""
};

/** 全角-半角変換マップ(記号) */
FieldValidator.EM2A_MAP_SIGN = {
	"　" : " ",
	"、" : "､",
	"。" : "｡",
	"，" : ",",
	"．" : ".",
	"・" : "･",
	"：" : ":",
	"；" : ";",
	"？" : "?",
	"！" : "!",
	"゛" : "ﾞ",
	"゜" : "ﾟ",
	"´" : "'",
	"｀" : "'",
	"＾" : "^",
	"￣" : "~",
	"＿" : "_",
	"ー" : "-",
	"―" : "-",
	"‐" : "-",
	"／" : "/",
	"＼" : "\\",
	"～" : "~",
	"∥" : "|",
	"｜" : "|",
	"‘" : "'",
	"’" : "'",
	"“" : "\"",
	"”" : "\"",
	"（" : "(",
	"）" : ")",
	"〔" : "[",
	"〕" : "]",
	"［" : "[",
	"］" : "]",
	"｛" : "{",
	"｝" : "}",
	"〈" : "<",
	"〉" : ">",
	"《" : "<",
	"》" : ">",
	"「" : "｢",
	"」" : "｣",
	"『" : "｢",
	"』" : "｣",
	"【" : "(",
	"】" : ")",
	"＋" : "+",
	"－" : "-",
	"×" : "*",
	"÷" : "/",
	"＝" : "=",
	"＜" : "<",
	"＞" : ">",
	"≦" : "<",
	"≧" : ">",
	"°" : "ﾟ",
	"′" : "'",
	"″" : "\"",
	"￥" : "\\",
	"＄" : "$",
	"％" : "%",
	"＃" : "#",
	"＆" : "&",
	"＊" : "*",
	"＠" : "@",
	"≪" : "<",
	"≫" : ">",
	"" : ""
};

/** 半角-全角変換マップ(数字) */
FieldValidator.A2EM_MAP_NUMERIC = {
	"0" : "０",
	"1" : "１",
	"2" : "２",
	"3" : "３",
	"4" : "４",
	"5" : "５",
	"6" : "６",
	"7" : "７",
	"8" : "８",
	"9" : "９",
	"" : ""
};

/** 半角-全角変換マップ(英字) */
FieldValidator.A2EM_MAP_ALPHABET = {
	"A" : "Ａ",
	"B" : "Ｂ",
	"C" : "Ｃ",
	"D" : "Ｄ",
	"E" : "Ｅ",
	"F" : "Ｆ",
	"G" : "Ｇ",
	"H" : "Ｈ",
	"I" : "Ｉ",
	"J" : "Ｊ",
	"K" : "Ｋ",
	"L" : "Ｌ",
	"M" : "Ｍ",
	"N" : "Ｎ",
	"O" : "Ｏ",
	"P" : "Ｐ",
	"Q" : "Ｑ",
	"R" : "Ｒ",
	"S" : "Ｓ",
	"T" : "Ｔ",
	"U" : "Ｕ",
	"V" : "Ｖ",
	"W" : "Ｗ",
	"X" : "Ｘ",
	"Y" : "Ｙ",
	"Z" : "Ｚ",
	"a" : "ａ",
	"b" : "ｂ",
	"c" : "ｃ",
	"d" : "ｄ",
	"e" : "ｅ",
	"f" : "ｆ",
	"g" : "ｇ",
	"h" : "ｈ",
	"i" : "ｉ",
	"j" : "ｊ",
	"k" : "ｋ",
	"l" : "ｌ",
	"m" : "ｍ",
	"n" : "ｎ",
	"o" : "ｏ",
	"p" : "ｐ",
	"q" : "ｑ",
	"r" : "ｒ",
	"s" : "ｓ",
	"t" : "ｔ",
	"u" : "ｕ",
	"v" : "ｖ",
	"w" : "ｗ",
	"x" : "ｘ",
	"y" : "ｙ",
	"z" : "ｚ",
	"" : ""
};

/** 半角-全角変換マップ(かな) */
FieldValidator.A2EM_MAP_KANA = {
	"ｱ" : "ア",
	"ｲ" : "イ",
	"ｳ" : "ウ",
	"ｴ" : "エ",
	"ｵ" : "オ",
	"ｶ" : "カ",
	"ｷ" : "キ",
	"ｸ" : "ク",
	"ｹ" : "ケ",
	"ｺ" : "コ",
	"ｻ" : "サ",
	"ｼ" : "シ",
	"ｽ" : "ス",
	"ｾ" : "セ",
	"ｿ" : "ソ",
	"ﾀ" : "タ",
	"ﾁ" : "チ",
	"ﾂ" : "ツ",
	"ﾃ" : "テ",
	"ﾄ" : "ト",
	"ﾅ" : "ナ",
	"ﾆ" : "ニ",
	"ﾇ" : "ヌ",
	"ﾈ" : "ネ",
	"ﾉ" : "ノ",
	"ﾊ" : "ハ",
	"ﾋ" : "ヒ",
	"ﾌ" : "フ",
	"ﾍ" : "ヘ",
	"ﾎ" : "ホ",
	"ﾏ" : "マ",
	"ﾐ" : "ミ",
	"ﾑ" : "ム",
	"ﾒ" : "メ",
	"ﾓ" : "モ",
	"ﾔ" : "ヤ",
	"ﾕ" : "ユ",
	"ﾖ" : "ヨ",
	"ﾗ" : "ラ",
	"ﾘ" : "リ",
	"ﾙ" : "ル",
	"ﾚ" : "レ",
	"ﾛ" : "ロ",
	"ﾜ" : "ワ",
	"ｦ" : "ヲ",
	"ﾝ" : "ン",
	"ｬ" : "ャ",
	"ｭ" : "ュ",
	"ｮ" : "ョ",
	"ｯ" : "ッ",
	"ｧ" : "ァ",
	"ｨ" : "ィ",
	"ｩ" : "ゥ",
	"ｪ" : "ェ",
	"ｫ" : "ォ",
	//	"ｶﾞ" : "ガ",
	//	"ｷﾞ" : "ギ",
	//	"ｸﾞ" : "グ",
	//	"ｹﾞ" : "ゲ",
	//	"ｺﾞ" : "ゴ",
	//	"ｻﾞ" : "ザ",
	//	"ｼﾞ" : "ジ",
	//	"ｽﾞ" : "ズ",
	//	"ｾﾞ" : "ゼ",
	//	"ｿﾞ" : "ゾ",
	//	"ﾀﾞ" : "ダ",
	//	"ﾁﾞ" : "ヂ",
	//	"ﾂﾞ" : "ヅ",
	//	"ﾃﾞ" : "デ",
	//	"ﾄﾞ" : "ド",
	//	"ﾊﾞ" : "バ",
	//	"ﾋﾞ" : "ビ",
	//	"ﾌﾞ" : "ブ",
	//	"ﾍﾞ" : "ベ",
	//	"ﾎﾞ" : "ボ",
	//	"ﾊﾟ" : "パ",
	//	"ﾋﾟ" : "ピ",
	//	"ﾌﾟ" : "プ",
	//	"ﾍﾟ" : "ペ",
	//	"ﾎﾟ" : "ポ",
	//	"ｳﾞ" : "ヴ",
	"ﾞ" : "゛",
	"ﾟ" : "゜",
	"" : ""
};

/** 半角-全角変換マップ(記号) */
FieldValidator.A2EM_MAP_SIGN = {
	" " : "　",
	"､" : "、",
	"｡" : "。",
	"," : "，",
	"." : "．",
	"･" : "・",
	":" : "：",
	";" : "；",
	"?" : "？",
	"!" : "！",
	"ﾞ" : "゛",
	"ﾟ" : "゜",
	//	"'" : "´",
	//	"'" : "｀",
	"^" : "＾",
	"~" : "￣",
	"_" : "＿",
	//	"-" : "ー",
	"-" : "―",
	//"-" : "‐",
	"/" : "／",
	//	"\\" : "＼",
	//	"~" : "～",
	//	"|" : "∥",
	"|" : "｜",
	//	"'" : "‘",
	"'" : "’",
	//	"\"" : "“",
	"\"" : "”",
	"(" : "（",
	")" : "）",
	//	"[" : "〔",
	//	"]" : "〕",
	"[" : "［",
	"]" : "］",
	"{" : "｛",
	"}" : "｝",
	//	"<" : "〈",
	//	">" : "〉",
	//	"<" : "《",
	//	">" : "》",
	"｢" : "「",
	"｣" : "」",
	//	"｢" : "『",
	//	"｣" : "』",
	//	"(" : "【",
	//	")" : "】",
	"+" : "＋",
	"-" : "－",
	//	"*" : "×",
	//	"/" : "÷",
	"=" : "＝",
	"<" : "＜",
	">" : "＞",
	//	"<" : "≦",
	//	">" : "≧",
	//	"ﾟ" : "°",
	//	"'" : "′",
	//	"\"" : "″",
	"\\" : "￥",
	"$" : "＄",
	"%" : "％",
	"#" : "＃",
	"&" : "＆",
	"*" : "＊",
	"@" : "＠",
	//	"<" : "≪",
	//	">" : "≫"
	"" : ""
};

/** 全角濁点カナ */
FieldValidator.SONAT_KANA_EM = {
	"か゛" : "が",
	"き゛" : "ぎ",
	"く゛" : "ぐ",
	"け゛" : "げ",
	"こ゛" : "ご",
	"さ゛" : "ざ",
	"し゛" : "じ",
	"す゛" : "ず",
	"せ゛" : "ぜ",
	"そ゛" : "ぞ",
	"た゛" : "だ",
	"ち゛" : "ぢ",
	"つ゛" : "づ",
	"て゛" : "で",
	"と゛" : "ど",
	"は゛" : "ば",
	"ひ゛" : "び",
	"ふ゛" : "ぶ",
	"へ゛" : "べ",
	"ほ゛" : "ぼ",
	"は゜" : "ぱ",
	"ひ゜" : "ぴ",
	"ふ゜" : "ぷ",
	"へ゜" : "ぺ",
	"ほ゜" : "ぽ",
	"カ゛" : "ガ",
	"キ゛" : "ギ",
	"ク゛" : "グ",
	"ケ゛" : "ゲ",
	"コ゛" : "ゴ",
	"サ゛" : "ザ",
	"シ゛" : "ジ",
	"ス゛" : "ズ",
	"セ゛" : "ゼ",
	"ソ゛" : "ゾ",
	"タ゛" : "ダ",
	"チ゛" : "ヂ",
	"ツ゛" : "ヅ",
	"テ゛" : "デ",
	"ト゛" : "ド",
	"ハ゛" : "バ",
	"ヒ゛" : "ビ",
	"フ゛" : "ブ",
	"ヘ゛" : "ベ",
	"ホ゛" : "ボ",
	"ハ゜" : "パ",
	"ヒ゜" : "ピ",
	"フ゜" : "プ",
	"ヘ゜" : "ペ",
	"ホ゜" : "ポ",
	"ウ゛" : "ヴ",
	"" : ""
};

/**
 * 各種チェック処理識別用スタイルクラスが指定された要素に対して
 * 一括でチェック処理イベントハンドラを追加します。<br>
 * @returns {Boolean} 正常終了した場合にtrueを返却
 */
FieldValidator.prototype.reflectHandler = function() {
	/*
	 * フォーカスアウト補正処理イベントハンドラ
	 */
	$("." + this.clnAscii).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceEm2ANumeric($(this).val()));
		$(this).val(FieldValidator.replaceEm2AAlphabet($(this).val()));
		$(this).val(FieldValidator.replaceEm2AKana($(this).val()));
		$(this).val(FieldValidator.replaceEm2ASign($(this).val()));
	});
	$("." + this.clnAsciiAlphabet).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceEm2AAlphabet($(this).val()));
	});
	$("." + this.clnAsciiAlphaNumeric).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceEm2ANumeric($(this).val()));
		$(this).val(FieldValidator.replaceEm2AAlphabet($(this).val()));
	});
	$("." + this.clnAsciiAlphaNumericSign).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceEm2ANumeric($(this).val()));
		$(this).val(FieldValidator.replaceEm2AAlphabet($(this).val()));
		$(this).val(FieldValidator.replaceEm2ASign($(this).val()));
	});
	$("." + this.clnAsciiKana).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceEm2AKana($(this).val()));
	});
	$("." + this.clnAsciiKanaNumericSign).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceEm2ANumeric($(this).val()));
		$(this).val(FieldValidator.replaceEm2AKana($(this).val()));
		$(this).val(FieldValidator.replaceEm2ASign($(this).val()));
	});
	$("." + this.clnAsciiKanaSign).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceEm2AKana($(this).val()));
		$(this).val(FieldValidator.replaceEm2ASign($(this).val()));
	});
	$("." + this.clnEm).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceA2EmNumeric($(this).val()));
		$(this).val(FieldValidator.replaceA2EmAlphabet($(this).val()));
		$(this).val(FieldValidator.replaceA2EmKana($(this).val()));
		$(this).val(FieldValidator.replaceA2EmSign($(this).val()));
	});
	$("." + this.clnEmAlphabet).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceA2EmAlphabet($(this).val()));
	});
	$("." + this.clnEmAlphaNumeric).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceA2EmNumeric($(this).val()));
		$(this).val(FieldValidator.replaceA2EmAlphabet($(this).val()));
	});
	$("." + this.clnEmAlphaNumericSign).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceA2EmNumeric($(this).val()));
		$(this).val(FieldValidator.replaceA2EmAlphabet($(this).val()));
		$(this).val(FieldValidator.replaceA2EmSign($(this).val()));
	});
	$("." + this.clnEmKana).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceA2EmKana($(this).val()));
	});
	$("." + this.clnEmKanaNumericSign).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceA2EmNumeric($(this).val()));
		$(this).val(FieldValidator.replaceA2EmKana($(this).val()));
		$(this).val(FieldValidator.replaceA2EmSign($(this).val()));
	});
	$("." + this.clnEmKanaSign).bind("blur.FieldValidator", function() {
		$(this).val(FieldValidator.replaceA2EmKana($(this).val()));
		$(this).val(FieldValidator.replaceA2EmSign($(this).val()));
	});

	return true;
};

/**
 * 指定された文字列を全角-半角(数字)変換して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldValidator.replaceEm2ANumeric = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldValidator.EM2A_MAP_NUMERIC[arg0] ? FieldValidator.EM2A_MAP_NUMERIC[arg0] : arg0;
		return string;
	});
};

/**
 * 指定された文字列を全角-半角(英字)変換して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldValidator.replaceEm2AAlphabet = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldValidator.EM2A_MAP_ALPHABET[arg0] ? FieldValidator.EM2A_MAP_ALPHABET[arg0] : arg0;
		return string;
	});
};

/**
 * 指定された文字列を全角-半角(かな)変換して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldValidator.replaceEm2AKana = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldValidator.EM2A_MAP_KANA[arg0] ? FieldValidator.EM2A_MAP_KANA[arg0] : arg0;
		return string;
	});
};

/**
 * 指定された文字列を全角-半角(記号)変換して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldValidator.replaceEm2ASign = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldValidator.EM2A_MAP_SIGN[arg0] ? FieldValidator.EM2A_MAP_SIGN[arg0] : arg0;
		return string;
	});
};

/**
 * 指定された文字列を半角-全角(数字)変換して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldValidator.replaceA2EmNumeric = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldValidator.A2EM_MAP_NUMERIC[arg0] ? FieldValidator.A2EM_MAP_NUMERIC[arg0] : arg0;
		return string;
	});
};

/**
 * 指定された文字列を半角-全角(英字)変換して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldValidator.replaceA2EmAlphabet = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldValidator.A2EM_MAP_ALPHABET[arg0] ? FieldValidator.A2EM_MAP_ALPHABET[arg0] : arg0;
		return string;
	});
};

/**
 * 指定された文字列を半角-全角(かな)変換して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldValidator.replaceA2EmKana = function(value) {
	var result = value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldValidator.A2EM_MAP_KANA[arg0] ? FieldValidator.A2EM_MAP_KANA[arg0] : arg0;
		return string;
	});
	for ( var kana in this.SONAT_KANA_EM) {
		result = result.replace(new RegExp(kana, "g"), this.SONAT_KANA_EM[kana]);
	}
	return result;
};

/**
 * 指定された文字列を半角-全角(記号)変換して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldValidator.replaceA2EmSign = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldValidator.A2EM_MAP_SIGN[arg0] ? FieldValidator.A2EM_MAP_SIGN[arg0] : arg0;
		return string;
	});
};

/**
 * 指定されたセレクタから値を取得します。<br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {String} セレクタが提供する値
 */
FieldValidator.getValue = function(selector) {
	var $element = $(selector);
	var value = $element.val ? $(selector).val() : "";
	return value;
};

/**
 * 必須入力をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkRequire = function(selector) {
	var value = this.getValue(selector);
	if (value.length <= 0) {
		return false;
	}
	return true;
};

/**
 * 必須入力をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateRequire = function(selector, name) {
	/*
	 * 入力値補正処理
	 */
	var value = this.getValue(selector);
	// 補正処理はなし

	/*
	 * チェック処理
	 */
	if (!this.checkRequire(selector)) {
		alert(Message.get("FCRE10001", [name]));
		$(selector).focus();
		return false;
	}
	return true;
};

/**
 * 指定された正規表現で許可される文字列かチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} regexp 正規表現文字列
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkRegExp = function(selector, regexp) {
	var value = this.getValue(selector);
	if (!value.match(new RegExp(regexp, "g"))) {
		return false;
	}
	return true;
};

/**
 * 指定された正規表現で許可される文字列チェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} regexp 正規表現文字列
 * @param {String} name 項目名称
 * @param {String} message エラーメッセージ
 * @param {Function} beforeProcess チェック処理前処理(引数はselector)
 * @param {Function} afterProcess チェック処理後処理(引数はselector)
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateRegExp = function(selector, regexp, name, message, beforeProcess, afterProcess) {
	// 入力値取得
	var value = this.getValue(selector);

	// 事前処理
	if (beforeProcess) {
		beforeProcess(selector);
	}

	// チェック処理
	if (!this.checkRegExp(selector, regexp)) {
		alert(message);
		if (afterProcess) {
			afterProcess(selector);
		}
		$(selector).focus();
		return false;
	}

	return true;
};

/**
 * 半角文字をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkAscii = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII + "]+$");
};

/**
 * 半角文字をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateAscii = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII + "]+$", name, //
	Message.get("FCRE10002", [name, "半角文字"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceEm2ANumeric(value);
		value = FieldValidator.replaceEm2AAlphabet(value);
		value = FieldValidator.replaceEm2AKana(value);
		value = FieldValidator.replaceEm2ASign(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 半角数字をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkAsciiNumeric = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII_NUMERIC + "]+$");
};

/**
 * 半角数字をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateAsciiNumeric = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII_NUMERIC + "]+$", name, //
	Message.get("FCRE10002", [name, "半角数字"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceEm2ANumeric(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 半角英字をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkAsciiAlphabet = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII_ALPHABET_UPPER + this.REGEXP_RANGE_ASCII_ALPHABET_LOWER + "]+$");
};

/**
 * 半角英字をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateAsciiAlphabet = function(selector, name) {
	return this.validateRegExp(selector, this.REGEXP_RANGE_ASCII_ALPHABET_UPPER + this.REGEXP_RANGE_ASCII_ALPHABET_LOWER, name, //
	Message.get("FCRE10002", [name, "半角英字"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceEm2AAlphabet(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 半角英数字をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkAsciiAlphaNumeric = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII_NUMERIC + this.REGEXP_RANGE_ASCII_ALPHABET_UPPER + this.REGEXP_RANGE_ASCII_ALPHABET_LOWER + "]+$");
};

/**
 * 半角英数字をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateAsciiAlphaNumeric = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII_NUMERIC + this.REGEXP_RANGE_ASCII_ALPHABET_UPPER + this.REGEXP_RANGE_ASCII_ALPHABET_LOWER + "]+$", name, //
	Message.get("FCRE10002", [name, "半角英数字"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceEm2AAlphabet(value);
		value = FieldValidator.replaceEm2ANumeric(value);
		value = value.toUpperCase();
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 半角英数字記号をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkAsciiAlphaNumericSign = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII_NUMERIC + this.REGEXP_RANGE_ASCII_ALPHABET_UPPER + this.REGEXP_RANGE_ASCII_ALPHABET_LOWER + this.REGEXP_RANGE_ASCII_SIGN + "]+$");
};

/**
 * 半角英数字記号をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateAsciiAlphaNumericSign = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII_NUMERIC + this.REGEXP_RANGE_ASCII_ALPHABET_UPPER + this.REGEXP_RANGE_ASCII_ALPHABET_LOWER + this.REGEXP_RANGE_ASCII_SIGN + "]+$", name, //
	Message.get("FCRE10002", [name, "半角英数字記号"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceEm2AAlphabet(value);
		value = FieldValidator.replaceEm2ANumeric(value);
		value = FieldValidator.replaceEm2ASign(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 半角カナをチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkAsciiKana = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII_KANA + "]+$");
};

/**
 * 半角カナをチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateAsciiKana = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII_KANA + "]+$", name, //
	Message.get("FCRE10002", [name, "半角カナ"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceEm2AKana(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 半角カナ記号をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkAsciiKanaSign = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII_KANA + this.REGEXP_RANGE_ASCII_SIGN + "]+$");
};

/**
 * 半角カナ記号をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateAsciiKanaSign = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII_KANA + this.REGEXP_RANGE_ASCII_SIGN + "]+$", name, //
	Message.get("FCRE10002", [name, "半角カナ記号"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceEm2AKana(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 半角カナ数字記号をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkAsciiKanaNumericSign = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII_NUMERIC + this.REGEXP_RANGE_ASCII_KANA + this.REGEXP_RANGE_ASCII_SIGN + "]+$");
};

/**
 * 半角カナ数字記号をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateAsciiKanaNumericSign = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_ASCII_NUMERIC + this.REGEXP_RANGE_ASCII_KANA + this.REGEXP_RANGE_ASCII_SIGN + "]+$", name, //
	Message.get("FCRE10002", [name, "半角カナ数字記号"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceEm2AKana(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 全角文字をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkEm = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_EM + "]+$");
};

/**
 * 全角文字をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateEm = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_EM + "]+$", name, //
	Message.get("FCRE10002", [name, "全角文字"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceA2EmNumeric(value);
		value = FieldValidator.replaceA2EmAlphabet(value);
		value = FieldValidator.replaceA2EmKana(value);
		value = FieldValidator.replaceA2EmSign(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 全角数字をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkEmNumeric = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_EM_NUMERIC + "]+$");
};

/**
 * 全角数字をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateEmNumeric = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_EM_NUMERIC + "]+$", name, //
	Message.get("FCRE10002", [name, "全角数字"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceA2EmNumeric(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 全角英字をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkEmAlphabet = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_EM_ALPHABET_UPPER + this.REGEXP_RANGE_EM_ALPHABET_LOWER + "]+$");
};

/**
 * 全角英字をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateEmAlphabet = function(selector, name) {
	return this.validateRegExp(selector, this.REGEXP_RANGE_EM_ALPHABET_UPPER + this.REGEXP_RANGE_EM_ALPHABET_LOWER, name, //
	Message.get("FCRE10002", [name, "全角英字"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceA2EmAlphabet(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 全角英数字をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkEmAlphaNumeric = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_EM_NUMERIC + this.REGEXP_RANGE_EM_ALPHABET_UPPER + this.REGEXP_RANGE_EM_ALPHABET_LOWER + "]+$");
};

/**
 * 全角英数字をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateEmAlphaNumeric = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_EM_NUMERIC + this.REGEXP_RANGE_EM_ALPHABET_UPPER + this.REGEXP_RANGE_EM_ALPHABET_LOWER + "]+$", name, //
	Message.get("FCRE10002", [name, "全角英数字"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceA2EmAlphabet(value);
		value = FieldValidator.replaceA2EmNumeric(value);
		value = value.toUpperCase();
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 全角英数字記号をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkEmAlphaNumericSign = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_EM_NUMERIC + this.REGEXP_RANGE_EM_ALPHABET_UPPER + this.REGEXP_RANGE_EM_ALPHABET_LOWER + this.REGEXP_RANGE_EM_SIGN + "]+$");
};

/**
 * 全角英数字記号をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateEmAlphaNumericSign = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_EM_NUMERIC + this.REGEXP_RANGE_EM_ALPHABET_UPPER + this.REGEXP_RANGE_EM_ALPHABET_LOWER + this.REGEXP_RANGE_EM_SIGN + "]+$", name, //
	Message.get("FCRE10002", [name, "全角英数字記号"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceA2EmAlphabet(value);
		value = FieldValidator.replaceA2EmNumeric(value);
		value = FieldValidator.replaceA2EmSign(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 全角カナをチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkEmKana = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_EM_KANA + "]+$");
};

/**
 * 全角カナをチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateEmKana = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_EM_KANA + "]+$", name, //
	Message.get("FCRE10002", [name, "全角カナ"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceA2EmKana(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 全角カナ記号をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkEmKanaSign = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_EM_KANA + this.REGEXP_RANGE_EM_SIGN + "]+$");
};

/**
 * 全角カナ記号をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateEmKanaSign = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_EM_KANA + this.REGEXP_RANGE_EM_SIGN + "]+$", name, //
	Message.get("FCRE10002", [name, "全角カナ記号"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceA2EmKana(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};

/**
 * 全角カナ数字記号をチェックします。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.checkEmKanaNumericSign = function(selector) {
	return this.checkRegExp(selector, "^[" + this.REGEXP_RANGE_EM_NUMERIC + this.REGEXP_RANGE_EM_KANA + this.REGEXP_RANGE_EM_SIGN + "]+$");
};

/**
 * 全角カナ数字記号をチェックハンドラを設定します。br>
 * @param {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト
 * @param {String} name 項目名称
 * @returns {Boolean} チェックの結果がエラーの場合にfalseを返却
 */
FieldValidator.validateEmKanaNumericSign = function(selector, name) {
	return this.validateRegExp(selector, "^[" + this.REGEXP_RANGE_EM_NUMERIC + this.REGEXP_RANGE_EM_KANA + this.REGEXP_RANGE_EM_SIGN + "]+$", name, //
	Message.get("FCRE10002", [name, "全角カナ数字記号"]), //
	function(selector) {
		var value = $(selector).val();
		value = FieldValidator.replaceA2EmKana(value);
		$(selector).val(value);
	}, //
	function(selector) {
	});
};
