/**
 * フィールド補完処理クラス<br>
 * <br>
 * 各種フィールド入力値の自動補完機能を実現する為のインタフェースを提供します。<br>
 *
 * @author Kitagawa<br>
 *
 *<!--
 * 更新日		更新者				更新内容
 * 2012/07/02	Kitagawa			新規作成
 *-->
 */

/**
 * コンストラクタ<br>
 * @param {Object} options 動作オプション<br>
 * <ul>
 * </ul>
 */
var FieldCorrector = function(options) {

	/** クラス名(半角文字) */
	this.clnAscii;

	/** クラス名(半角数字) */
	this.clnAsciiNumeric;

	/** クラス名(半角英字) */
	this.clnAsciiAlphabet;

	/** クラス名(半角英数字) */
	this.clnAsciiAlphaNumeric;

	/** クラス名(半角英数字記号) */
	this.clnAsciiAlphaNumericSign;

	/** クラス名(半角カナ) */
	this.clnAsciiKana;

	/** クラス名(半角カナ) */
	this.clnAsciiKanaSign;

	/** クラス名(半角カナ数字記号) */
	this.clnAsciiKanaNumericSign;

	/** クラス名(全角文字) */
	this.clnEm;

	/** クラス名(全角数字) */
	this.clnEmNumeric;

	/** クラス名(全角英字) */
	this.clnEmAlphabet;

	/** クラス名(全角英数字) */
	this.clnEmAlphaNumeric;

	/** クラス名(全角英数字記号) */
	this.clnEmAlphaNumericSign;

	/** クラス名(全角カナ) */
	this.clnEmKana;

	/** クラス名(全角カナ) */
	this.clnEmKanaSign;

	/** クラス名(全角カナ数字記号) */
	this.clnEmKanaNumericSign;

	/** クラス名(半角数字前ゼロ) */
	this.clnZerosupH;

	/** クラス名(半角数字後ゼロ) */
	this.clnZerosupF;

	/**
	 * コンストラクタ処理
	 */
	var defaults = {
		clnAscii : "correctAscii",
		clnAsciiNumeric : "correctAsciiNumeric",
		clnAsciiAlphabet : "correctAsciiAlphabet",
		clnAsciiAlphaNumeric : "correctAsciiAlphaNumeric",
		clnAsciiAlphaNumericSign : "correctAsciiAlphaNumericSign",
		clnAsciiKana : "correctAsciiKana",
		clnAsciiKanaSign : "correctAsciiKanaSign",
		clnAsciiKanaNumericSign : "correctAsciiKanaNumericSign",
		clnEm : "correctEm",
		clnEmNumeric : "correctEmNumeric",
		clnEmAlphabet : "correctEmAlphabet",
		clnEmAlphaNumeric : "correctEmAlphaNumeric",
		clnEmAlphaNumericSign : "correctEmAlphaNumericSign",
		clnEmKana : "correctEmKana",
		clnEmKanaSign : "correctEmKanaSign",
		clnEmKanaNumericSign : "correctEmKanaNumericSign",
		clnZerosupH : "correctZerosupH",
		clnZerosupF : "correctZerosupF"
	};
	var mergedOptions = $.extend({}, defaults, options);
	{
		this.clnAscii = mergedOptions.clnAscii;
		this.clnAsciiNumeric = mergedOptions.clnAsciiNumeric;
		this.clnAsciiAlphabet = mergedOptions.clnAsciiAlphabet;
		this.clnAsciiAlphaNumeric = mergedOptions.clnAsciiAlphaNumeric;
		this.clnAsciiAlphaNumericSign = mergedOptions.clnAsciiAlphaNumericSign;
		this.clnAsciiKana = mergedOptions.clnAsciiKana;
		this.clnAsciiKanaSign = mergedOptions.clnAsciiKanaSign;
		this.clnAsciiKanaNumericSign = mergedOptions.clnAsciiKanaNumericSign;
		this.clnEm = mergedOptions.clnEm;
		this.clnEmNumeric = mergedOptions.clnEmNumeric;
		this.clnEmAlphabet = mergedOptions.clnEmAlphabet;
		this.clnEmAlphaNumeric = mergedOptions.clnEmAlphaNumeric;
		this.clnEmAlphaNumericSign = mergedOptions.clnEmAlphaNumericSign;
		this.clnEmKana = mergedOptions.clnEmKana;
		this.clnEmKanaSign = mergedOptions.clnEmKanaSign;
		this.clnEmKanaNumericSign = mergedOptions.clnEmKanaNumericSign;
		this.clnZerosupH = mergedOptions.clnZerosupH;
		this.clnZerosupF = mergedOptions.clnZerosupF;
	}
};

/** 全角-半角変換マップ(数字) */
FieldCorrector.EM2A_MAP_NUMERIC = {
	"０" : "0",
	"１" : "1",
	"２" : "2",
	"３" : "3",
	"４" : "4",
	"５" : "5",
	"６" : "6",
	"７" : "7",
	"８" : "8",
	"９" : "9",
	"" : ""
};

/** 全角-半角変換マップ(英字) */
FieldCorrector.EM2A_MAP_ALPHABET = {
	"Ａ" : "A",
	"Ｂ" : "B",
	"Ｃ" : "C",
	"Ｄ" : "D",
	"Ｅ" : "E",
	"Ｆ" : "F",
	"Ｇ" : "G",
	"Ｈ" : "H",
	"Ｉ" : "I",
	"Ｊ" : "J",
	"Ｋ" : "K",
	"Ｌ" : "L",
	"Ｍ" : "M",
	"Ｎ" : "N",
	"Ｏ" : "O",
	"Ｐ" : "P",
	"Ｑ" : "Q",
	"Ｒ" : "R",
	"Ｓ" : "S",
	"Ｔ" : "T",
	"Ｕ" : "U",
	"Ｖ" : "V",
	"Ｗ" : "W",
	"Ｘ" : "X",
	"Ｙ" : "Y",
	"Ｚ" : "Z",
	"ａ" : "a",
	"ｂ" : "b",
	"ｃ" : "c",
	"ｄ" : "d",
	"ｅ" : "e",
	"ｆ" : "f",
	"ｇ" : "g",
	"ｈ" : "h",
	"ｉ" : "i",
	"ｊ" : "j",
	"ｋ" : "k",
	"ｌ" : "l",
	"ｍ" : "m",
	"ｎ" : "n",
	"ｏ" : "o",
	"ｐ" : "p",
	"ｑ" : "q",
	"ｒ" : "r",
	"ｓ" : "s",
	"ｔ" : "t",
	"ｕ" : "u",
	"ｖ" : "v",
	"ｗ" : "w",
	"ｘ" : "x",
	"ｙ" : "y",
	"ｚ" : "z",
	"" : ""
};

/** 全角-半角変換マップ(かな) */
FieldCorrector.EM2A_MAP_KANA = {
	"あ" : "ｱ",
	"い" : "ｲ",
	"う" : "ｳ",
	"え" : "ｴ",
	"お" : "ｵ",
	"か" : "ｶ",
	"き" : "ｷ",
	"く" : "ｸ",
	"け" : "ｹ",
	"こ" : "ｺ",
	"さ" : "ｻ",
	"し" : "ｼ",
	"す" : "ｽ",
	"せ" : "ｾ",
	"そ" : "ｿ",
	"た" : "ﾀ",
	"ち" : "ﾁ",
	"つ" : "ﾂ",
	"て" : "ﾃ",
	"と" : "ﾄ",
	"な" : "ﾅ",
	"に" : "ﾆ",
	"ぬ" : "ﾇ",
	"ね" : "ﾈ",
	"の" : "ﾉ",
	"は" : "ﾊ",
	"ひ" : "ﾋ",
	"ふ" : "ﾌ",
	"へ" : "ﾍ",
	"ほ" : "ﾎ",
	"ま" : "ﾏ",
	"み" : "ﾐ",
	"む" : "ﾑ",
	"め" : "ﾒ",
	"も" : "ﾓ",
	"や" : "ﾔ",
	"ゆ" : "ﾕ",
	"よ" : "ﾖ",
	"ら" : "ﾗ",
	"り" : "ﾘ",
	"る" : "ﾙ",
	"れ" : "ﾚ",
	"ろ" : "ﾛ",
	"わ" : "ﾜ",
	"を" : "ｦ",
	"ん" : "ﾝ",
	"ゃ" : "ｬ",
	"ゅ" : "ｭ",
	"ょ" : "ｮ",
	"っ" : "ｯ",
	"ぁ" : "ｧ",
	"ぃ" : "ｨ",
	"ぅ" : "ｩ",
	"ぇ" : "ｪ",
	"ぉ" : "ｫ",
	"が" : "ｶﾞ",
	"ぎ" : "ｷﾞ",
	"ぐ" : "ｸﾞ",
	"げ" : "ｹﾞ",
	"ご" : "ｺﾞ",
	"ざ" : "ｻﾞ",
	"じ" : "ｼﾞ",
	"ず" : "ｽﾞ",
	"ぜ" : "ｾﾞ",
	"ぞ" : "ｿﾞ",
	"だ" : "ﾀﾞ",
	"ぢ" : "ﾁﾞ",
	"づ" : "ﾂﾞ",
	"で" : "ﾃﾞ",
	"ど" : "ﾄﾞ",
	"ば" : "ﾊﾞ",
	"び" : "ﾋﾞ",
	"ぶ" : "ﾌﾞ",
	"べ" : "ﾍﾞ",
	"ぼ" : "ﾎﾞ",
	"ぱ" : "ﾊﾟ",
	"ぴ" : "ﾋﾟ",
	"ぷ" : "ﾌﾟ",
	"ぺ" : "ﾍﾟ",
	"ぽ" : "ﾎﾟ",
	"ア" : "ｱ",
	"イ" : "ｲ",
	"ウ" : "ｳ",
	"エ" : "ｴ",
	"オ" : "ｵ",
	"カ" : "ｶ",
	"キ" : "ｷ",
	"ク" : "ｸ",
	"ケ" : "ｹ",
	"コ" : "ｺ",
	"サ" : "ｻ",
	"シ" : "ｼ",
	"ス" : "ｽ",
	"セ" : "ｾ",
	"ソ" : "ｿ",
	"タ" : "ﾀ",
	"チ" : "ﾁ",
	"ツ" : "ﾂ",
	"テ" : "ﾃ",
	"ト" : "ﾄ",
	"ナ" : "ﾅ",
	"ニ" : "ﾆ",
	"ヌ" : "ﾇ",
	"ネ" : "ﾈ",
	"ノ" : "ﾉ",
	"ハ" : "ﾊ",
	"ヒ" : "ﾋ",
	"フ" : "ﾌ",
	"ヘ" : "ﾍ",
	"ホ" : "ﾎ",
	"マ" : "ﾏ",
	"ミ" : "ﾐ",
	"ム" : "ﾑ",
	"メ" : "ﾒ",
	"モ" : "ﾓ",
	"ヤ" : "ﾔ",
	"ユ" : "ﾕ",
	"ヨ" : "ﾖ",
	"ラ" : "ﾗ",
	"リ" : "ﾘ",
	"ル" : "ﾙ",
	"レ" : "ﾚ",
	"ロ" : "ﾛ",
	"ワ" : "ﾜ",
	"ヲ" : "ｦ",
	"ン" : "ﾝ",
	"ャ" : "ｬ",
	"ュ" : "ｭ",
	"ョ" : "ｮ",
	"ッ" : "ｯ",
	"ァ" : "ｧ",
	"ィ" : "ｨ",
	"ゥ" : "ｩ",
	"ェ" : "ｪ",
	"ォ" : "ｫ",
	"ガ" : "ｶﾞ",
	"ギ" : "ｷﾞ",
	"グ" : "ｸﾞ",
	"ゲ" : "ｹﾞ",
	"ゴ" : "ｺﾞ",
	"ザ" : "ｻﾞ",
	"ジ" : "ｼﾞ",
	"ズ" : "ｽﾞ",
	"ゼ" : "ｾﾞ",
	"ゾ" : "ｿﾞ",
	"ダ" : "ﾀﾞ",
	"ヂ" : "ﾁﾞ",
	"ヅ" : "ﾂﾞ",
	"デ" : "ﾃﾞ",
	"ド" : "ﾄﾞ",
	"バ" : "ﾊﾞ",
	"ビ" : "ﾋﾞ",
	"ブ" : "ﾌﾞ",
	"ベ" : "ﾍﾞ",
	"ボ" : "ﾎﾞ",
	"パ" : "ﾊﾟ",
	"ピ" : "ﾋﾟ",
	"プ" : "ﾌﾟ",
	"ペ" : "ﾍﾟ",
	"ポ" : "ﾎﾟ",
	"ヴ" : "ｳﾞ",
	"ヵ" : "ｶ",
	"ヶ" : "ｹ",
	"" : ""
};

/** 全角-半角変換マップ(記号) */
FieldCorrector.EM2A_MAP_SIGN = {
	"　" : " ",
	"、" : "､",
	"。" : "｡",
	"，" : ",",
	"．" : ".",
	"・" : "･",
	"：" : ":",
	"；" : ";",
	"？" : "?",
	"！" : "!",
	"゛" : "ﾞ",
	"゜" : "ﾟ",
	"´" : "'",
	"｀" : "'",
	"＾" : "^",
	"￣" : "~",
	"＿" : "_",
	"ー" : "-",
	"―" : "-",
	"‐" : "-",
	"／" : "/",
	"＼" : "\\",
	"～" : "~",
	"∥" : "|",
	"｜" : "|",
	"‘" : "'",
	"’" : "'",
	"“" : "\"",
	"”" : "\"",
	"（" : "(",
	"）" : ")",
	"〔" : "[",
	"〕" : "]",
	"［" : "[",
	"］" : "]",
	"｛" : "{",
	"｝" : "}",
	"〈" : "<",
	"〉" : ">",
	"《" : "<",
	"》" : ">",
	"「" : "｢",
	"」" : "｣",
	"『" : "｢",
	"』" : "｣",
	"【" : "(",
	"】" : ")",
	"＋" : "+",
	"－" : "-",
	"×" : "*",
	"÷" : "/",
	"＝" : "=",
	"＜" : "<",
	"＞" : ">",
	"≦" : "<",
	"≧" : ">",
	"°" : "ﾟ",
	"′" : "'",
	"″" : "\"",
	"￥" : "\\",
	"＄" : "$",
	"％" : "%",
	"＃" : "#",
	"＆" : "&",
	"＊" : "*",
	"＠" : "@",
	"≪" : "<",
	"≫" : ">",
	"" : ""
};

/** 半角-全角変換マップ(数字) */
FieldCorrector.A2EM_MAP_NUMERIC = {
	"0" : "０",
	"1" : "１",
	"2" : "２",
	"3" : "３",
	"4" : "４",
	"5" : "５",
	"6" : "６",
	"7" : "７",
	"8" : "８",
	"9" : "９",
	"" : ""
};

/** 半角-全角変換マップ(英字) */
FieldCorrector.A2EM_MAP_ALPHABET = {
	"A" : "Ａ",
	"B" : "Ｂ",
	"C" : "Ｃ",
	"D" : "Ｄ",
	"E" : "Ｅ",
	"F" : "Ｆ",
	"G" : "Ｇ",
	"H" : "Ｈ",
	"I" : "Ｉ",
	"J" : "Ｊ",
	"K" : "Ｋ",
	"L" : "Ｌ",
	"M" : "Ｍ",
	"N" : "Ｎ",
	"O" : "Ｏ",
	"P" : "Ｐ",
	"Q" : "Ｑ",
	"R" : "Ｒ",
	"S" : "Ｓ",
	"T" : "Ｔ",
	"U" : "Ｕ",
	"V" : "Ｖ",
	"W" : "Ｗ",
	"X" : "Ｘ",
	"Y" : "Ｙ",
	"Z" : "Ｚ",
	"a" : "ａ",
	"b" : "ｂ",
	"c" : "ｃ",
	"d" : "ｄ",
	"e" : "ｅ",
	"f" : "ｆ",
	"g" : "ｇ",
	"h" : "ｈ",
	"i" : "ｉ",
	"j" : "ｊ",
	"k" : "ｋ",
	"l" : "ｌ",
	"m" : "ｍ",
	"n" : "ｎ",
	"o" : "ｏ",
	"p" : "ｐ",
	"q" : "ｑ",
	"r" : "ｒ",
	"s" : "ｓ",
	"t" : "ｔ",
	"u" : "ｕ",
	"v" : "ｖ",
	"w" : "ｗ",
	"x" : "ｘ",
	"y" : "ｙ",
	"z" : "ｚ",
	"" : ""
};

/** 半角-全角変換マップ(かな) */
FieldCorrector.A2EM_MAP_KANA = {
	"ｱ" : "ア",
	"ｲ" : "イ",
	"ｳ" : "ウ",
	"ｴ" : "エ",
	"ｵ" : "オ",
	"ｶ" : "カ",
	"ｷ" : "キ",
	"ｸ" : "ク",
	"ｹ" : "ケ",
	"ｺ" : "コ",
	"ｻ" : "サ",
	"ｼ" : "シ",
	"ｽ" : "ス",
	"ｾ" : "セ",
	"ｿ" : "ソ",
	"ﾀ" : "タ",
	"ﾁ" : "チ",
	"ﾂ" : "ツ",
	"ﾃ" : "テ",
	"ﾄ" : "ト",
	"ﾅ" : "ナ",
	"ﾆ" : "ニ",
	"ﾇ" : "ヌ",
	"ﾈ" : "ネ",
	"ﾉ" : "ノ",
	"ﾊ" : "ハ",
	"ﾋ" : "ヒ",
	"ﾌ" : "フ",
	"ﾍ" : "ヘ",
	"ﾎ" : "ホ",
	"ﾏ" : "マ",
	"ﾐ" : "ミ",
	"ﾑ" : "ム",
	"ﾒ" : "メ",
	"ﾓ" : "モ",
	"ﾔ" : "ヤ",
	"ﾕ" : "ユ",
	"ﾖ" : "ヨ",
	"ﾗ" : "ラ",
	"ﾘ" : "リ",
	"ﾙ" : "ル",
	"ﾚ" : "レ",
	"ﾛ" : "ロ",
	"ﾜ" : "ワ",
	"ｦ" : "ヲ",
	"ﾝ" : "ン",
	"ｬ" : "ャ",
	"ｭ" : "ュ",
	"ｮ" : "ョ",
	"ｯ" : "ッ",
	"ｧ" : "ァ",
	"ｨ" : "ィ",
	"ｩ" : "ゥ",
	"ｪ" : "ェ",
	"ｫ" : "ォ",
	//	"ｶﾞ" : "ガ",
	//	"ｷﾞ" : "ギ",
	//	"ｸﾞ" : "グ",
	//	"ｹﾞ" : "ゲ",
	//	"ｺﾞ" : "ゴ",
	//	"ｻﾞ" : "ザ",
	//	"ｼﾞ" : "ジ",
	//	"ｽﾞ" : "ズ",
	//	"ｾﾞ" : "ゼ",
	//	"ｿﾞ" : "ゾ",
	//	"ﾀﾞ" : "ダ",
	//	"ﾁﾞ" : "ヂ",
	//	"ﾂﾞ" : "ヅ",
	//	"ﾃﾞ" : "デ",
	//	"ﾄﾞ" : "ド",
	//	"ﾊﾞ" : "バ",
	//	"ﾋﾞ" : "ビ",
	//	"ﾌﾞ" : "ブ",
	//	"ﾍﾞ" : "ベ",
	//	"ﾎﾞ" : "ボ",
	//	"ﾊﾟ" : "パ",
	//	"ﾋﾟ" : "ピ",
	//	"ﾌﾟ" : "プ",
	//	"ﾍﾟ" : "ペ",
	//	"ﾎﾟ" : "ポ",
	//	"ｳﾞ" : "ヴ",
	"ﾞ" : "゛",
	"ﾟ" : "゜",
	"" : ""
};

/** 半角-全角変換マップ(記号) */
FieldCorrector.A2EM_MAP_SIGN = {
	" " : "　",
	"､" : "、",
	"｡" : "。",
	"," : "，",
	"." : "．",
	"･" : "・",
	":" : "：",
	";" : "；",
	"?" : "？",
	"!" : "！",
	"ﾞ" : "゛",
	"ﾟ" : "゜",
	//	"'" : "´",
	//	"'" : "｀",
	"^" : "＾",
	"~" : "￣",
	"_" : "＿",
	//	"-" : "ー",
	"-" : "―",
	//"-" : "‐",
	"/" : "／",
	//	"\\" : "＼",
	//	"~" : "～",
	//	"|" : "∥",
	"|" : "｜",
	//	"'" : "‘",
	"'" : "’",
	//	"\"" : "“",
	"\"" : "”",
	"(" : "（",
	")" : "）",
	//	"[" : "〔",
	//	"]" : "〕",
	"[" : "［",
	"]" : "］",
	"{" : "｛",
	"}" : "｝",
	//	"<" : "〈",
	//	">" : "〉",
	//	"<" : "《",
	//	">" : "》",
	"｢" : "「",
	"｣" : "」",
	//	"｢" : "『",
	//	"｣" : "』",
	//	"(" : "【",
	//	")" : "】",
	"+" : "＋",
	"-" : "－",
	//	"*" : "×",
	//	"/" : "÷",
	"=" : "＝",
	"<" : "＜",
	">" : "＞",
	//	"<" : "≦",
	//	">" : "≧",
	//	"ﾟ" : "°",
	//	"'" : "′",
	//	"\"" : "″",
	"\\" : "￥",
	"$" : "＄",
	"%" : "％",
	"#" : "＃",
	"&" : "＆",
	"*" : "＊",
	"@" : "＠",
	//	"<" : "≪",
	//	">" : "≫"
	"" : ""
};

/** 全角濁点カナ */
FieldCorrector.SONAT_KANA_EM = {
	"か゛" : "が",
	"き゛" : "ぎ",
	"く゛" : "ぐ",
	"け゛" : "げ",
	"こ゛" : "ご",
	"さ゛" : "ざ",
	"し゛" : "じ",
	"す゛" : "ず",
	"せ゛" : "ぜ",
	"そ゛" : "ぞ",
	"た゛" : "だ",
	"ち゛" : "ぢ",
	"つ゛" : "づ",
	"て゛" : "で",
	"と゛" : "ど",
	"は゛" : "ば",
	"ひ゛" : "び",
	"ふ゛" : "ぶ",
	"へ゛" : "べ",
	"ほ゛" : "ぼ",
	"は゜" : "ぱ",
	"ひ゜" : "ぴ",
	"ふ゜" : "ぷ",
	"へ゜" : "ぺ",
	"ほ゜" : "ぽ",
	"カ゛" : "ガ",
	"キ゛" : "ギ",
	"ク゛" : "グ",
	"ケ゛" : "ゲ",
	"コ゛" : "ゴ",
	"サ゛" : "ザ",
	"シ゛" : "ジ",
	"ス゛" : "ズ",
	"セ゛" : "ゼ",
	"ソ゛" : "ゾ",
	"タ゛" : "ダ",
	"チ゛" : "ヂ",
	"ツ゛" : "ヅ",
	"テ゛" : "デ",
	"ト゛" : "ド",
	"ハ゛" : "バ",
	"ヒ゛" : "ビ",
	"フ゛" : "ブ",
	"ヘ゛" : "ベ",
	"ホ゛" : "ボ",
	"ハ゜" : "パ",
	"ヒ゜" : "ピ",
	"フ゜" : "プ",
	"ヘ゜" : "ペ",
	"ホ゜" : "ポ",
	"ウ゛" : "ヴ",
	"" : ""
};

/**
 * 各種チェック処理識別用スタイルクラスが指定された要素に対して
 * 一括でチェック処理イベントハンドラを追加します。<br>
 * @returns {Boolean} 正常終了した場合にtrueを返却
 */
FieldCorrector.prototype.reflectHandler = function() {
	/*
	 * フォーカスアウト補正処理イベントハンドラ
	 */
	$("." + this.clnAscii).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctAsciiNumeric($(this).val()));
		$(this).val(FieldCorrector.correctAsciiAlphabet($(this).val()));
		$(this).val(FieldCorrector.correctAsciiKana($(this).val()));
		$(this).val(FieldCorrector.correctAsciiSign($(this).val()));
	});
	$("." + this.clnAsciiAlphabet).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctAsciiAlphabet($(this).val()));
	});
	$("." + this.clnAsciiAlphaNumeric).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctAsciiNumeric($(this).val()));
		$(this).val(FieldCorrector.correctAsciiAlphabet($(this).val()));
	});
	$("." + this.clnAsciiAlphaNumericSign).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctAsciiNumeric($(this).val()));
		$(this).val(FieldCorrector.correctAsciiAlphabet($(this).val()));
		$(this).val(FieldCorrector.correctAsciiSign($(this).val()));
	});
	$("." + this.clnAsciiKana).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctAsciiKana($(this).val()));
	});
	$("." + this.clnAsciiKanaNumericSign).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctAsciiNumeric($(this).val()));
		$(this).val(FieldCorrector.correctAsciiKana($(this).val()));
		$(this).val(FieldCorrector.correctAsciiSign($(this).val()));
	});
	$("." + this.clnAsciiKanaSign).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctAsciiKana($(this).val()));
		$(this).val(FieldCorrector.correctAsciiSign($(this).val()));
	});
	$("." + this.clnEm).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctEmNumeric($(this).val()));
		$(this).val(FieldCorrector.correctEmAlphabet($(this).val()));
		$(this).val(FieldCorrector.correctEmKana($(this).val()));
		$(this).val(FieldCorrector.correctEmSign($(this).val()));
	});
	$("." + this.clnEmAlphabet).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctEmAlphabet($(this).val()));
	});
	$("." + this.clnEmAlphaNumeric).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctEmNumeric($(this).val()));
		$(this).val(FieldCorrector.correctEmAlphabet($(this).val()));
	});
	$("." + this.clnEmAlphaNumericSign).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctEmNumeric($(this).val()));
		$(this).val(FieldCorrector.correctEmAlphabet($(this).val()));
		$(this).val(FieldCorrector.correctEmSign($(this).val()));
	});
	$("." + this.clnEmKana).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctEmKana($(this).val()));
	});
	$("." + this.clnEmKanaNumericSign).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctEmNumeric($(this).val()));
		$(this).val(FieldCorrector.correctEmKana($(this).val()));
		$(this).val(FieldCorrector.correctEmSign($(this).val()));
	});
	$("." + this.clnEmKanaSign).bind("blur.FieldCorrector change.FieldCorrector", function() {
		$(this).val(FieldCorrector.correctEmKana($(this).val()));
		$(this).val(FieldCorrector.correctEmSign($(this).val()));
	});
	$("." + this.clnZerosupH).bind("blur.FieldCorrector change.FieldCorrector", function() {
		var maxlength = $(this).attr("maxlength");
		if (maxlength) {
			$(this).val(FieldCorrector.correctZerosupH($(this).val(), maxlength));
		}
	});
	$("." + this.clnZerosupF).bind("blur.FieldCorrector change.FieldCorrector", function() {
		var maxlength = $(this).attr("maxlength");
		if (maxlength) {
			$(this).val(FieldCorrector.correctZerosupF($(this).val(), maxlength));
		}
	});

	return true;
};

/**
 * 指定された文字列を全角-半角(数字)補完して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldCorrector.correctAsciiNumeric = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldCorrector.EM2A_MAP_NUMERIC[arg0] ? FieldCorrector.EM2A_MAP_NUMERIC[arg0] : arg0;
		return string;
	});
};

/**
 * 指定された文字列を全角-半角(英字)補完して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldCorrector.correctAsciiAlphabet = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldCorrector.EM2A_MAP_ALPHABET[arg0] ? FieldCorrector.EM2A_MAP_ALPHABET[arg0] : arg0;
		return string;
	});
};

/**
 * 指定された文字列を全角-半角(かな)補完して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldCorrector.correctAsciiKana = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldCorrector.EM2A_MAP_KANA[arg0] ? FieldCorrector.EM2A_MAP_KANA[arg0] : arg0;
		return string;
	});
};

/**
 * 指定された文字列を全角-半角(記号)補完して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldCorrector.correctAsciiSign = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldCorrector.EM2A_MAP_SIGN[arg0] ? FieldCorrector.EM2A_MAP_SIGN[arg0] : arg0;
		return string;
	});
};

/**
 * 指定された文字列を半角-全角(数字)補完して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldCorrector.correctEmNumeric = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldCorrector.A2EM_MAP_NUMERIC[arg0] ? FieldCorrector.A2EM_MAP_NUMERIC[arg0] : arg0;
		return string;
	});
};

/**
 * 指定された文字列を半角-全角(英字)補完して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldCorrector.correctEmAlphabet = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldCorrector.A2EM_MAP_ALPHABET[arg0] ? FieldCorrector.A2EM_MAP_ALPHABET[arg0] : arg0;
		return string;
	});
};

/**
 * 指定された文字列を半角-全角(かな)補完して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldCorrector.correctEmKana = function(value) {
	var result = value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldCorrector.A2EM_MAP_KANA[arg0] ? FieldCorrector.A2EM_MAP_KANA[arg0] : arg0;
		return string;
	});
	for ( var kana in this.SONAT_KANA_EM) {
		result = result.replace(new RegExp(kana, "g"), this.SONAT_KANA_EM[kana]);
	}
	return result;
};

/**
 * 指定された文字列を半角-全角(記号)補完して提供します。<br>
 * @param {String} value 文字列
 * @returns 変換後文字列
 */
FieldCorrector.correctEmSign = function(value) {
	return value.replace(new RegExp(".", "g"), function(arg0, arg1, arg2, arg3) {
		var string = arg0;
		string = FieldCorrector.A2EM_MAP_SIGN[arg0] ? FieldCorrector.A2EM_MAP_SIGN[arg0] : arg0;
		return string;
	});
};

/**
 * 指定された文字列を半角数字前ゼロ補完して提供します。<br>
 * @param {String} value 文字列
 * @param {Number} length 文字長
 * @returns 変換後文字列
 */
FieldCorrector.correctZerosupH = function(value, length) {
	value = this.correctAsciiNumeric(value);
	if (!StringUtil.isNumeric(value)) {
		return value;
	}
	while (value.length < length) {
		value = "0" + value;
	}
	return value;
};

/**
 * 指定された文字列を半角数字後ゼロ補完して提供します。<br>
 * @param {String} value 文字列
 * @param {Number} length 文字長
 * @returns 変換後文字列
 */
FieldCorrector.correctZerosupF = function(value, length) {
	value = this.correctAsciiNumeric(value);
	if (!StringUtil.isNumeric(value)) {
		return value;
	}
	while (value.length < length) {
		value = value + "0";
	}
	return value;
};
