/**
 * ロックレイヤクラス<br>
 * <br>
 * イベントやコンテンツ要素のロック操作インタフェースを提供します。<br>
 *
 * @author Kitagawa<br>
 *
 *<!--
 * 更新日		更新者				更新内容
 * 2012/07/02	Kitagawa			新規作成
 *-->
 */

/*
 * スタイルシートロード
 */
if (!$("link[href='" + ScriptUtil.getScriptPath("LockLayer.js") + "/LockLayer.css" + "']")[0]) {
	var $source = $("<link type=\"text/css\"></script>");
	$source.attr("rel", "stylesheet");
	$source.attr("media", "all");
	$source.attr("href", ScriptUtil.getScriptPath("LockLayer.js") + "/LockLayer.css?_=" + new Date().getTime());
	$("head").append($source);
}

/**
 * コンストラクタ<br>
 * @param {Object} options 動作オプション<br>
 * <ul>
 * <li>target : ロックターゲット要素又はセレクタ(ディフォルト:null(Window))</li>
 * <li>minZIndex : ロックレイヤzIndex最小値(ディフォルト:-1)</li>
 * <li><s>maxZIndex : ロックレイヤzIndex最大値(ディフォルト:2147483646(0x7FFFFFFE))</s></li>
 * <li>maxZIndex : ロックレイヤzIndex最大値(ディフォルト:ターゲット要素のzIndex)</li>
 * <li>showPanel : ロックレイヤパネル表示フラグ(ディフォルト:true)</li>
 * <li>showPanelDelay : ロックレイヤパネル表示ディレイ(ディフォルト:0)</li>
 * <li>showPanelDuration : ロックレイヤパネル表示アニメーション時間(ディフォルト:200)</li>
 * <li>showPanelOpacity : ロックレイヤパネル透明度(ディフォルト:0.15)</li>
 * <li>excludeEvents : ロック除外イベント(ディフォルト:[])</li>
 * </ul>
 */
var LockLayer = function(options) {

	/** ロックターゲット */
	this.target;

	/** ロックレイヤzIndex最小値 */
	this.minZIndex;

	/** ロックレイヤzIndex最大値 */
	this.maxZIndex;

	/** ロックレイヤパネル表示フラグ */
	this.showPanel;

	/** ロックレイヤパネル表示ディレイ */
	this.showPanelDelay;

	/** ロックレイヤパネル表示アニメーション時間 */
	this.showPanelDuration;

	/** ロックレイヤパネル透明度 */
	this.showPanelOpacity;

	/** ロック除外イベント */
	this.excludeEvents;

	/** ロック対象イベント */
	this.lockTargetEvents;

	/** ロックレイヤID */
	this.layerId;

	/** ロックレイヤソース */
	this.layerHtml;

	/** ロックレイヤパネルエレメント */
	this.layerPanel;

	/** ロックフラグ */
	this.locked;

	/**
	 * コンストラクタ処理
	 */
	var defaults = {
		target : null,
		minZIndex : -1,
		//maxZIndex : 2147483646,
		showPanel : true,
		showPanelDelay : 0,
		showPanelDuration : 200,
		showPanelOpacity : 0.15,
		excludeEvents : []
	};
	var mergedOptions = $.extend({}, defaults, options);
	{
		var uid = UniqueID.generate();
		this.target = mergedOptions.target;
		this.minZIndex = mergedOptions.minZIndex;
		//this.maxZIndex = mergedOptions.maxZIndex;
		this.maxZIndex = mergedOptions.maxZIndex ? mergedOptions.maxZIndex : ElementUtil.getZIndex(mergedOptions.target);
		this.showPanel = mergedOptions.showPanel;
		this.showPanelDelay = mergedOptions.showPanelDelay;
		this.showPanelDuration = mergedOptions.showPanelDuration;
		this.showPanelOpacity = mergedOptions.showPanelOpacity;
		this.excludeEvents = mergedOptions.excludeEvents;
		this.lockTargetEvents = [];
		this.layerId = "LockLayer" + uid;
		this.layerHTML = StringUtil.bind("<div id=\"{0}\" class=\"webcore-locklayer-panel\"></div>", [this.layerId]);
		this.layerPanel = null;
		this.locked = false;
		for ( var i = 0; i <= EventUtil.EVENT_NAMES.length - 1; i++) {
			var targetName = EventUtil.EVENT_NAMES[i];
			for ( var j = 0; j <= this.excludeEvents.length - 1; j++) {
				var excludeName = this.excludeEvents[j];
				if (targetName === excludeName) {
					continue;
				}
				this.lockTargetEvents[this.lockTargetEvents.length] = targetName;
			}
		}
	}
};

/**
 * ロックレイヤIDを取得します。<br>
 * @returns {String} ロックレイヤID
 */
LockLayer.prototype.getLayerId = function() {
	return this.layerId;
};

/**
 * 当ロックレイヤがロックしているイベントを削除します。<br>
 * @param selector {Object} selector 対象エレメントセレクタ文字列/jQueryオブジェクト(未指定時は全要素)
 * @returns {Boolean} 正常終了した場合にtrueを返却
 */
LockLayer.prototype.removeLockEvent = function(selector) {
	for ( var i = 0; i <= EventUtil.EVENT_NAMES.length - 1; i++) {
		var targetName = EventUtil.EVENT_NAMES[i];
		var targetEvent = targetName + ".LockLayer" + "-" + this.getLayerId();
		$(selector).unbind(targetEvent);
	}
	return true;
};

/**
 * クラスに指定されているレイヤ情報でロックを実施します。<br>
 * @returns {Boolean} 正常終了した場合にtrueを返却
 */
LockLayer.prototype.lock = function() {
	var instance = this;

	/*
	 * ロック状態の場合は処理スキップ
	 */
	if (this.locked) {
		return false;
	}

	/*
	 * 処理対象ターゲット
	 */
	var $target = this.target ? $(this.target) : $(window);

	/*
	 * ロックレイヤパネル表示
	 */
	if (this.showPanel) {
		// イベント識別子生成
		var eventSuffix = "LockLayer-" + this.uid;

		// レイヤパネル生成
		this.layerPanel = $(this.layerHTML);

		// レイヤパネルスタイル設定
		this.layerPanel.css({
			opacity : 0,
			zIndex : this.maxZIndex + 1
		});

		// レイヤパネルプログレスアイコンクラス設定
		var targetSize = $target.width() > $target.height() ? $target.height() : $target.width();
		var iconClass = "webcore-locklayer-progress-48";
		if (targetSize < 48) {
			iconClass = "webcore-locklayer-progress-24";
		}
		if (targetSize < 24) {
			iconClass = "webcore-locklayer-progress-16";
		}
		if (targetSize < 16) {
			iconClass = "";
		}
		if (!StringUtil.isEmpty(iconClass)) {
			this.layerPanel.addClass(iconClass);
		}

		// レイヤパネル配置
		this.layerPanel.appendTo($("body"));
		this.layerPanel.show();

		// レイヤパネルフィックス
		LayoutUtil.fix(this.layerPanel, $target);
		$(window).bind("resize." + eventSuffix, function() {
			LayoutUtil.fix(instance.layerPanel, $target);
			return true;
		});
		$(window).bind("scroll." + eventSuffix, function() {
			LayoutUtil.fix(instance.layerPanel, $target);
			return true;
		});

		// レイヤパネル表示
		setTimeout(function() {
			instance.layerPanel.animate({
				opacity : instance.showPanelOpacity
			}, instance.showPanelDuration, function() {
			});
		}, this.showPanelDelay);
	}

	/*
	 * イベントロック処理
	 */
	var $elements = $target[0] === $(window)[0] || $target[0] === $(document)[0] ? $("*") : $target;
	$elements.each(function() {
		for ( var i = 0; i <= instance.lockTargetEvents.length - 1; i++) {
			var targetName = instance.lockTargetEvents[i];
			var targetEvent = targetName + "." + "LockLayer-" + instance.getLayerId();
			EventUtil.bindAtFirst($(this), targetEvent, function(e) {
				var targetZindex = ElementUtil.getZIndex($(this));

				// 対象階層範囲の場合にイベントキャンセル
				if (targetZindex >= instance.minZIndex && targetZindex <= instance.maxZIndex) {
					return EventUtil.cancelEvent(e);
				}

				return true;
			});
		}
		return true;
	});

	/*
	 * ロックフラグ設定
	 */
	this.locked = true;

	return true;
};

/**
 * ロックされたレイヤを解除します。<br>
 * @returns {Boolean} 正常終了した場合にtrueを返却
 */
LockLayer.prototype.unlock = function() {
	var instance = this;

	/*
	 * 非ロック状態の場合は処理スキップ
	 */
	if (!this.locked) {
		return false;
	}

	/*
	 * 処理対象ターゲット
	 */
	var $target = this.target ? $(this.target) : $(window);

	/*
	 * イベントロック解除処理
	 */
	var $elements = $target[0] === $(window)[0] || $target[0] === $(document)[0] ? $("*") : $target;
	$elements.each(function() {
		for ( var i = 0; i <= instance.lockTargetEvents.length - 1; i++) {
			var targetName = instance.lockTargetEvents[i];
			var targetEvent = targetName + "." + "LockLayer-" + instance.getLayerId();
			$(this).unbind(targetEvent);
		}
		return true;
	});

	/*
	 * ロックレイヤパネル削除
	 */
	if (this.showPanel) {
		var eventSuffix = "LockLayer-" + this.uid;
		setTimeout(function() {
			instance.layerPanel.animate({
				opacity : 0
			}, instance.showPanelDuration, function() {
				$(window).unbind("resize." + eventSuffix);
				$(window).unbind("scroll." + eventSuffix);
				instance.layerPanel.show();
				instance.layerPanel.remove();

				/*
				 * ロックフラグ解除
				 */
				instance.locked = false;
			});
		}, 0);
	}

	return true;
};
